/******************************************************************************

ASI Linux Hardware Programming Interface (HPI) Wrapper
This module contains the main HPI entry point HPI_Message which uses ioctl
to pass the message to the driver.

Copyright (C) 1997-2017 AudioScience, Inc. All rights reserved.

This software is provided 'as-is', without any express or implied warranty.
In no event will AudioScience Inc. be held liable for any damages arising
from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
   claim that you wrote the original software. If you use this software
   in a product, an acknowledgment in the product documentation would be
   appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.
3. This copyright notice and list of conditions may not be altered or removed
   from any source distribution.

AudioScience, Inc. <support@audioscience.com>

( This license is GPL compatible see http://www.gnu.org/licenses/license-list.html#GPLCompatibleLicenses )

******************************************************************************/

#define SOURCEFILE_NAME "hpiwrap.c"

#include <stdbool.h>		/* for close */
#include <unistd.h>		/* for close */
#include <fcntl.h>		/* for open */
#include <limits.h>		/* for PATH_MAX */
#include <sys/ioctl.h>		/* for ioctl */
#include <stdio.h>

#include "hpi_internal.h"
#include "hpi_version.h"
#include "hpidebug.h"
#include "hpimsginit.h"

#ifdef HPI_BUILD_HPIDUMP
#include "hpidump.c"
#else
static void hpi_dump_open(void) {};
static void hpi_dump_close(void) {};
static void hpi_dump_message(struct hpi_message *phm, struct hpi_response *phr) {};
static void hpi_dump_response(struct hpi_message *phm, struct hpi_response *phr) {};
#endif

/** List of device file names to probe for HPI ioctl
Covers both normal HPI device, and ALSA hwdep devices.
Note that each ALSA HPI hwdep device is equivalent to the standard HPI device,
and gives access to all HPI cards, not just the ALSA card it is associated with.
*/
static char *devnames[] = {
	NULL, /* Will be replaced by LIBHPI_DEV env pointer */
	"/dev/asihpi", /* Normal HPI device */
	"/dev/snd/hwC0D0", /* ALSA driver hwdep, may support HPI ioctl */
	"/dev/snd/hwC1D0",
	"/dev/snd/hwC2D0",
	"/dev/snd/hwC3D0",
	"/dev/snd/hwC4D0",
};

#define HPI_DEV_FMT "/dev/asihpi%d"
#define ALSA_DEV_FMT "/dev/snd/hwC%dD0"


static int adapter_count = 0;
static struct {
	int fd;
	uint32_t version;
	uint16_t type;
} hpi_adapters[HPI_MAX_ADAPTERS];
static int dev_index_to_hpi_index[HPI_MAX_ADAPTERS];

#if defined HPI_BUILD_MULTIINTERFACE
	#if defined HPI_BUILD_SSX2
		#include "hpissx2.h"
		#define HPI_MESSAGE_LOCAL_ADAPTERS(m,r) HPI_MESSAGE_LOWER_LAYER(m,r,NULL)
	#else
		void HPI_MessageIoctl(HPI_MESSAGE *phm, HPI_RESPONSE *phr);
		#define HPI_MESSAGE_LOCAL_ADAPTERS(m,r) HPI_MessageIoctl(m,r)
	#endif
#else
	#define HPI_MESSAGE_LOCAL_ADAPTERS(m,r) HPI_Message(m,r)
#endif

static void reset_globals(void)
{
	int i;

	memset(hpi_adapters, 0, sizeof(hpi_adapters));
	for (i = 0 ; i < ARRAY_SIZE(hpi_adapters); i++) {
		hpi_adapters[i].fd = -1;
		dev_index_to_hpi_index[i] = -1;
	}
	adapter_count = 0;
}

static int drv_hpi_ioctl(int fd, struct hpi_message *phm, struct hpi_response *phr)
{
	int status;
	struct hpi_ioctl_linux hpi_ioctl_data;

	hpi_ioctl_data.phm = phm;
	hpi_ioctl_data.phr = phr;

	hpi_dump_message(phm, phr);
	status = ioctl(fd, HPI_IOCTL_LINUX, (unsigned long)&hpi_ioctl_data);
	if (status < 0) {
		if (hpiDebugLevel) {
			perror("HPI_Message error");
		}
		HPI_DEBUG_MESSAGE(ERROR, phm);
		phr->wError = HPI_ERROR_PROCESSING_MESSAGE;
	} else {
		HPI_DEBUG_MESSAGE(DEBUG, phm);
	}
	hpi_dump_response(phm, phr);
	HPI_DEBUG_RESPONSE(phr);
	return status;
}

static int open_dev(const char *fmt, int dev_index)
{
	HPI_MESSAGE hm;
	HPI_RESPONSE hr;
	char devname[PATH_MAX];
	uint32_t version;
	int fd, adapter_index;

	snprintf(devname, sizeof(devname), fmt, dev_index);
	fd = open(devname, O_RDWR);
	if (fd < 0) { // file couldn't be opened
		goto skip_index;
	}

	HPI_InitMessageResponse(&hm, &hr, HPI_OBJ_SUBSYSTEM, HPI_SUBSYS_GET_VERSION);
	drv_hpi_ioctl(fd, &hm, &hr);
	version = hr.u.s.dwData;
	if (hr.wError) {
		HPI_DEBUG_LOG1(ERROR, "Error: HPI_SUBSYS_GET_VERSION returned %d\n", hr.wError);
		goto skip_index;
	}

	{
		const uint8_t drv_major_version = (hr.u.s.dwData >> 16) & 0xFF;
		const uint8_t lib_major_version = ((HPI_VER) >> 16) & 0xFF;
		if (drv_major_version != lib_major_version) {
			HPI_DEBUG_LOG2(ERROR, "Error: incompatible library (0x%05X) and driver (0x%05X) versions\n",
				HPI_VER, hr.u.s.dwData);
			goto skip_index;
		}
	}

	HPI_InitMessageResponse(&hm, &hr, HPI_OBJ_SUBSYSTEM, HPI_SUBSYS_GET_ADAPTER);
	hm.wObjIndex = 0;
	drv_hpi_ioctl(fd, &hm, &hr);
	if (hr.wError) {
		HPI_DEBUG_LOG1(ERROR, "Error: HPI_SUBSYS_GET_ADAPTER returned %d\n", hr.wError);
		goto skip_index;
	}
	adapter_index = hr.u.s.wAdapterIndex;

	if (hpi_adapters[adapter_index].fd >= 0) {
		/* adapter index already taken! */
		HPI_DEBUG_LOG3(INFO, "Device %s has HPI index %d but that index is already taken by ASI%04hX\n",
				devname, adapter_index, hpi_adapters[adapter_index].type);
		goto skip_index;
	}

	dev_index_to_hpi_index[adapter_count] = adapter_index;
	hpi_adapters[adapter_index].version = version;
	hpi_adapters[adapter_index].fd = fd;
	hpi_adapters[adapter_index].type = hr.u.s.wAdapterType;
	adapter_count++;

	HPI_DEBUG_LOG3(INFO,
		"lib version 0x%05X opened driver version 0x%05X at %s\n",
		HPI_VER, hpi_adapters[adapter_index].version, devname);

	/* success, try next index */
	return 0;

skip_index:
	/* failed, try next index */
	if (fd >= 0) {
		close(fd);
	}
	return -1;
}

#if defined HPI_BUILD_MULTIINTERFACE
uint16_t HPI_DriverOpenIoctl(void)
#else
uint16_t HPI_DriverOpen(void)
#endif
{
	int i;
	char *env;

	if (adapter_count) {
		if (hpiDebugLevel) {
			HPI_DEBUG_LOG0(ERROR, "Driver already open\n");
		}
		return 1;
	}

	reset_globals();

	env = getenv("LIBHPI_DEBUG_LEVEL");
	if ((env) && (*env >= '0') && (*env <= '9'))
		hpiDebugLevel = *env - '0';

	devnames[0] = getenv("LIBHPI_DEV");

	/* Probe the list of device files */
	for (i = 0 ; i < ARRAY_SIZE(hpi_adapters); i++) {
		open_dev(HPI_DEV_FMT, i);
		open_dev(ALSA_DEV_FMT, i);
	}

	if (!adapter_count) {
		if (hpiDebugLevel) {
			perror("HPI_DriverOpen error");
		}
		return 0;
	}

	env = getenv("LIBHPI_DEBUG_DUMP");
	if (env)
		hpi_dump_open();

	return 1;  // Success! Note that this is not an error code
}

#if defined HPI_BUILD_MULTIINTERFACE
void HPI_MessageIoctl(
#else
void HPI_Message(
#endif
		 struct hpi_message *phm,
		 struct hpi_response *phr)
{
	int adapter_index = phm->wAdapterIndex;
	int fd;

	/* auto-open driver if adapter_count == 0 */
	if (!adapter_count) {
		if (!HPI_DriverOpen()) {
			phr->wError = HPI_ERROR_PROCESSING_MESSAGE;
			return;
		}
	}

	/* HPI functions that do not specify an adapter index have to be processed separately */
	if (phm->wObject == HPI_OBJ_SUBSYSTEM) {
		if (phm->wAdapterIndex != HPI_ADAPTER_INDEX_INVALID) {
			HPI_DEBUG_LOG2(WARNING, "Suspicious adapter index %d in subsys message 0x%x.\n",
				phm->wAdapterIndex, phm->wFunction);
		}

		if (phm->wFunction == HPI_SUBSYS_GET_VERSION) {
			HPI_InitResponse(phr, HPI_OBJ_SUBSYSTEM, HPI_SUBSYS_GET_VERSION, 0);
			phr->u.s.dwVersion = HPI_VER >> 8;	// return major.minor
			phr->u.s.dwData = HPI_VER;	// return major.minor.release
			phr->wError = 0;
			return;
		}

		if (phm->wFunction == HPI_SUBSYS_GET_NUM_ADAPTERS) {
			HPI_InitResponse(phr, HPI_OBJ_SUBSYSTEM, HPI_SUBSYS_GET_NUM_ADAPTERS, 0);
			phr->u.s.wNumAdapters = adapter_count;
			phr->wError = 0;
			return;
		}

		if (phm->wFunction == HPI_SUBSYS_GET_ADAPTER) {
			HPI_InitResponse(phr, HPI_OBJ_SUBSYSTEM, HPI_SUBSYS_GET_ADAPTER, 0);
			if (phm->wObjIndex >= ARRAY_SIZE(hpi_adapters)) {
				phr->wError = HPI_ERROR_INVALID_OBJ_INDEX;
				return;
			}

			adapter_index = dev_index_to_hpi_index[phm->wObjIndex];
			if (adapter_index < 0 || hpi_adapters[adapter_index].fd < 0) {
				phr->wError = HPI_ERROR_INVALID_OBJ_INDEX;
				return;
			}

			phr->u.s.wAdapterIndex = adapter_index;
			phr->u.s.wAdapterType = hpi_adapters[adapter_index].type;
			phr->wError = 0;
			return;
		}

		/* All unrecognized HPI_OBJ_SUBSYSTEM messages are unimplemented */
		phr->wError = HPI_ERROR_UNIMPLEMENTED;
		return;
	}

	if (phm->wObject == HPI_OBJ_ADAPTER) {
		/* Setting SSX2 on/off will return an error as if SSX2 were unsupported.
		 * This makes the relevant UI section disappear in ASIControl
		 */
		if (phm->wFunction == HPI_ADAPTER_GET_PROPERTY &&
			phm->wObjIndex == HPI_ADAPTER_PROPERTY_SSX2_SETTING) {
			phr->wError = HPI_ERROR_INVALID_CONTROL_ATTRIBUTE;
			return;
		}
		/* Unrecognized HPI_OBJ_ADAPTER messages are passed down to the hardware */
	}

	if (adapter_index >= ARRAY_SIZE(hpi_adapters)) {
		HPI_InitResponse(phr, phm->wObject, phm->wFunction, 0);
		phr->wError = HPI_ERROR_BAD_ADAPTER_NUMBER;
		return;
	}

	fd = hpi_adapters[adapter_index].fd;
	if (fd < 0) {
		phr->wError = HPI_ERROR_BAD_ADAPTER_NUMBER;
		return;
	}

	/* Let the rest of HPI messages through */
	drv_hpi_ioctl(fd, phm, phr);
}

#if defined HPI_BUILD_MULTIINTERFACE
void HPI_DriverCloseIoctl(void)
#else
void HPI_DriverClose(void)
#endif
{
	int i;

	for (i = 0 ; i < ARRAY_SIZE(hpi_adapters); i++) {
		int status;
		int fd = hpi_adapters[i].fd;
		if (fd < 0) {
			continue;
		}
		status = close(fd);
		if (status < 0 && hpiDebugLevel) {
			perror("HPI_DriverClose error");
		}
	}
	reset_globals();
	hpi_dump_close();
}
