/***********************************************************************

    AudioScience HPI driver
    Functions for reading DSP code using hotplug firmware loader

    Copyright (C) 1997-2017  AudioScience Inc. <support@audioscience.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of version 2 of the GNU General Public License as
    published by the Free Software Foundation;

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

***********************************************************************/
#define SOURCEFILE_NAME "hpidspcd.c"
#include "hpidspcd.h"
#include "hpidebug.h"
#include "hpi_version.h"

struct dsp_code_private {
	/**  Firmware descriptor */
	const struct firmware *firmware;
	struct pci_dev *dev;
};

/*-------------------------------------------------------------------*/
short HpiDspCode_Open(uint32_t adapter,
		      struct hpi_os_adapter *os_data,
		      struct dsp_code *dsp_code)
{
	const struct firmware *firmware;
	struct pci_dev *dev = os_data->pci_dev;
	struct code_header header;
	char fw_name[30];
	short err_ret = HPI_ERROR_DSP_FILE_NOT_FOUND;
	int err;

	snprintf(fw_name, sizeof(fw_name), "asihpi/%d.%02d/dsp%04x.bin",
	         HPI_VER_MAJOR(HPI_VER), HPI_VER_MINOR(HPI_VER), adapter);

	err = request_firmware(&firmware, fw_name, &dev->dev);

	if (err || !firmware) {
		dev_err(&dev->dev, "%d, request_firmware failed for %s\n", err, fw_name);
		goto error1;
	}
	if (firmware->size < sizeof(header)) {
		dev_err(&dev->dev, "Header size too small %s\n", fw_name);
		goto error2;
	}
	memcpy(&header, firmware->data, sizeof(header));

	if ((header.type != 0x45444F43) || /* "CODE" */
	    (header.adapter != adapter) ||
	    (header.size != firmware->size)) {
		dev_err(&dev->dev,
		    "Invalid firmware header size %d != file %zd\n", header.size, firmware->size);
		goto error2;
	}

	if (HPI_VER_MAJOR(header.version) != HPI_VER_MAJOR(HPI_VER)) {
		/* Major version change probably means Host-DSP protocol change */
		dev_err(&dev->dev, "Incompatible firmware version DSP image %X != Driver %X\n",
		    header.version, HPI_VER);
		goto error2;
	}

	if (header.version != HPI_VER) {
		dev_warn(&dev->dev,
			"Firmware version mismatch: DSP image %X != Driver %X\n",
		    header.version, HPI_VER);
	}

	HPI_DEBUG_LOG(DEBUG, "Dsp code %s opened\n", fw_name);
	dsp_code->pvt = kmalloc(sizeof(*dsp_code->pvt), GFP_KERNEL);
	if (!dsp_code->pvt) {
		err_ret = HPI_ERROR_MEMORY_ALLOC;
		goto error2;
	}

	dsp_code->pvt->dev = dev;
	dsp_code->pvt->firmware = firmware;
	dsp_code->header = header;
	dsp_code->block_length = header.size / sizeof(uint32_t);
	dsp_code->word_count = sizeof(header) / sizeof(uint32_t);
	return 0;

error2:
	release_firmware(firmware);
error1:
	dsp_code->block_length = 0;
	return err_ret;
}

/*-------------------------------------------------------------------*/
void HpiDspCode_Close(struct dsp_code *dsp_code)
{
	HPI_DEBUG_LOG(DEBUG, "Dsp code closed\n");
	release_firmware(dsp_code->pvt->firmware);
	kfree(dsp_code->pvt);
}

/*-------------------------------------------------------------------*/
void HpiDspCode_Rewind(struct dsp_code *dsp_code)
{
	/* Go back to start of  data, after header */
	dsp_code->word_count = sizeof(struct code_header) / sizeof(uint32_t);
}

/*-------------------------------------------------------------------*/
short HpiDspCode_ReadWord(struct dsp_code *dsp_code, uint32_t *pdwWord)
{
	if (dsp_code->word_count + 1 > dsp_code->block_length)
		return HPI_ERROR_DSP_FILE_FORMAT;

	*pdwWord =
	    ((uint32_t *)(dsp_code->pvt->firmware->data))[dsp_code->word_count];
	dsp_code->word_count++;
	return 0;
}

/*-------------------------------------------------------------------*/
short HpiDspCode_ReadBlock(size_t nWordsRequested,
			   struct dsp_code *dsp_code, uint32_t **ppdwBlock)
{
	if (dsp_code->word_count + nWordsRequested >
	    dsp_code->block_length)
		return HPI_ERROR_DSP_FILE_FORMAT;

	*ppdwBlock =
	    ((uint32_t *)(dsp_code->pvt->firmware->data)) + dsp_code->word_count;
	dsp_code->word_count += nWordsRequested;
	return 0;
}
