/**
 Common functions used by hpixxxx.c modules

Copyright (C) 1997-2017 AudioScience, Inc. All rights reserved.

This software is provided 'as-is', without any express or implied warranty.
In no event will AudioScience Inc. be held liable for any damages arising
from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
   claim that you wrote the original software. If you use this software
   in a product, an acknowledgment in the product documentation would be
   appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.
3. This copyright notice and list of conditions may not be altered or removed
   from any source distribution.

AudioScience, Inc. <support@audioscience.com>

( This license is GPL compatible see http://www.gnu.org/licenses/license-list.html#GPLCompatibleLicenses )

*/

#ifdef __cplusplus
extern "C" {
#endif

struct hpi_adapter_obj;

typedef int adapter_initialize_func(struct hpi_adapter_obj *);
typedef void adapter_shutdown_func(struct hpi_adapter_obj *);

/* a function that takes an adapter obj and returns an int */
typedef int adapter_interrupt_func(struct hpi_adapter_obj *pao, uint32_t message);

typedef void adapter_transaction_func(
	struct hpi_adapter_obj *adapter,
	struct hpi_message *request,
	struct hpi_response *response
);

/*////////////////////////////////////////////////////////////////////////// */
/* declarations for individual HPI entry points */
extern adapter_initialize_func HPI6700_adapter_init;
extern adapter_initialize_func HPI6205_adapter_init;
extern adapter_initialize_func HPI6000_adapter_init;
extern adapter_initialize_func HPI1100_adapter_init;

#define HPI_IRQ_NONE		(0)
#define HPI_IRQ_MESSAGE		(1)
#define HPI_IRQ_MIXER		(2)

struct hpi_open_state {
	void *hOwner;
	uint16_t error;
	uint16_t is_open;
};

struct hpi_endpoint {
	const char *name;
	adapter_initialize_func *adapter_init;
	adapter_shutdown_func* shutdown;
	adapter_interrupt_func *irq_query_and_clear;
	adapter_transaction_func *transact;
};

struct hpi_adapter_obj {
	struct hpi_os_adapter os;
	struct hpi_endpoint ep;

	uint16_t type;	// 0x6644 == ASI6644 etc
	uint16_t index;
	uint8_t instreams;
	uint8_t outstreams;

	uint8_t wDspCrashed;
	uint8_t wHasControlCache;

	// use these to keep track of opens from user mode apps/DLLs
	struct hpi_open_state outstream_user_open[HPI_MAX_STREAMS];
	struct hpi_open_state instream_user_open[HPI_MAX_STREAMS];

#ifdef HPI_LOCKING
	HpiOs_SpinLock dspLock;
#endif

	void *priv;  // backend-specific data

	struct hpi_hostbuffer_status *aInStreamHostBufferStatus;
	struct hpi_hostbuffer_status *aOutStreamHostBufferStatus;
};
#ifndef HPI_BUILD_REASSIGN_DUPLICATE_ADAPTER_IDX
struct hpi_duplicate_index_adapter_obj {
	uint16_t type;	// 0x6644 == ASI6644 etc
	uint16_t index;
};
#endif

struct hpi_control_cache {
	/** indicates whether the structures are initialized */
	uint16_t dwInit;
	uint16_t adap_idx;
	uint32_t dwControlCount;
	uint32_t dwCacheSizeInBytes;
	/** pointer to allocated memory of lookup pointers. */
	struct hpi_control_cache_info **pInfo;
	/** pointer to DSP's control cache. */
	uint8_t *pCache;
};

struct hpi_adapter_obj *HpiGetAdapter(uint16_t wAdapterIndex);

hpi_err_t HpiAddAdapter(struct hpi_adapter_obj *pao);

void HpiRemoveAdapter(struct hpi_adapter_obj *pao);

short HpiFindControlCache(
	struct hpi_control_cache *pCache,
	int ControlIndex,
	struct hpi_control_cache_single **pC
);

short HpiCheckControlCache(
	struct hpi_control_cache *pC,
	struct hpi_message *phm,
	struct hpi_response *phr
);

short HpiCheckControlCache_Single(
	struct hpi_control_cache_single *pC,
	struct hpi_message *phm,
	struct hpi_response *phr
);

struct hpi_control_cache * HpiAllocControlCache(
	const uint32_t dwNumberOfControls,
	const uint32_t dwSizeInBytes,
	uint8_t *pDSPControlBuffer
);

void HpiFreeControlCache(struct hpi_control_cache *pCache);

void HpiCmn_ControlCache_SyncToMsg(
	struct hpi_control_cache *pC,
	struct hpi_message *phm,
	struct hpi_response *phr
);

void HpiCmn_ControlCache_SyncToMsg_Single(
	struct hpi_control_cache_single *pC,
	struct hpi_message *phm,
	struct hpi_response *phr
);

hpi_err_t HpiValidateResponse(
	struct hpi_message *phm,
	struct hpi_response *phr
);

void HPI_COMMON(
	struct hpi_message *phm,
	struct hpi_response *phr
);

#ifdef __cplusplus
}
#endif
