/*****************************************************************************/
/** \file hpi.h

 AudioScience Hardware Programming Interface (HPI)
 public API definition.

 The HPI is a low-level hardware abstraction layer to all
 AudioScience digital audio adapters

Copyright (C) 1997-2017 AudioScience, Inc. All rights reserved.

This software is provided 'as-is', without any express or implied warranty.
In no event will AudioScience Inc. be held liable for any damages arising
from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
   claim that you wrote the original software. If you use this software
   in a product, an acknowledgment in the product documentation would be
   appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.
3. This copyright notice and list of conditions may not be altered or removed
   from any source distribution.

AudioScience, Inc. <support@audioscience.com>

( This license is GPL compatible see http://www.gnu.org/licenses/license-list.html#GPLCompatibleLicenses )

*/
#ifdef HPI_OS_DELETE
/*
 You must define one operating system that the HPI is to be compiled under
 HPI_OS_WIN32_USER   32bit Windows
 HPI_OS_DSP_C6000    DSP TI C6000  (automatically set)
 HPI_OS_WDM          Windows WDM kernel driver
 HPI_OS_LINUX        Linux userspace
 HPI_OS_LINUX_KERNEL Linux kernel (automatically set)
 HPI_OS_OSX          OSX user/kernel

******************************************************************************/
#endif

#ifndef _HPI_H_
#define _HPI_H_

#ifndef HPI_OS_LINUX_KERNEL
#ifdef _DOX_ONLY_
/*****************************************************************************/
/** \defgroup compile_time_defines HPI compile time defines

This section descibes the usage of HPI defines to set the target compile
environment.
The definitions are used in the build environment to control how HPI is built.
They are NOT predefined in hpi.h!
\{
*/
/** \defgroup os_type_defines HPI_OS_XXX build environment defines

A developer wishing to create an application that runs on any 32-bit Windows
environment would compile code with the following two defines set:
HPI_OS_WIN32_USER and HPIDLL_IMPORTS. They would link against
asihpi32.lib so as to make calles to asihpi32.dll.

\{
*/
/** Define when creating a 32 bit or 64 bit Windows user application. */
#define HPI_OS_WIN32_USER
/** Define when creating a Linux application.*/
#define HPI_OS_LINUX
/**\}*/
/** Define this when building a 64 bit application.
When not defined a 32 bit environment will be assumed.
*/
#define HPI_BUILD_64BIT
/** Define this to enable deprecated functions and defines.
*/
#define HPI_BUILD_INCLUDE_DEPRECATED
/** Define this to remove legacy type warnings. Applications that do not
use C99 types will not generate compiler warnings.
*/
#define HPI_BUILD_DISABLE_DEPRECATED_TYPE_WARNINGS
/** \defgroup hpi_dll_defines HPI DLL function attributes
DLL environment defines.
\{
*/
/** Define when building an application that links to ASIHPI32.LIB
   (imports HPI functions from ASIHPI32.DLL). */
#define HPIDLL_IMPORTS
/** Define when creating an application that uses the HPI DLL. */
#define HPIDLL_EXPORTS
/** Define when building an application that compiles in
    HPIFUNC.C and does not use ASIHPI32.DLL */
#define HPIDLL_STATIC
/**\}*/

/**\}*/
#undef HPI_OS_WIN32_USER
#undef HPI_OS_LINUX
#undef HPIDLL_EXPORTS
#undef HPIDLL_STATIC
#undef HPI_BUILD_INCLUDE_DEPRECATED
#undef HPI_BUILD_DISABLE_DEPRECATED_TYPE_WARNINGS
#undef HPI_BUILD_64BIT
/*****************************************************************************/
#endif				/* #ifdef _DOX_ONLY_ */
#endif

#ifdef __cplusplus
/* *INDENT-OFF* */
extern "C" {
/* *INDENT-ON* */
#endif

#ifndef HPI_OS_LINUX_KERNEL
/* detect OS based on compiler defines */
#if __linux__ && defined(__KERNEL__)
#define HPI_OS_LINUX_KERNEL
#elif defined(_TMS320C6X) && ! defined HPI_OS_DSP_C6000
#define HPI_OS_DSP_C6000
#endif
#endif

#if defined(HPI_OS_LINUX_KERNEL)
#include <linux/types.h>
#define HPI_BUILD_KERNEL_MODE

#elif defined(HPI_OS_DSP_C6000)
#include <stdlib.h>
#include <string.h>
#define DISABLE_PRAGMA_PACK1
#if !defined HPI_BUILD_USE_CSLR && !defined _STDINT_H_
#include <csl_stdinc.h>
typedef Uint8  uint8_t;
typedef Int8  int8_t;
typedef Uint16 uint16_t;
typedef Uint32 uint32_t;
typedef Int32 int32_t;
typedef struct { uint8_t d[8]; } uint64_t;
typedef struct { uint8_t d[8]; } int64_t;
#else
#include <stdint.h>
#endif

#elif defined(HPI_OS_WIN32_USER)
#include <stdlib.h>
#include <stdio.h>
#if defined(HPIDLL_STATIC)
#elif HPIDLL_EXPORTS
#define HPIDECL dllexport
#elif defined(HPIDLL_IMPORTS)
#define HPIDECL dllimport
#else
#error Need to define either HPIDLL_EXPORTS, HPIDLL_IMPORTS or HPIDLL_STATIC.
// Conditions for using each of the above defines is as follows.
// HPIDLL_EXPORTS - for building asihpi32.dll (exports HPI functions).
// HPIDLL_IMPORTS - for building a WIN32 application that links to asihpi32.lib (imports HPI functions).
// HPIDLL_STATIC  - for building a WIN32 application that compiles in hpifunc.c and does not use asihpi32.dll.
#endif
#ifdef HPIDECL
/* define HPIDECL as either dllexport or dllimport */
#define HPI_API(type) __declspec(HPIDECL) type __stdcall
#endif
#ifdef _CRT_DEPRECATE_TEXT
#ifndef HPI_BUILD_DISABLE_DEPRECATED_TYPE_WARNINGS
#define HPI_DEPRECATED _CRT_DEPRECATE_TEXT("This function may be removed in future")
#endif
#endif
typedef unsigned char  uint8_t;
typedef unsigned short uint16_t;
typedef unsigned int   uint32_t;
typedef int            int32_t;
typedef unsigned long long uint64_t;
typedef long long      int64_t;

#elif defined(HPI_OS_WDM)
#include <stdlib.h>
typedef unsigned char		uint8_t;
typedef unsigned short		uint16_t;
typedef unsigned int		uint32_t;
typedef unsigned long long	uint64_t;
typedef long long		int64_t;
#define HPI_BUILD_KERNEL_MODE
#elif defined(HPI_OS_OSX)
#include <sys/types.h>
#include <stdint.h>
#else
#include <stdlib.h>
#include <stdint.h>
#endif

#ifndef HPI_BUILD_SANITISE
#if __GNUC__ >= 3
#ifndef HPI_DEPRECATED
#define HPI_DEPRECATED __attribute__((deprecated))
#endif
#endif

#if    __GNUC__ > 3 || (__GNUC__ == 3 && __GNUC_MINOR__ >= 4)
#define HPI_API(type)  __attribute__((__warn_unused_result__)) type
#define HPI_API_VOID(type) type
#endif

#ifndef HPI_API
#define HPI_API(a) a
#endif
#ifndef HPI_API_VOID
#define HPI_API_VOID(a) HPI_API(a)
#endif

#ifndef HPI_DEPRECATED
#define HPI_DEPRECATED
#endif

#if 1
#ifdef HPI_DEBUG
#error Use HPI_BUILD_DEBUG
#endif

#ifdef HPI_KERNEL_MODE
#error Use HPI_BUILD_KERNEL_MODE
#endif

#endif

#ifndef HPI_BUILD_EXCLUDE_ALL_DEPRECATED
HPI_DEPRECATED typedef uint8_t  HW8;
HPI_DEPRECATED typedef uint16_t HW16;
HPI_DEPRECATED typedef uint32_t HW32;
#endif

#endif

/******************************************************************************/
/********       HPI API DEFINITIONS                                       *****/
/******************************************************************************/

/*******************************************/
/**  Audio format types
\ingroup stream
*/
enum HPI_FORMATS {
/** Used internally on adapter. */
	HPI_FORMAT_MIXER_NATIVE = 0,
/** 8-bit unsigned PCM. Windows equivalent is WAVE_FORMAT_PCM. */
	HPI_FORMAT_PCM8_UNSIGNED = 1,
/** 16-bit signed PCM. Windows equivalent is WAVE_FORMAT_PCM. */
	HPI_FORMAT_PCM16_SIGNED = 2,
/** MPEG-1 Layer-1. */
	HPI_FORMAT_MPEG_L1 = 3,
/** MPEG-1 Layer-2.

Windows equivalent is WAVE_FORMAT_MPEG.

The following table shows what combinations of mode and bitrate are possible:

<table border=1 cellspacing=0 cellpadding=5>
<tr>
<td><p><b>Bitrate (kbs)</b></p>
<td><p><b>Mono</b></p>
<td><p><b>Stereo,<br>Joint Stereo or<br>Dual Channel</b></p>

<tr><td>32<td>X<td>_
<tr><td>40<td>_<td>_
<tr><td>48<td>X<td>_
<tr><td>56<td>X<td>_
<tr><td>64<td>X<td>X
<tr><td>80<td>X<td>_
<tr><td>96<td>X<td>X
<tr><td>112<td>X<td>X
<tr><td>128<td>X<td>X
<tr><td>160<td>X<td>X
<tr><td>192<td>X<td>X
<tr><td>224<td>_<td>X
<tr><td>256<td>-<td>X
<tr><td>320<td>-<td>X
<tr><td>384<td>_<td>X
</table>
*/
	HPI_FORMAT_MPEG_L2 = 4,
/** MPEG-1 Layer-3.
Windows equivalent is WAVE_FORMAT_MPEG.

The following table shows what combinations of mode and bitrate are possible:

<table border=1 cellspacing=0 cellpadding=5>
<tr>
<td><p><b>Bitrate (kbs)</b></p>
<td><p><b>Mono<br>Stereo @ 8,<br>11.025 and<br>12kHz*</b></p>
<td><p><b>Mono<br>Stereo @ 16,<br>22.050 and<br>24kHz*</b></p>
<td><p><b>Mono<br>Stereo @ 32,<br>44.1 and<br>48kHz</b></p>

<tr><td>16<td>X<td>X<td>_
<tr><td>24<td>X<td>X<td>_
<tr><td>32<td>X<td>X<td>X
<tr><td>40<td>X<td>X<td>X
<tr><td>48<td>X<td>X<td>X
<tr><td>56<td>X<td>X<td>X
<tr><td>64<td>X<td>X<td>X
<tr><td>80<td>_<td>X<td>X
<tr><td>96<td>_<td>X<td>X
<tr><td>112<td>_<td>X<td>X
<tr><td>128<td>_<td>X<td>X
<tr><td>144<td>_<td>X<td>_
<tr><td>160<td>_<td>X<td>X
<tr><td>192<td>_<td>_<td>X
<tr><td>224<td>_<td>_<td>X
<tr><td>256<td>-<td>_<td>X
<tr><td>320<td>-<td>_<td>X
</table>
\b * Available on the ASI6000 series only
*/
	HPI_FORMAT_MPEG_L3 =		5,
/** Dolby AC-2. */
	HPI_FORMAT_DOLBY_AC2 =		6,
/** Dolbt AC-3. */
	HPI_FORMAT_DOLBY_AC3 =		7,
/** 16-bit PCM big-endian. */
	HPI_FORMAT_PCM16_BIGENDIAN =	8,
/** TAGIT-1 algorithm - hits. */
	HPI_FORMAT_AA_TAGIT1_HITS =	9,
/** TAGIT-1 algorithm - inserts. */
	HPI_FORMAT_AA_TAGIT1_INSERTS =	10,
/** 32-bit signed PCM. Windows equivalent is WAVE_FORMAT_PCM.
Each sample is a 32bit word. The most significant 24 bits contain a 24-bit
sample and the least significant 8 bits are set to 0.
*/
	HPI_FORMAT_PCM32_SIGNED =	11,
/** Raw bitstream - unknown format. */
	HPI_FORMAT_RAW_BITSTREAM =	12,
/** TAGIT-1 algorithm hits - extended. */
	HPI_FORMAT_AA_TAGIT1_HITS_EX1 =	13,
/** 32-bit PCM as an IEEE float. Windows equivalent is WAVE_FORMAT_IEEE_FLOAT.
Each sample is a 32bit word in IEEE754 floating point format.
The range is +1.0 to -1.0, which corresponds to digital fullscale.
*/
	HPI_FORMAT_PCM32_FLOAT =	14,
/** 24-bit PCM signed. Windows equivalent is WAVE_FORMAT_PCM. */
	HPI_FORMAT_PCM24_SIGNED =	15,
/** OEM format 1 - private. */
	HPI_FORMAT_OEM1 =		16,
/** OEM format 2 - private. */
	HPI_FORMAT_OEM2 =		17,
/** Undefined format. */
	HPI_FORMAT_UNDEFINED =	0xffff
};

/*******************************************/
/** Stream States
\ingroup stream
*/
enum HPI_STREAM_STATES {
	/** State stopped - stream is stopped. */
	HPI_STATE_STOPPED = 1,
	/** State playing - stream is playing audio. */
	HPI_STATE_PLAYING = 2,
	/** State recording - stream is recording. */
	HPI_STATE_RECORDING = 3,
	/** State drained - playing stream ran out of data to play. */
	HPI_STATE_DRAINED = 4,
	/** State generate sine - to be implemented. */
	HPI_STATE_SINEGEN = 5,
	/** State wait - used for inter-card sync to mean waiting for all
		cards to be ready. */
	HPI_STATE_WAIT = 6
};
/*******************************************/
/** Source node types
\ingroup mixer
*/
enum HPI_SOURCENODES {
	/** This define can be used instead of 0 to indicate
	that there is no valid source node. A control that
	exists on a destination node can be searched for using a source
	node value of either 0, or HPI_SOURCENODE_NONE */
	HPI_SOURCENODE_NONE =		100,
#ifdef HPI_BUILD_INCLUDE_DEPRECATED
	/** \deprecated Use HPI_SOURCENODE_NONE instead. */
	HPI_SOURCENODE_BASE =		100,
#endif
	/** Out Stream (Play) node. */
	HPI_SOURCENODE_OSTREAM =	101,
	/** Line in node - could be analog, AES/EBU or network. */
	HPI_SOURCENODE_LINEIN =		102,
	HPI_SOURCENODE_AESEBU_IN =	103, /**< AES/EBU input node. */
	HPI_SOURCENODE_TUNER =		104, /**< Tuner node. */
	HPI_SOURCENODE_RF =		105, /**< RF input node. */
	HPI_SOURCENODE_CLOCK_SOURCE =	106, /**< Clock source node. */
	HPI_SOURCENODE_RAW_BITSTREAM =	107, /**< Raw bitstream node. */
	HPI_SOURCENODE_MICROPHONE = 	108, /**< Microphone node. */
	/** Cobranet input node -
	    Audio samples come from the Cobranet network and into the device. */
	HPI_SOURCENODE_COBRANET =	109,
	HPI_SOURCENODE_ANALOG =		110, /**< Analog input node. */
	HPI_SOURCENODE_ADAPTER =	111, /**< Adapter node. */
	/** RTP stream input node - This node is a destination for
	    packets of RTP audio samples from other devices. */
	HPI_SOURCENODE_RTP_DESTINATION =	112,
	HPI_SOURCENODE_INTERNAL =	113, /**< Node internal to the device. */
	HPI_SOURCENODE_AVB_STREAM =	114, /**< AVB input stream */
	HPI_SOURCENODE_BLULINK =	115, /**< BLU-Link input channel */
	HPI_SOURCENODE_AVB_AUDIO =      116, /**< AVB audio. */
	/* !!!Update this  AND hpidebug.h if you add a new sourcenode type!!! */
	HPI_SOURCENODE_LAST_INDEX =	116  /**< largest ID */
	/* AX6 max sourcenode types = 31 (see #define AX6_MAX_NODE_INDEX in axctrl_base.c) */
};

/*******************************************/
/** Destination node types
\ingroup mixer
*/
enum HPI_DESTNODES {
	/** This define can be used instead of 0 to indicate
	that there is no valid destination node. A control that
	exists on a source node can be searched for using a destination
	node value of either 0, or HPI_DESTNODE_NONE */
	HPI_DESTNODE_NONE =		200,
#ifdef HPI_BUILD_INCLUDE_DEPRECATED
	/** \deprecated Use HPI_DESTNODE_NONE instead. */
	 HPI_DESTNODE_BASE =		200,
#endif
	/** In Stream (Record) node. */
	HPI_DESTNODE_ISTREAM =		201,
	HPI_DESTNODE_LINEOUT =		202, /**< Line Out node. */
	HPI_DESTNODE_AESEBU_OUT =	203, /**< AES/EBU output node. */
	HPI_DESTNODE_RF =		204, /**< RF output node. */
	HPI_DESTNODE_SPEAKER =		205, /**< Speaker output node. */
	/** Cobranet output node -
	    Audio samples from the device are sent out on the Cobranet network.*/
	HPI_DESTNODE_COBRANET =		206,
	HPI_DESTNODE_ANALOG =		207, /**< Analog output node. */
	/** RTP stream output node - This node is a source for
	    packets of RTP audio samples that are sent to other devices. */
	HPI_DESTNODE_RTP_SOURCE =	208,
	HPI_DESTNODE_AVB_STREAM =	209, /**< AVB output stream */
	HPI_DESTNODE_INTERNAL =		210, /**< Node internal to the device. */
	HPI_DESTNODE_BLULINK =		211, /**< BLU-Link output channel. */
	HPI_DESTNODE_AVB_AUDIO =        212, /**< AVB audio. */
	/* !!!Update this AND hpidebug.h if you add a new destnode type!!! */
	HPI_DESTNODE_LAST_INDEX =	212  /**< largest ID */
	/* AX6 max sourcenode types = 31 (see #define AX6_MAX_NODE_INDEX in axctrl_base.c) */
};

/*******************************************/
/** Mixer control types
\ingroup mixer
*/
enum HPI_CONTROLS {
	HPI_CONTROL_GENERIC	 = 0,	/**< Generic control. */
	HPI_CONTROL_CONNECTION = 1, /**< A connection between nodes. */
	HPI_CONTROL_VOLUME	 = 2 ,/**< Volume control - works in dBFs. */
	HPI_CONTROL_METER = 3,	/**< Peak meter control. */
	HPI_CONTROL_MUTE = 4,	/*Mute control - not used at present. */
	HPI_CONTROL_MULTIPLEXER = 5,	/**< Multiplexer control. */

	HPI_CONTROL_AESEBU_TRANSMITTER = 6, /**< AES/EBU transmitter control */
	HPI_CONTROL_AESEBUTX = 6, /* HPI_CONTROL_AESEBU_TRANSMITTER */

	HPI_CONTROL_AESEBU_RECEIVER = 7, /**< AES/EBU receiver control. */
	HPI_CONTROL_AESEBURX = 7, /* HPI_CONTROL_AESEBU_RECEIVER */

	HPI_CONTROL_LEVEL = 8, /**< Level/trim control - works in dBu. */
	HPI_CONTROL_TUNER = 9,	/**< Tuner control. */
/*	HPI_CONTROL_ONOFFSWITCH =	10 */
	HPI_CONTROL_VOX = 11,	/**< Vox control. */
/*	HPI_CONTROL_AES18_TRANSMITTER = 12 */
/*	HPI_CONTROL_AES18_RECEIVER = 13 */
/*	HPI_CONTROL_AES18_BLOCKGENERATOR  = 14 */
	HPI_CONTROL_CHANNEL_MODE = 15,	/**< Channel mode control. */

	HPI_CONTROL_BITSTREAM = 16,	/**< Bitstream control. */
	HPI_CONTROL_SAMPLECLOCK = 17,	/**< Sample clock control. */
	HPI_CONTROL_MICROPHONE	 = 18,	/**< Microphone control. */
	HPI_CONTROL_PARAMETRIC_EQ = 19,	/**< Parametric EQ control. */
	HPI_CONTROL_EQUALIZER = 19, /*HPI_CONTROL_PARAMETRIC_EQ*/

	HPI_CONTROL_COMPANDER = 20,	/**< Compander control. */
	HPI_CONTROL_COBRANET = 21,	/**< Cobranet control. */
	HPI_CONTROL_TONEDETECTOR = 22,	/**< Tone detector control. */
	HPI_CONTROL_SILENCEDETECTOR = 23,	/**< Silence detector control. */
	HPI_CONTROL_PAD = 24,	/**< Tuner PAD control. */
	HPI_CONTROL_SRC	 = 25,	/**< Samplerate converter control. */
	HPI_CONTROL_UNIVERSAL = 26,	/**< Universal control. */

/*  !!! Update this AND hpidebug.h if you add a new control type!!!*/
	HPI_CONTROL_LAST_INDEX = 26 /**<Highest control type ID */

/* WARNING types 256 or greater impact bit packing in all AX6 DSP code */
};

/*******************************************/
/** Adapter properties
These are used in HPI_AdapterSetProperty() and HPI_AdapterGetProperty()
\ingroup adapter
*/
enum HPI_ADAPTER_PROPERTIES {
/** \internal Used in dwProperty field of HPI_AdapterSetProperty() and
HPI_AdapterGetProperty(). This errata applies to all ASI6000 cards with both
analog and digital outputs. The CS4224 A/D+D/A has a one sample delay between
left and right channels on both its input (ADC) and output (DAC).
More details are available in Cirrus Logic errata ER284B2.
PDF available from www.cirrus.com, released by Cirrus in 2001.
*/
	HPI_ADAPTER_PROPERTY_ERRATA_1 = 1,

/** Adapter grouping property
Indicates whether the adapter supports the grouping API (for ASIO and SSX2)
*/
	HPI_ADAPTER_PROPERTY_GROUPING =	2,

/** Driver SSX2 property
Tells the kernel driver to turn on SSX2 stream mapping.
This feature is not used by the DSP. In fact the call is completely processed
by the driver and is not passed on to the DSP at all.
*/
	HPI_ADAPTER_PROPERTY_ENABLE_SSX2 = 3,

/** Adapter SSX2 property
Indicates the state of the adapter's SSX2 setting. This setting is stored in
non-volatile memory on the adapter. A typical call sequence would be to use
HPI_ADAPTER_PROPERTY_SSX2_SETTING to set SSX2 on the adapter and then to reload
the driver. The driver would query HPI_ADAPTER_PROPERTY_SSX2_SETTING during
startup and if SSX2 is set, it would then call HPI_ADAPTER_PROPERTY_ENABLE_SSX2
to enable SSX2 stream mapping within the kernel level of the driver.
*/
	HPI_ADAPTER_PROPERTY_SSX2_SETTING = 4,

/** Enables/disables PCI(e) IRQ.
A setting of 0 indicates that no interrupts are being generated. A DSP boot
this property is set to 0. Setting to a non-zero value specifies the number
of frames of audio that should be processed between interrupts. This property
should be set to multiple of the mixer interval as read back from the
HPI_ADAPTER_PROPERTY_INTERVAL property.
*/
	HPI_ADAPTER_PROPERTY_IRQ_RATE = 5,

/** Base number for readonly properties */
	HPI_ADAPTER_PROPERTY_READONLYBASE = 256,

/** Readonly adapter latency property.
This property returns in the input and output latency in samples.
Property 1 is the estimated input latency
in samples, while Property 2 is that output latency in	samples.
*/
	HPI_ADAPTER_PROPERTY_LATENCY = 256,

/** Readonly adapter granularity property.
The granulariy is the smallest size chunk of stereo samples that is processed by
the adapter.
This property returns the record granularity in samples in Property 1.
Property 2 returns the play granularity.
*/
	HPI_ADAPTER_PROPERTY_GRANULARITY = 257,

/** Readonly adapter number of current channels property.
Property 1 is the number of record channels per record device.
Property 2 is the number of play channels per playback device.*/
	HPI_ADAPTER_PROPERTY_CURCHANNELS = 258,

/** Readonly adapter software version.
The SOFTWARE_VERSION property returns the version of the software running
on the adapter as Major.Minor.Release.
Property 1 contains Major in bits 15..8 and Minor in bits 7..0.
Property 2 contains Release in bits 7..0. */
	HPI_ADAPTER_PROPERTY_SOFTWARE_VERSION =259,

/** Readonly adapter MAC address MSBs.
The MAC_ADDRESS_MSB property returns
the most significant 32 bits of the MAC address.
Property 1 contains bits 47..32 of the MAC address.
Property 2 contains bits 31..16 of the MAC address. */
	HPI_ADAPTER_PROPERTY_MAC_ADDRESS_MSB = 260,

/** Readonly adapter MAC address LSBs
The MAC_ADDRESS_LSB property returns
the least significant 16 bits of the MAC address.
Property 1 contains bits 15..0 of the MAC address. */
	HPI_ADAPTER_PROPERTY_MAC_ADDRESS_LSB = 261,

/** Readonly extended adapter type number
The EXTENDED_ADAPTER_TYPE property returns the 4 digits of an extended
adapter type, i.e ASI8920-0022, 0022 is the extended type.
The digits are returned as ASCII characters rather than the hex digits that
are returned for the main type
Property 1 returns the 1st two (left most) digits, i.e "00"
in the example above, the upper byte being the left most digit.
Property 2 returns the 2nd two digits, i.e "22" in the example above*/
	HPI_ADAPTER_PROPERTY_EXTENDED_ADAPTER_TYPE = 262,

/** Readonly debug log buffer information */
	HPI_ADAPTER_PROPERTY_LOGTABLEN = 263,
	HPI_ADAPTER_PROPERTY_LOGTABBEG = 264,

/** Readonly adapter IP address
For 192.168.1.101
Property 1 returns the 1st two (left most) digits, i.e 192*256 + 168
in the example above, the upper byte being the left most digit.
Property 2 returns the 2nd two digits, i.e 1*256 + 101 in the example above, */
	HPI_ADAPTER_PROPERTY_IP_ADDRESS = 265,

/** Readonly adapter buffer processed count. Returns a buffer processed count
that is incremented every time all buffers for all streams are updated. This
is useful for checking completion of all stream operations across the adapter
when using grouped streams.
*/
	HPI_ADAPTER_PROPERTY_BUFFER_UPDATE_COUNT = 266,

/** Readonly mixer and stream intervals

These intervals are  measured in mixer frames.
To convert to time, divide  by the adapter samplerate.

The mixer interval is the number of frames processed in one mixer iteration.
The stream update interval is the interval at which streams check for and
process data, and BBM host buffer counters are updated.

Property 1 is the mixer interval in mixer frames.
Property 2 is the stream update interval in mixer frames.
*/
	HPI_ADAPTER_PROPERTY_INTERVAL = 267,
/** Adapter capabilities 1
Property 1 - adapter can do multichannel (SSX1)
Property 2 - adapter can do stream grouping (supports SSX2)
*/
	HPI_ADAPTER_PROPERTY_CAPS1  = 268,
/** Adapter capabilities 2
Property 1 - adapter can do samplerate conversion (MRX)
Property 2 - adapter can do timestretch (TSX)
*/
	HPI_ADAPTER_PROPERTY_CAPS2  = 269,

/** Readonly adapter sync header connection count.
*/
	HPI_ADAPTER_PROPERTY_SYNC_HEADER_CONNECTIONS  = 270,
/** Readonly supports SSX2 property.
Indicates the adapter supports SSX2 in some mode setting. The
return value is true (1) or false (0). If the current adapter
mode is MONO SSX2 is disabled, even though this property will
return true.
*/
	HPI_ADAPTER_PROPERTY_SUPPORTS_SSX2 = 271,
/** Readonly supports PCI(e) IRQ.
Indicates that the adapter in it's current mode supports interrupts
across the host bus. Note, this does not imply that interrupts are
enabled. Instead it indicates that they can be enabled.
*/
	HPI_ADAPTER_PROPERTY_SUPPORTS_IRQ = 272,
/** Readonly supports firmware updating.
Indicates that the adapter implements an interface to update firmware
on the adapter.
*/
	HPI_ADAPTER_PROPERTY_SUPPORTS_FW_UPDATE = 273,
/** Readonly Firmware IDs
Identifiy firmware independent of individual adapter type.
May be used as a filter for firmware update images.
Property 1 = Bootloader ID
Property 2 = Main program ID
*/
	HPI_ADAPTER_PROPERTY_FIRMWARE_ID = 274,
/** Readonly OutStream formats supported
Return bitmap indicating HPI_FORMATS supported by this
audio adapter
Property 1 = MS short of bitmap of supported OutStream formats
Property 2 = LS short of bitmap of supported OutStream formats
*/
	HPI_ADAPTER_PROPERTY_OUTSTREAM_FORMATS_SUPPORTED = 275,
	/** Readonly InStream formats supported
	Return bitmap indicating HPI_FORMATS supported by this
	audio adapter
	Property 1 = MS short of bitmap of supported InStream formats
	Property 2 = LS short of bitmap of supported InStream formats
	*/
	HPI_ADAPTER_PROPERTY_INSTREAM_FORMATS_SUPPORTED = 276,
	/** Readonly Stream host buffer required
	Property 1 = Nonzero if host buffer is required for recording
	Property 2 = Nonzero if host buffer is required for playing
	*/
	HPI_ADAPTER_PROPERTY_STREAM_HOST_BUFFER_REQUIRED = 277
};

/** Adapter mode commands

Used in wQueryOrSet parameter of HPI_AdapterSetModeEx().
\ingroup adapter
*/
enum HPI_ADAPTER_MODE_CMDS {
	/** Set the mode to the given parameter */
	HPI_ADAPTER_MODE_SET = 0,
	/** Return 0 or error depending whether mode is valid,
	but don't set the mode */
	HPI_ADAPTER_MODE_QUERY = 1
};

/** Adapter Modes
 These are used by HPI_AdapterSetModeEx()

\warning - more than 16 possible modes breaks
a bitmask in the Windows WAVE DLL
\ingroup adapter
*/
enum HPI_ADAPTER_MODES {
/** 4 outstream mode.
- ASI6114: 1 instream
- ASI6044: 4 instreams
- ASI6012: 1 instream
- ASI6102: no instreams
- ASI6022, ASI6122: 2 instreams
- ASI5111, ASI5101: 2 instreams
- ASI652x, ASI662x: 2 instreams
- ASI654x, ASI664x: 4 instreams
*/
	HPI_ADAPTER_MODE_4OSTREAM  = 1,

/** 6 outstream mode.
- ASI6012: 1 instream,
- ASI6022, ASI6122: 2 instreams
- ASI652x, ASI662x: 4 instreams
*/
	HPI_ADAPTER_MODE_6OSTREAM  = 2,

/** 8 outstream mode.
- ASI6114: 8 instreams
- ASI6118: 8 instreams
- ASI6585: 8 instreams
*/
	HPI_ADAPTER_MODE_8OSTREAM = 3,

/** 16 outstream mode.
- ASI6416 16 instreams
- ASI6518, ASI6618 16 instreams
- ASI6118 16 mono out and in streams
*/
	HPI_ADAPTER_MODE_16OSTREAM = 4,

/** one outstream mode.
- ASI5111 1 outstream, 1 instream
*/
	HPI_ADAPTER_MODE_1OSTREAM = 5,

/** ASI504X mode 1. 12 outstream, 4 instream 0 to 48kHz sample rates
	(see ASI504X datasheet for more info).
*/
	HPI_ADAPTER_MODE_1 = 6,

/** ASI504X mode 2. 4 outstreams, 4 instreams at 0 to 192kHz sample rates
	(see ASI504X datasheet for more info).
*/
	HPI_ADAPTER_MODE_2 = 7,

/** ASI504X mode 3. 4 outstreams, 4 instreams at 0 to 192kHz sample rates
	(see ASI504X datasheet for more info).
*/
	HPI_ADAPTER_MODE_3 = 8,

/** ASI504X multichannel mode.
	2 outstreams -> 4 line outs = 1 to 8 channel streams),
	4 lineins -> 1 instream (1 to 8 channel streams) at 0-48kHz.
	For more info see the SSX Specification.
*/
	HPI_ADAPTER_MODE_MULTICHANNEL = 9,

/** 12 outstream mode.
- ASI6514, ASI6614: 2 instreams
- ASI6540,ASI6544: 8 instreams
- ASI6640,ASI6644: 8 instreams
*/
	HPI_ADAPTER_MODE_12OSTREAM = 10,

/** 9 outstream mode.
- ASI6044: 8 instreams
*/
	HPI_ADAPTER_MODE_9OSTREAM = 11,

/** mono mode.
- ASI6416: 16 outstreams/instreams
- ASI5402: 2 outstreams/instreams
*/
	HPI_ADAPTER_MODE_MONO = 12,

/** Low latency mode.
- ASI6416/ASI6316: 1 16 channel outstream and instream
*/
	HPI_ADAPTER_MODE_LOW_LATENCY = 13,

/** 24 outstream mode.
- ASI6788/5788
*/
	HPI_ADAPTER_MODE_24OSTREAM = 14,

/** 32 outstream mode.
- ASI1100
*/
	HPI_ADAPTER_MODE_32OSTREAM = 15
};

/* Note, adapters can have more than one capability -
encoding as bitfield is recommended. */
#define HPI_CAPABILITY_NONE		(0)
#define HPI_CAPABILITY_MPEG_LAYER3	(1)

/* Set this equal to maximum capability index,
Must not be greater than 32 - see axnvdef.h */
#define HPI_CAPABILITY_MAX			1
/* #define HPI_CAPABILITY_AAC		   2 */

/******************************************* STREAM ATTRIBUTES ****/

/** MPEG Ancillary Data modes

The mode for the ancillary data insertion or extraction to operate in.
\ingroup stream
*/
enum HPI_MPEG_ANC_MODES {
	/** the MPEG frames have energy information stored in them (5 bytes per stereo frame, 3 per mono) */
	HPI_MPEG_ANC_HASENERGY = 0,
	/** the entire ancillary data field is taken up by data from the Anc data buffer
	On encode, the encoder will insert the energy bytes before filling the remainder
	of the ancillary data space with data from the ancillary data buffer.
	*/
	HPI_MPEG_ANC_RAW = 1
};

/** Ancillary Data Alignment
\ingroup instream
*/
enum HPI_ISTREAM_MPEG_ANC_ALIGNS {
	/** data is packed against the end of data, then padded to the end of frame */
	HPI_MPEG_ANC_ALIGN_LEFT = 0,
	/** data is packed against the end of the frame */
	HPI_MPEG_ANC_ALIGN_RIGHT = 1
};

/** MPEG modes
MPEG modes - can be used optionally for HPI_FormatCreate()
parameter dwAttributes.

Using any mode setting other than HPI_MPEG_MODE_DEFAULT
with single channel format will return an error.
\ingroup stream
*/
enum HPI_MPEG_MODES {
/** Causes the MPEG-1 Layer II bitstream to be recorded
in single_channel mode when the number of channels is 1 and in stereo when the
number of channels is 2. */
	HPI_MPEG_MODE_DEFAULT = 0,
	/** Standard stereo without joint-stereo compression */
	HPI_MPEG_MODE_STEREO = 1,
	/** Joint stereo  */
	HPI_MPEG_MODE_JOINTSTEREO = 2,
	/** Left and Right channels are completely independent */
	HPI_MPEG_MODE_DUALCHANNEL = 3
};
/******************************************* MIXER ATTRIBUTES ****/

/* \defgroup mixer_flags Mixer flags for HPI_MIXER_GET_CONTROL_MULTIPLE_VALUES
{
*/
#define HPI_MIXER_GET_CONTROL_MULTIPLE_CHANGED	(0)
#define HPI_MIXER_GET_CONTROL_MULTIPLE_RESET	(1)
/*}*/

/** Commands used by HPI_MixerStore()
\ingroup mixer
*/
enum HPI_MIXER_STORE_COMMAND {
/** Save all mixer control settings. */
	HPI_MIXER_STORE_SAVE = 1,
/** Restore all controls from saved. */
	HPI_MIXER_STORE_RESTORE = 2,
/** Delete saved control settings. */
	HPI_MIXER_STORE_DELETE = 3,
/** Enable auto storage of some control settings. */
	HPI_MIXER_STORE_ENABLE = 4,
/** Disable auto storage of some control settings. */
	HPI_MIXER_STORE_DISABLE = 5,
/** Unimplemented - save the attributes of a single control. */
	HPI_MIXER_STORE_SAVE_SINGLE = 6
};

/****************************/
/* CONTROL ATTRIBUTE VALUES */
/****************************/

/** Used by mixer plugin enable functions

E.g. HPI_ParametricEQ_SetState()
\ingroup mixer
*/
enum HPI_SWITCH_STATES {
	HPI_SWITCH_OFF = 0,	/**< Turn the mixer plugin on. */
	HPI_SWITCH_ON =1	/**< Turn the mixer plugin off. */
};

/* Volume control special gain values */

/** volumes units are 100ths of a dB
\ingroup volume
*/
#define HPI_UNITS_PER_dB		100
/** turns volume control OFF or MUTE
\ingroup volume
*/
#define HPI_GAIN_OFF			(-100 * HPI_UNITS_PER_dB)

/** channel mask specifying all channels
\ingroup volume
*/
#define HPI_BITMASK_ALL_CHANNELS	(0xFFFFFFFF)

/** value returned for no signal
\ingroup meter
*/
#define HPI_METER_MINIMUM		(-150 * HPI_UNITS_PER_dB)

/** autofade profiles
\ingroup volume
*/
enum HPI_VOLUME_AUTOFADES {
/** log fade - dB attenuation changes linearly over time */
	HPI_VOLUME_AUTOFADE_LOG = 2,
/** linear fade - amplitude changes linearly */
	HPI_VOLUME_AUTOFADE_LINEAR	= 3
};

/** The physical encoding format of the AESEBU I/O.

Used in HPI_AESEBU_Transmitter_SetFormat(), HPI_AESEBU_Receiver_SetFormat()
along with related Get and Query functions
\ingroup aestx
*/
enum  HPI_AESEBU_FORMATS {
/** AES/EBU physical format - AES/EBU balanced "professional"  */
	HPI_AESEBU_FORMAT_AESEBU = 1,
/** AES/EBU physical format - S/PDIF unbalanced "consumer"	*/
	HPI_AESEBU_FORMAT_SPDIF	 = 2
};

/** AES/EBU error status bits

Returned by HPI_AESEBU_Receiver_GetErrorStatus()
\ingroup aesrx
*/
enum HPI_AESEBU_ERRORS {
/**  bit0: 1 when PLL is not locked */
	HPI_AESEBU_ERROR_NOT_LOCKED	 = 0x01,
/**  bit1: 1 when signal quality is poor */
	HPI_AESEBU_ERROR_POOR_QUALITY = 0x02,
/** bit2: 1 when there is a parity error */
	HPI_AESEBU_ERROR_PARITY_ERROR = 0x04,
/**  bit3: 1 when there is a bi-phase coding violation */
	HPI_AESEBU_ERROR_BIPHASE_VIOLATION = 0x08,
/**  bit4: 1 when the validity bit is high */
	HPI_AESEBU_ERROR_VALIDITY = 0x10,
/**  bit5: 1 when the CRC error bit is high */
	HPI_AESEBU_ERROR_CRC = 0x20
};

/** \addtogroup pad
\{
*/
/** The text string containing the station/channel combination. */
#define HPI_PAD_CHANNEL_NAME_LEN	16
/** The text string containing the artist. */
#define HPI_PAD_ARTIST_LEN		64
/** The text string containing the title. */
#define HPI_PAD_TITLE_LEN		64
/** The text string containing the comment. */
#define HPI_PAD_COMMENT_LEN		256
/** The PTY when the tuner has not received any PTY. */
#define HPI_PAD_PROGRAM_TYPE_INVALID	0xffff
/** \} */

/** Data types for PTY string translation.
\ingroup rds
*/
enum eHPI_RDS_type {
	HPI_RDS_DATATYPE_RDS = 0,	/**< RDS bitstream.*/
	HPI_RDS_DATATYPE_RBDS = 1	/**< RBDS bitstream.*/
};

/** Tuner bands

Used for HPI_Tuner_SetBand(),HPI_Tuner_GetBand()
\ingroup tuner
*/
enum HPI_TUNER_BAND {
	HPI_TUNER_BAND_AM = 1,	 /**< AM band */
	HPI_TUNER_BAND_FM = 2,	 /**< FM band (mono) */
	HPI_TUNER_BAND_TV_NTSC_M = 3,	 /**< NTSC-M TV band*/
	HPI_TUNER_BAND_TV = 3,	/* use TV_NTSC_M */
	HPI_TUNER_BAND_FM_STEREO = 4,	 /**< FM band (stereo) */
	HPI_TUNER_BAND_AUX = 5,	 /**< Auxiliary input */
	HPI_TUNER_BAND_TV_PAL_BG = 6,	 /**< PAL-B/G TV band*/
	HPI_TUNER_BAND_TV_PAL_I = 7,	 /**< PAL-I TV band*/
	HPI_TUNER_BAND_TV_PAL_DK = 8,	 /**< PAL-D/K TV band*/
	HPI_TUNER_BAND_TV_SECAM_L = 9,	 /**< SECAM-L TV band*/
	HPI_TUNER_BAND_DAB = 10,
	HPI_TUNER_BAND_LAST = 10 /**< The index of the last tuner band. */
};

/** Tuner mode attributes

Used by HPI_Tuner_SetMode(), HPI_Tuner_GetMode()
\ingroup tuner

*/
enum HPI_TUNER_MODES {
	HPI_TUNER_MODE_RSS = 1,	/**< Control  RSS */
	HPI_TUNER_MODE_RDS = 2	/**< Control  RBDS/RDS */
};

/** Tuner mode attribute values

Used by HPI_Tuner_SetMode(), HPI_Tuner_GetMode()
\ingroup tuner
*/
enum HPI_TUNER_MODE_VALUES {
/* RSS attribute values */
	HPI_TUNER_MODE_RSS_DISABLE = 0, /**< RSS disable */
	HPI_TUNER_MODE_RSS_ENABLE  = 1, /**< RSS enable */

/* RDS mode attributes */
	HPI_TUNER_MODE_RDS_DISABLE = 0, /**< RDS - disabled */
	HPI_TUNER_MODE_RDS_RDS  = 1, /**< RDS - RDS mode */
	HPI_TUNER_MODE_RDS_RBDS = 2 /**<  RDS - RBDS mode */
};

/** Tuner Status Bits

These bitfield values are returned by a call to HPI_Tuner_GetStatus().
Multiple fields are returned from a single call.
\ingroup tuner
*/
enum HPI_TUNER_STATUS_BITS {
	HPI_TUNER_VIDEO_COLOR_PRESENT = 0x0001, /**< Video color is present. */
	HPI_TUNER_VIDEO_IS_60HZ = 0x0020, /**< 60 Hz video detected. */
	HPI_TUNER_VIDEO_HORZ_SYNC_MISSING = 0x0040, /**< Video HSYNC is missing. */
	HPI_TUNER_VIDEO_STATUS_VALID = 0x0100, /**< Video status is valid. */
	HPI_TUNER_DIGITAL = 0x0200, /**< Tuner reports digital programming. */
	HPI_TUNER_MULTIPROGRAM = 0x0400, /**< Tuner reports multiple programs. */
	HPI_TUNER_PLL_LOCKED = 0x1000, /**< The tuner's PLL is locked. */
	HPI_TUNER_FM_STEREO = 0x2000, /**< Tuner reports back FM stereo. */
	HPI_TUNER_FW_LOAD = 0x0800
};

/** Channel Modes
Used for HPI_ChannelModeSet/Get()
\ingroup channelmode
*/
enum HPI_CHANNEL_MODES {
/** Left channel out = left channel in, Right channel out = right channel in. */
	HPI_CHANNEL_MODE_NORMAL = 1,
/** Left channel out = right channel in, Right channel out = left channel in. */
	HPI_CHANNEL_MODE_SWAP = 2,
/** Left channel out = left channel in, Right channel out = left channel in. */
	HPI_CHANNEL_MODE_LEFT_TO_STEREO = 3,
/** Left channel out = right channel in, Right channel out = right channel in.*/
	HPI_CHANNEL_MODE_RIGHT_TO_STEREO = 4,
/** Left channel out = (left channel in + right channel in)/2,
    Right channel out = mute. */
	HPI_CHANNEL_MODE_STEREO_TO_LEFT = 5,
/** Left channel out = mute,
    Right channel out = (right channel in + left channel in)/2. */
	HPI_CHANNEL_MODE_STEREO_TO_RIGHT = 6,
	HPI_CHANNEL_MODE_LAST = 6
};

/** SampleClock source values
\ingroup sampleclock
*/
enum HPI_SAMPLECLOCK_SOURCES {
/** The sampleclock output is derived from its local samplerate generator.
    The local samplerate may be set using HPI_SampleClock_SetLocalRate(). */
	HPI_SAMPLECLOCK_SOURCE_LOCAL = 1,
#ifdef HPI_BUILD_INCLUDE_DEPRECATED
/** \deprecated Use HPI_SAMPLECLOCK_SOURCE_LOCAL instead */
	HPI_SAMPLECLOCK_SOURCE_ADAPTER = 1,
#endif
/** The adapter is clocked from a dedicated AES/EBU SampleClock input.*/
	HPI_SAMPLECLOCK_SOURCE_AESEBU_SYNC = 2,
/** From external wordclock connector */
	HPI_SAMPLECLOCK_SOURCE_WORD = 3,
/** Board-to-board header */
	HPI_SAMPLECLOCK_SOURCE_WORD_HEADER = 4,
/** FUTURE - SMPTE clock. */
	HPI_SAMPLECLOCK_SOURCE_SMPTE  = 5,
/** One of the aesebu inputs */
	HPI_SAMPLECLOCK_SOURCE_AESEBU_INPUT  = 6,
#ifdef HPI_BUILD_INCLUDE_DEPRECATED
/** \deprecated The first aesebu input with a valid signal
Superseded by separate Auto enable flag
*/
	HPI_SAMPLECLOCK_SOURCE_AESEBU_AUTO = 7,
#endif
/** From a network interface e.g. Cobranet or Livewire at either 48 or 96kHz */
	HPI_SAMPLECLOCK_SOURCE_NETWORK = 8,
/** From previous adjacent module (ASI2416 only)*/
	HPI_SAMPLECLOCK_SOURCE_PREV_MODULE = 10,
/** Blu link sample clock*/
	HPI_SAMPLECLOCK_SOURCE_BLULINK = 11,
/*! Update this if you add a new clock source.*/
	HPI_SAMPLECLOCK_SOURCE_LAST = 11
};

/** Equalizer filter types. Used by HPI_ParametricEQ_SetBand()
\ingroup parmeq
*/
enum HPI_FILTER_TYPE {
	HPI_FILTER_TYPE_BYPASS = 0,	/**< Filter is turned off */

	HPI_FILTER_TYPE_LOWSHELF = 1,	/**< EQ low shelf */
	HPI_FILTER_TYPE_HIGHSHELF = 2,	/**< EQ high shelf */
	HPI_FILTER_TYPE_EQ_BAND = 3,	/**< EQ gain */

	HPI_FILTER_TYPE_LOWPASS = 4,	/**< Standard low pass */
	HPI_FILTER_TYPE_HIGHPASS = 5,	/**< Standard high pass */
	HPI_FILTER_TYPE_BANDPASS = 6,	/**< Standard band pass */
	HPI_FILTER_TYPE_BANDSTOP = 7	/**< Standard band stop/notch */
};

/** Async Event sources
\ingroup async
*/
enum ASYNC_EVENT_SOURCES {
	HPI_ASYNC_EVENT_GPIO = 1,	/**< GPIO event. */
	HPI_ASYNC_EVENT_SILENCE = 2,	/**< Silence event detected. */
	HPI_ASYNC_EVENT_TONE = 3	/**< tone event detected. */
} ;
/*******************************************/
/** HPI Error codes

Almost all HPI functions return an error code
A return value of zero means there was no error.
Otherwise one of these error codes is returned.
Error codes can be converted to a descriptive string using HPI_GetErrorText()

\note When a new error code is added HPI_GetErrorText() MUST be updated.
\note Codes 1-100 are reserved for driver use
\ingroup utility
*/
enum HPI_ERROR_CODES {
	/** Message type does not exist. */
	HPI_ERROR_INVALID_TYPE = 100,
	/** Object type does not exist. */
	HPI_ERROR_INVALID_OBJ = 101,
	/** Function does not exist. */
	HPI_ERROR_INVALID_FUNC = 102,
	/** The specified object does not exist. */
	HPI_ERROR_INVALID_OBJ_INDEX = 103,
	/** Trying to access an object that has not been opened yet. */
	HPI_ERROR_OBJ_NOT_OPEN = 104,
	/** Trying to open an already open object. */
	HPI_ERROR_OBJ_ALREADY_OPEN = 105,
	/** PCI, ISA resource not valid. */
	HPI_ERROR_INVALID_RESOURCE = 106,
	/* HPI_ERROR_SUBSYSFINDADAPTERS_GETINFO= 107 */
	/** Default response was never updated with actual error code. */
	HPI_ERROR_INVALID_RESPONSE = 108,
	/** wSize field of response was not updated,
	indicating that the message was not processed. */
	HPI_ERROR_PROCESSING_MESSAGE = 109,
	/** The network did not respond in a timely manner. */
	HPI_ERROR_NETWORK_TIMEOUT = 110,
	/* An HPI handle is invalid (uninitialised?). */
	HPI_ERROR_INVALID_HANDLE = 111,
	/** A function or attribute has not been implemented yet. */
	HPI_ERROR_UNIMPLEMENTED	= 112,
	/** There are too many clients attempting
	    to access a network resource. */
	HPI_ERROR_NETWORK_TOO_MANY_CLIENTS= 113,
	/** Response buffer passed to HPI_Message
	    was smaller than returned response.
	    wSpecificError field of hpi response contains the required size.
	*/
	HPI_ERROR_RESPONSE_BUFFER_TOO_SMALL = 114,
	/** The returned response did not match the sent message */
	HPI_ERROR_RESPONSE_MISMATCH = 115,
	/** A control setting that should have been cached was not. */
	HPI_ERROR_CONTROL_CACHING = 116,
	/** A message buffer in the path to the adapter was smaller
	    than the message size.
	    wSpecificError field of hpi response contains the actual size.
	*/
	HPI_ERROR_MESSAGE_BUFFER_TOO_SMALL = 117,

	/* HPI_ERROR_TOO_MANY_ADAPTERS= 200 */
	/** Bad adpater. */
	HPI_ERROR_BAD_ADAPTER = 201,
	/** Adapter number out of range or not set properly. */
	HPI_ERROR_BAD_ADAPTER_NUMBER = 202,
	/** 2 adapters with the same adapter number. */
	HPI_ERROR_DUPLICATE_ADAPTER_NUMBER = 203,
	/** DSP code failed to bootload. Usually a DSP memory test failure. */
	HPI_ERROR_DSP_BOOTLOAD	= 204,
	/** Couldn't find or open the DSP code file. */
	HPI_ERROR_DSP_FILE_NOT_FOUND = 206,
	/** Internal DSP hardware error. */
	HPI_ERROR_DSP_HARDWARE = 207,
	/** Could not allocate memory */
	HPI_ERROR_MEMORY_ALLOC = 208,
	/** Failed to correctly load/config PLD. (unused) */
	HPI_ERROR_PLD_LOAD = 209,
	/** Unexpected end of file, block length too big etc. */
	HPI_ERROR_DSP_FILE_FORMAT = 210,

	/** Found but could not open DSP code file. */
	HPI_ERROR_DSP_FILE_ACCESS_DENIED = 211,
	/** First DSP code section header not found in DSP file. */
	HPI_ERROR_DSP_FILE_NO_HEADER = 212,
	/* HPI_ERROR_DSP_FILE_READ_ERROR= 213, */
	/** DSP code for adapter family not found. */
	HPI_ERROR_DSP_SECTION_NOT_FOUND = 214,
	/** Other OS specific error opening DSP file. */
	HPI_ERROR_DSP_FILE_OTHER_ERROR = 215,
	/** Sharing violation opening DSP code file. */
	HPI_ERROR_DSP_FILE_SHARING_VIOLATION = 216,
	/** DSP code section header had size == 0. */
	HPI_ERROR_DSP_FILE_NULL_HEADER = 217,
	/** Internal C6205 DSP hardware error. */
	HPI_ERROR_DSP_0 = 218,
	/** Internal C6713 DSP hardware error. */
	HPI_ERROR_DSP_1 = 219,

	/* HPI_ERROR_FLASH = 220, */

	/** Flash has bad checksum */
	HPI_ERROR_BAD_CHECKSUM = 221,
	HPI_ERROR_BAD_SEQUENCE = 222,
	HPI_ERROR_FLASH_ERASE = 223,
	HPI_ERROR_FLASH_PROGRAM = 224,
	HPI_ERROR_FLASH_VERIFY = 225,
	HPI_ERROR_FLASH_TYPE = 226,
	HPI_ERROR_FLASH_START = 227,
	HPI_ERROR_FLASH_READ = 228,
	HPI_ERROR_FLASH_READ_NO_FILE = 229,
	HPI_ERROR_FLASH_SIZE = 230,

	/** Reserved for OEMs. */
	HPI_ERROR_RESERVED_1	= 290,

	/* HPI_ERROR_INVALID_STREAM = 300 use HPI_ERROR_INVALID_OBJ_INDEX */
	/** Invalid compression format. */
	HPI_ERROR_INVALID_FORMAT = 301,
	/** Invalid format samplerate */
	HPI_ERROR_INVALID_SAMPLERATE = 302,
	/** Invalid format number of channels. */
	HPI_ERROR_INVALID_CHANNELS = 303,
	/** Invalid format bitrate. */
	HPI_ERROR_INVALID_BITRATE = 304,
	/** Invalid datasize used for stream read/write. */
	HPI_ERROR_INVALID_DATASIZE = 305,
	/* HPI_ERROR_BUFFER_FULL = 306 use HPI_ERROR_INVALID_DATASIZE */
	/* HPI_ERROR_BUFFER_EMPTY = 307 use HPI_ERROR_INVALID_DATASIZE */
	/** Null data pointer used for stream read/write. */
	HPI_ERROR_INVALID_DATA_POINTER = 308,
	/** Packet ordering error for stream read/write. */
	HPI_ERROR_INVALID_PACKET_ORDER = 309,

	/** Object can't do requested operation in its current
	    state, eg set format, change rec mux state while recording.*/
	HPI_ERROR_INVALID_OPERATION= 310,

	/** Where a SRG is shared amongst streams, an incompatible samplerate
	    is one that is different to any currently active stream. */
	HPI_ERROR_INCOMPATIBLE_SAMPLERATE = 311,
	/** Adapter mode is illegal.*/
	HPI_ERROR_BAD_ADAPTER_MODE = 312,

	/** There have been too many attempts to set the adapter's
	capabilities (using bad keys), the card should be returned
	to ASI if further capabilities updates are required */
	HPI_ERROR_TOO_MANY_CAPABILITY_CHANGE_ATTEMPTS = 313,
	/** Streams on different adapters cannot be grouped. */
	HPI_ERROR_NO_INTERADAPTER_GROUPS = 314,
	/** Streams on different DSPs cannot be grouped. */
	HPI_ERROR_NO_INTERDSP_GROUPS = 315,
	/** Stream wait cancelled before threshold reached. */
	HPI_ERROR_WAIT_CANCELLED = 316,
	/** A character string is invalid. */
	HPI_ERROR_INVALID_STRING = 317,
	/** A host buffer is required for this operation but it
	has not been allocated. */
	HPI_ERROR_HOST_BUFFER_NOT_ALLOCATED = 317,

	/** Invalid mixer node for this adapter. */
	HPI_ERROR_INVALID_NODE	= 400,
	/** Invalid control. */
	HPI_ERROR_INVALID_CONTROL = 401,
	/** Invalid control value was passed. */
	HPI_ERROR_INVALID_CONTROL_VALUE = 402,
	/** Control attribute not supported by this control. */
	HPI_ERROR_INVALID_CONTROL_ATTRIBUTE = 403,
	/** Control is disabled. */
	HPI_ERROR_CONTROL_DISABLED = 404,
	/** I2C transaction failed due to a missing ACK. */
	HPI_ERROR_CONTROL_I2C_MISSING_ACK= 405,
	HPI_ERROR_I2C_MISSING_ACK = 405,
#ifdef HPI_BUILD_INCLUDE_DEPRECATED
	/** Control attribute is valid, but not supported by this hardware.
	   \deprecated Use #HPI_ERROR_INVALID_CONTROL_ATTRIBUTE
	*/
	HPI_ERROR_UNSUPPORTED_CONTROL_ATTRIBUTE = 406,
#endif
	/** Control is busy, or coming out of
	reset and cannot be accessed at this time. */
	HPI_ERROR_CONTROL_NOT_READY = 407,

	/** Non volatile memory */
	HPI_ERROR_NVMEM_BUSY = 450,
	HPI_ERROR_NVMEM_FULL = 451,
	HPI_ERROR_NVMEM_FAIL = 452,

	/** I2C */
	HPI_ERROR_I2C_BAD_ADR	= 460,

	/** Entity type did not match requested type */
	HPI_ERROR_ENTITY_TYPE_MISMATCH = 470,
	/** Entity item count did not match requested count */
	HPI_ERROR_ENTITY_ITEM_COUNT = 471,
	/** Entity type is not one of the valid types */
	HPI_ERROR_ENTITY_TYPE_INVALID = 472,
	/** Entity role is not one of the valid roles */
	HPI_ERROR_ENTITY_ROLE_INVALID = 473,
	/** Entity size doesn't match target size */
	HPI_ERROR_ENTITY_SIZE_MISMATCH = 474,

	/* AES18 specific errors were 500..507 */

	/** custom error to use for debugging */
	HPI_ERROR_CUSTOM = 600,

	/** hpioct32.c can't obtain mutex */
	HPI_ERROR_MUTEX_TIMEOUT = 700,
	/** Wait for event timeout out */
	HPI_ERROR_WAIT_TIMEOUT = 701,
	/** Wait for event was interrupted */
	HPI_ERROR_WAIT_INTERRUPTED = 702,

	/** Backend errors used to be greater than this.
	    \deprecated Now, all backends return only errors defined here in hpi.h
	*/
	HPI_ERROR_BACKEND_BASE = 900,

	/** Communication with DSP failed */
	HPI_ERROR_DSP_COMMUNICATION = 900
	/* Note that the dsp communication error is set to this value so that
	   it remains compatible with any software that expects such errors
	   to be backend errors i.e. >= 900.
	   Do not define any new error codes with values > 900.
	*/
};

/** \defgroup maximums HPI maximum values
\{
*/
/** Maximum number of PCI HPI adapters */
#define HPI_MAX_ADAPTERS		20
/** Maximum number of in or out streams per adapter */
#define HPI_MAX_STREAMS			32
#define HPI_MAX_CHANNELS		2	/* per stream */
#define HPI_MAX_NODES			8	/* per mixer ? */
#define HPI_MAX_CONTROLS		4	/* per node ? */
/** maximum number of ancillary bytes per MPEG frame */
#define HPI_MAX_ANC_BYTES_PER_FRAME	(64)
#define HPI_STRING_LEN			16

/** Networked adapters have index >= 100 */
#define HPI_MIN_NETWORK_ADAPTER_IDX 100

/** Velocity units */
#define HPI_OSTREAM_VELOCITY_UNITS	4096
/** OutStream timescale units */
#define HPI_OSTREAM_TIMESCALE_UNITS	10000
/** OutStream timescale passthrough - turns timescaling on in passthough mode */
#define HPI_OSTREAM_TIMESCALE_PASSTHROUGH	99999

/**\}*/

/**************/
/* STRUCTURES */
#ifndef DISABLE_PRAGMA_PACK1
#pragma pack(push, 1)
#endif

#ifndef HPI_OS_LINUX_KERNEL
#ifndef SWIG
union hpi_version {
	struct {
		unsigned int hwassembly:3;	/**< Assembly variant 0..7 */
		unsigned int hwrev:4;		/**< encode A-P as 0-15    */
		unsigned int swminor:6;		/**< sw minor version 0-63 */
		unsigned int swmajor:3;		/**< sw major version 0-7  */
	} s;
	uint16_t w;
	uint32_t dw;
};
#endif
#endif

/** Structure containing sample format information.
    See also HPI_FormatCreate().
  */
struct hpi_format {
	uint32_t dwSampleRate;
				/**< 11025, 32000, 44100 ... */
	uint32_t dwBitRate;		/**< for MPEG */
	uint32_t dwAttributes;
				/**< Stereo/JointStereo/Mono */
	uint16_t wModeLegacy;
				/**< Legacy ancillary mode or idle bit	*/
	uint16_t wUnused;		/**< Unused */
	uint16_t wChannels;	/**< 1,2..., (or ancillary mode or idle bit */
	uint16_t wFormat;	/**< HPI_FORMAT_PCM16, _MPEG etc. See #HPI_FORMATS. */
};

#ifdef HPI_BUILD_INCLUDE_DEPRECATED
/**
Don't access fields in HPI_DATA.  Preferably switch  to
using HPI_OutStreamWriteBuf(), HPI_InStreamReadBuf().
Less desirably, always use HPI_DataCreate() to fill in the fields of
HPI_DATA.

\deprecated HPI_DATA will eventually disappear from this API
*/
struct hpi_data {
	uint32_t opaque_fixed_size[7];
};
#endif

struct hpi_anc_frame {
	uint32_t dwValidBitsInThisFrame;
	uint8_t bData[HPI_MAX_ANC_BYTES_PER_FRAME];
};

/** An object for containing a single async event.
*/
struct hpi_async_event {
	uint16_t wEventType; /**< Type of event. \sa async_event  */
	uint16_t wSequence; /**< Sequence number, allows lost event detection */
	uint32_t dwState; /**< New state */
	uint32_t hObject; /**< Handle to the object returning the event. */
	union {
		struct {
			uint16_t wIndex; /**< GPIO bit index. */
		} gpio;
		struct {
			uint16_t wNodeIndex; /**< What node is the control on ? */
			uint16_t wNodeType; /**< What type of node is the control on ? */
		} control;
	} u;
};

/* Public HPI Entity related definitions  */

struct hpi_entity;

/** Type tags used in hpi_entity structures

The value of an entity is an array of a single type.

\ingroup entity
*/
enum e_entity_type {
	entity_type_null = 0,
	/** Sequence of potentially heterogeneous entities.
	Conceptually similar to C struct.
	*/
	entity_type_sequence = 1,
	/** Reference to another entity */
	entity_type_reference = 2,
	/** A 32 bit integer */
	entity_type_int = 3,
	/** An ieee754 binary 32 bit float */
	entity_type_float = 4,
	/** An ieee754 binary 64 bit double */
	entity_type_double = 5,
	/** ASCII character.
	\note An array of these (i.e. a string) may or may not be zero terminated
	within the encoded entity */
	entity_type_cstring = 6,
	/** An octet (8 bits) */
	entity_type_octet = 7,
	/** 4 byte IP4 address */
	entity_type_ip4_address = 8,
	/** 16 byte IP6 address */
	entity_type_ip6_address = 9,
	/** 6 byte ethernet MAC address */
	entity_type_mac_address = 10,
	/** Array of booleans packed as one boolean per octet,
	using 'T' or 'F' or 'x' */
	entity_type_boolean = 11,

	LAST_ENTITY_TYPE = 12
};

/** Role tags used in hpi_entity structures

Conceptually similar to C struct member names.

\ingroup entity
*/
enum e_entity_role {
	/** no specific role */
	entity_role_null = 0,
	/** value of addressed object */
	entity_role_value = 1,
	/** the name of addressed object */
	entity_role_classname = 2,
	/** A string representing the value units
	E.g. "dB", "Hz"
	*/
	entity_role_units = 3,
	/** Entity flags
	A bitfield composed from values from ::e_entity_flag_values */
	entity_role_flags = 4,
	/** The range of a value
	For numeric values, the range is [minimum, maximum, step] */
	entity_role_range = 5,
	/** A mapping between sets of values of different types
	For instance mapping integers to strings.
	Encoded as two items, first with #entity_role_key,
	second with #entity_role_value */
	entity_role_mapping = 6,
	/** A sequence of allowed values
	for the item with #entity_role_value */
	entity_role_enum = 7,
	entity_role_instance_of = 8,
	entity_role_depends_on = 9,
	entity_role_member_of_group = 10,
	/** Sequence of entities describing constraints on the value
	Typically #entity_role_range, #entity_role_enum	*/
	entity_role_value_constraint = 11,
	/** Identifies a sequence representing a parameter port */
	entity_role_parameter_port = 12,
	/** Identifies a sequence representing a block */
	entity_role_block = 13,
	/** A label */
	entity_role_label = 14,
	/** Identifies the key part in #entity_role_mapping */
	entity_role_key = 15,
	/** Identifies a mapping between values and labels */
	entity_role_value_label = 16,
	LAST_ENTITY_ROLE = 17
	/* entity_role_node_group = , */
	/* entity_role_audio_port = , */
	/* entity_role_clock_port = , */
};

/** Flag values used with #entity_role_flags

\ingroup entity
*/
enum e_entity_flag_values {
	/** The entity value may be written (changed) */
	entity_flag_writeable = 1,
	/** The entity has a value that can be read */
	entity_flag_readable = 2,
	/** The entity value can change apart from being written by a controller */
	entity_flag_volatile = 4,
	/** The entity info can change apart from being written by a controller */
	entity_flag_volatile_info = 8
};

#ifndef HPI_OS_LINUX_KERNEL
/* DAB related structs */

/** Basic component information.
 *
 * (Detailed component information is obtained separately using
 * HPI_Tuner_GetDabComponentInfo())
 *
 * \ingroup tuner
 */
struct hpi_dab_service_component {
	uint16_t id;  ///< component ID, big-endian order
	uint8_t info;  ///< ASCTy[7:2]/ DSCTy[7:2]; Pri/Sec[1]; CA Flag[0]
	uint8_t valid;  ///< User App Info Valid[0]
};

#define DAB_LABEL_LEN 16
#define DAB_MAX_COMPONENTS 15

/** Information about a DAB service
 *  \ingroup tuner
 */
struct hpi_dab_service_info {
	uint32_t id;  ///< Service global ID, big-endian order
	uint8_t info1;  ///< SrvLinking Info Flag[6]; Pty[5:1]; P/D Flag[0]
	uint8_t info2;  ///< local[7]; CAId[6:4]; NUM_COMP[3:0]
	uint8_t info3;  ///< SlCharset[3:0]
	uint8_t num_components;  ///< extracted from info2, max 15
	char label[DAB_LABEL_LEN]; ///< label for service. (not a zero-terminated string)
	/// Array of num_components basic component information
	struct hpi_dab_service_component components[DAB_MAX_COMPONENTS];
};

/** Detailed DAB component information
 *
 * The User Application (UA) information fields are repeated up to 5 times.
 * As they are variable sized, only the first is shown in this struct. The
 * ua_data field represents the maximum possible size of 5 items.
 *
 * This number of UA Types is given in the num_ua field of this structure.
 * This is only applicable for application data transported over XPAD.
 *
 *  \ingroup tuner
 */

struct hpi_dab_component_info {
	uint8_t global_id;  ///< Global ID when concatenated with service id
	uint8_t pad1;  ///< Alignment padding
	uint8_t language;  ///< Language of this component
	uint8_t charset;  ///< Character set code for component label
	uint8_t label[DAB_LABEL_LEN];  ///< Component label (not a zero-terminated string)
	uint16_t label_abbrev_mask;  ///< Bitmap of chars from label for abbrev label
	uint8_t num_ua;  ///< Number of UA records
	uint8_t len_ua;  ///< Total length of UA info, including padding
	/* Following fields represent first of up to 5 uA items */
	uint16_t ua_type;  ///< UAType[10:0]
	uint8_t ua_datalen;  ///< Length[4:0] of ua_data
	uint8_t ua_data[250]; ///< Array, with length given by ua_datalen field
	uint8_t align_pad;  ///< Pad only present for each ua if ua_datalen is even!
};

/** Header for DAB data packets
 *
 *  \ingroup tuner
 */
struct hpi_service_data_packet_info {
	uint8_t flags;  ///< flags[1] = overflow - packets have been lost
	uint8_t buff_count;  ///< Number of packets remaining in queue
	uint8_t srv_state;  ///< Status indicator
	/** data_type[7:6] is type indicator:
	 * 0 - Standard data service, data_type[5:0] = DSCTy
	 * 1 - Non-DLS PAD, data_type[5:0] = DSCTy
	 * 2 - DLS PAD, data_type[5:0] = 0
	 * 3 - reserved
	 */
	uint8_t data_type;
	uint32_t serv_id;  ///< Service ID of data source
	uint32_t comp_id;  ///< Component ID of data source
	uint16_t ua_type;  ///< UAType[10:0]
	uint16_t byte_count;  ///< Number of bytes in payload
	uint16_t seg_num;  ///< Segment number of a segmented data block
	uint16_t num_segs;  ///< Total number of segments, zero for stream data
	// payload follows
};

#endif

#ifndef HPI_BUILD_SANITISE
/*************/
/* HPI TYPES */

struct hpi_hsubsys;
/** An HPI error code \ref HPI_ERROR_CODES */
typedef uint16_t hpi_err_t;
/** An HPI object handle */
typedef uint32_t hpi_handle_t;

typedef struct hpi_hsubsys hpi_hsubsys_t;
typedef struct hpi_format hpi_format_t;

/** Help structure for representing control location and type.
    See HPI_MixerGetControl() and HPI_MixerGetControlByIndex().
  */
struct hpi_control_t {
	uint16_t wControlType;
	uint16_t wSrcNodeType;
	uint16_t wSrcNodeIndex;
	uint16_t wDstNodeType;
	uint16_t wDstNodeIndex;
	hpi_handle_t hControl;
};

#ifndef HPI_BUILD_EXCLUDE_ALL_DEPRECATED
/* Deprecated typedefs.
Please use defined-from type directly
E.g use struct hpi_format instead of HPI_FORMAT
*/

HPI_DEPRECATED typedef struct hpi_hsubsys HPI_HSUBSYS;

HPI_DEPRECATED typedef hpi_handle_t HPI_HANDLE;
HPI_DEPRECATED typedef hpi_handle_t HPI_HOSTREAM;
HPI_DEPRECATED typedef hpi_handle_t HPI_HISTREAM;
HPI_DEPRECATED typedef hpi_handle_t HPI_HSTREAM;	/* either InStream or OutStream */
HPI_DEPRECATED typedef hpi_handle_t HPI_HMIXER;
HPI_DEPRECATED typedef hpi_handle_t HPI_HCONTROL;
HPI_DEPRECATED typedef hpi_handle_t HPI_HNVMEMORY;
HPI_DEPRECATED typedef hpi_handle_t HPI_HGPIO;
HPI_DEPRECATED typedef hpi_handle_t HPI_HWATCHDOG;
HPI_DEPRECATED typedef hpi_handle_t HPI_HCLOCK;
HPI_DEPRECATED typedef hpi_handle_t HPI_HPROFILE;
HPI_DEPRECATED typedef hpi_handle_t HPI_HASYNC;

HPI_DEPRECATED typedef hpi_err_t HPI_ERR;

HPI_DEPRECATED typedef struct hpi_format HPI_FORMAT;
#ifdef HPI_BUILD_INCLUDE_DEPRECATED
HPI_DEPRECATED typedef struct hpi_data HPI_DATA;
#endif
HPI_DEPRECATED typedef struct hpi_anc_frame HPI_ANC_FRAME;
#endif
#endif

#ifndef DISABLE_PRAGMA_PACK1
#pragma pack(pop)
#endif

/*****************/
/* HPI FUNCTIONS */
/*****************/

#ifndef HPI_OS_LINUX_KERNEL
/* Stream */
HPI_API (hpi_err_t) HPI_StreamEstimateBufferSize(
	struct hpi_format *pF,
	uint32_t dwHostPollingRateInMilliSeconds,
	uint32_t *dwRecommendedBufferSize
);

/************************/
/*  Entity manipulation */
/************************/
HPI_API (hpi_err_t) HPI_Entity_FindNext(
	struct hpi_entity *container_entity,
	enum e_entity_type type,
	enum e_entity_role role,
	int recursive_flag,
	struct hpi_entity **current_match
);

HPI_API (hpi_err_t) HPI_Entity_CopyValueFrom(
	struct hpi_entity *entity,
	enum e_entity_type type,
	size_t item_count,
	void *value_dst_p,
	size_t value_size
);

HPI_API (hpi_err_t) HPI_Entity_Unpack(
	struct hpi_entity *entity,
	enum e_entity_type *type,
	size_t *items,
	enum e_entity_role *role,
	void **value
);

HPI_API (const char *) HPI_Entity_GetTypeName(
	struct hpi_entity *entity
);

HPI_API (const char *) HPI_Entity_GetRoleName(
	struct hpi_entity *entity
);

HPI_API (hpi_err_t) HPI_Entity_AllocAndPack(
	const enum e_entity_type type,
	const size_t item_count,
	const enum e_entity_role role,
	const void *value,
	const size_t value_size,
	struct hpi_entity **entity
);

HPI_API_VOID (void) HPI_Entity_Free(
	struct hpi_entity *entity
);

/* Generic object entity */
HPI_API (hpi_err_t) HPI_Object_UriToHandle(
	const char *uri,
	hpi_handle_t *h);

HPI_API (hpi_err_t) HPI_Object_GetInfoEntity(
	hpi_handle_t h,
	struct hpi_entity **info);

HPI_API (hpi_err_t) HPI_Object_GetValueEntity(
	hpi_handle_t h,
	struct hpi_entity **value);

HPI_API (hpi_err_t) HPI_Object_SetValueEntity(
		hpi_handle_t h,
		const struct hpi_entity *value);

HPI_API (hpi_err_t) HPI_Object_GetValue(
	hpi_handle_t h,
	enum e_entity_type type,
	size_t count,
	void *value,
	size_t value_size
);

HPI_API (hpi_err_t) HPI_Object_SetValue(
	hpi_handle_t h,
	enum e_entity_type type,
	size_t count,
	const void *value,
	size_t value_size
);

HPI_API (hpi_err_t) HPI_Object_GetRole(
	hpi_handle_t hObject,
	enum e_entity_role *r
);

HPI_API (hpi_err_t) HPI_Object_GetInfo(
	hpi_handle_t hObject,
	enum e_entity_type type,
	enum e_entity_role role,
	void *value,
	size_t *value_size,
	size_t *value_count
);

HPI_API (hpi_err_t) HPI_Object_BlockParameters(
	hpi_handle_t hMixer,
	hpi_handle_t block,
	hpi_handle_t *parameters,
	size_t *param_count
);

HPI_API (hpi_err_t) HPI_Object_BlockHandle(
	hpi_handle_t hMixer,
	uint16_t wSrcNodeType,
	uint16_t wSrcNodeTypeIndex,
	uint16_t wDstNodeType,
	uint16_t wDstNodeTypeIndex,
	const char *block_name,
	hpi_handle_t * phBlock
);

HPI_API (hpi_err_t) HPI_Object_ParameterHandle(
	hpi_handle_t hMixer,
	hpi_handle_t hBlock,
	const char *parameter_name,
	hpi_handle_t * phParameter
);

HPI_API_VOID (void) HPI_MemFree(
	void *mem
);

#endif

/*************/
/* SubSystem */
/*************/
#ifndef HPI_BUILD_KERNEL_MODE
HPI_API (hpi_hsubsys_t *) HPI_SubSysCreate(
	void
);

HPI_API_VOID (void) HPI_SubSysFree(
	const hpi_hsubsys_t *phSubSys
);
#endif

HPI_API (hpi_err_t) HPI_SubSysGetVersionEx(
	const hpi_hsubsys_t *phSubSys,
	uint32_t *pdwVersionEx
);

#ifndef HPI_OS_LINUX_KERNEL
HPI_API (hpi_err_t) HPI_SubSysGetNumAdapters(
	const hpi_hsubsys_t *phSubSys,
	int *pnNumAdapters
);

HPI_API (hpi_err_t) HPI_SubSysGetAdapter(
	const hpi_hsubsys_t *phSubSys,
	int nIterator,
	uint32_t *pdwAdapterIndex,
	uint16_t *pwAdapterType
);

HPI_API (uint16_t) HPI_SubSysSsx2Bypass(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wBypass
);

HPI_API (hpi_err_t) HPI_SubSysSetHostNetworkInterface(
	const hpi_hsubsys_t *phSubSys,
	const char *szInterface
);

#endif

#ifdef HPI_BUILD_INCLUDE_DEPRECATED
HPI_DEPRECATED HPI_API (hpi_err_t) HPI_SubSysGetVersion(
	const hpi_hsubsys_t *phSubSys,
	uint32_t *pdwVersion
);

HPI_DEPRECATED HPI_API (hpi_err_t) HPI_SubSysGetInfo(
	const hpi_hsubsys_t *phSubSys,
	uint32_t *pdwVersion,
	uint16_t *pwNumAdapters,
	uint16_t awAdapterList[],
	uint16_t wListLength
);

HPI_DEPRECATED HPI_API (hpi_err_t) HPI_SubSysFindAdapters(
	const hpi_hsubsys_t *phSubSys,
	uint16_t *pwNumAdapters,
	uint16_t awAdapterList[],
	uint16_t wListLength
);
#endif
/***********/
/* Adapter */
/***********/

#ifndef HPI_OS_LINUX_KERNEL
HPI_API (hpi_err_t) HPI_AdapterOpen(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex
);

HPI_API_VOID (hpi_err_t) HPI_AdapterClose(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex
);
#endif

HPI_API (hpi_err_t) HPI_AdapterGetInfo(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	uint16_t *pwNumOutStreams,
	uint16_t *pwNumInStreams,
	uint16_t *pwVersion,
	uint32_t *pdwSerialNumber,
	uint16_t *pwAdapterType
);

HPI_API_VOID (hpi_err_t) HPI_AdapterRestart(
	uint16_t wAdapterIndex
);

#ifndef HPI_OS_LINUX_KERNEL
HPI_API (hpi_err_t) HPI_AdapterGetModuleByIndex(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	uint16_t wModuleIndex,
	uint16_t *pwNumOutputs,
	uint16_t *pwNumInputs,
	uint16_t *pwVersion,
	uint32_t *pdwSerialNumber,
	uint16_t *pwModuleType,
	hpi_handle_t * phModule
);

HPI_API (hpi_err_t) HPI_AdapterSetMode(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	uint32_t dwAdapterMode
);

HPI_API (hpi_err_t) HPI_AdapterSetModeEx(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	uint32_t dwAdapterMode,
	uint16_t wQueryOrSet
);

HPI_API (hpi_err_t) HPI_AdapterGetMode(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	uint32_t *pdwAdapterMode
);
#endif

#ifdef HPI_BUILD_INCLUDE_DEPRECATED
HPI_DEPRECATED HPI_API (hpi_err_t) HPI_AdapterGetAssert(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	uint16_t *wAssertPresent,
	char *pszAssert,
	uint16_t *pwLineNumber
);

HPI_DEPRECATED HPI_API (hpi_err_t) HPI_AdapterGetAssertEx(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	uint16_t *wAssertCount,
	char *pszAssert,
	uint32_t *pdwLineNumber,
	uint16_t *pwAssertOnDsp
);
#endif

HPI_API (hpi_err_t) HPI_AdapterGetAssert2(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	uint16_t *pAssertCount,
	char *pszAssert,
	uint32_t *pParam1,
	uint32_t *pParam2,
	uint32_t *pDspStringAddr,
	uint16_t *pProcessorId
);

HPI_API (hpi_err_t) HPI_AdapterTestAssert(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	uint16_t wAssertId
);

HPI_API (hpi_err_t) HPI_AdapterEnableCapability(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	uint16_t wCapability,
	uint32_t dwKey
);

HPI_API (hpi_err_t) HPI_AdapterSelfTest(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex
);

HPI_API (hpi_err_t) HPI_AdapterDebugRead(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
 	uint32_t dwDspAddress,
 	char *pBytes,
 	int *dwCountBytes
);

HPI_API (hpi_err_t) HPI_AdapterSetProperty(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	uint16_t wProperty,
	uint16_t wParamter1,
	uint16_t wParamter2
);

HPI_API (hpi_err_t) HPI_AdapterGetProperty(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	uint16_t wProperty,
	uint16_t *pwParamter1,
	uint16_t *pwParamter2
);

#ifdef HPI_BUILD_INCLUDE_DEPRECATED
HPI_DEPRECATED HPI_API (hpi_err_t) HPI_AdapterFindObject(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	uint16_t wObjectType,
	uint16_t wObjectIndex,
	uint16_t *pDspIndex
);
#endif

#ifndef HPI_OS_LINUX_KERNEL
HPI_API (hpi_err_t) HPI_AdapterEnumerateProperty(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	uint16_t wIndex,
	uint16_t wWhatToEnumerate,
	uint16_t wPropertyIndex,
	uint32_t *pdwSetting
);
#endif

#ifndef HPI_OS_LINUX_KERNEL
/************/
/* NvMemory */
/************/
HPI_API (hpi_err_t) HPI_NvMemoryOpen(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	hpi_handle_t * phNvMemory,
	uint16_t *pwSizeInBytes
);

HPI_API (hpi_err_t) HPI_NvMemoryReadByte(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hNvMemory,
	uint16_t wIndex,
	uint16_t *pwData
);

HPI_API (hpi_err_t) HPI_NvMemoryWriteByte(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hNvMemory,
	uint16_t wIndex,
	uint16_t wData
);

/********/
/* Gpio */
/********/
HPI_API (hpi_err_t) HPI_GpioOpen(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	hpi_handle_t * phGpio,
	uint16_t *pwNumberInputBits,
	uint16_t *pwNumberOutputBits
);

HPI_API (hpi_err_t) HPI_GpioReadBit(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hGpio,
	uint16_t wBitIndex,
	uint16_t *pwBitData
);

HPI_API (hpi_err_t) HPI_GpioReadAllBits(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hGpio,
	uint16_t awAllBitData[4]
);

HPI_API (hpi_err_t) HPI_GpioWriteBit(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hGpio,
	uint16_t wBitIndex,
	uint16_t wBitData
);

HPI_API (hpi_err_t) HPI_GpioWriteStatus(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hGpio,
	uint16_t awAllBitData[4]
);

/***************/
/* Async Event */
/***************/
HPI_API (hpi_err_t) HPI_AsyncEventOpen(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	hpi_handle_t * phAsync
);

HPI_API_VOID (hpi_err_t) HPI_AsyncEventClose(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hAsync
);

HPI_API (hpi_err_t) HPI_AsyncEventWait(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hAsync,
	uint16_t wMaximumEvents,
	struct hpi_async_event *pEvents,
	uint16_t *pwNumberReturned
);

HPI_API (hpi_err_t) HPI_AsyncEventGetCount(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hAsync,
	uint16_t *pwCount
);

HPI_API (hpi_err_t) HPI_AsyncEventGet(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hAsync,
	uint16_t wMaximumEvents,
	struct hpi_async_event *pEvents,
	uint16_t *pwNumberReturned
);

/************/
/* Watchdog */
/************/
HPI_API (hpi_err_t) HPI_WatchdogOpen(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	hpi_handle_t * phWatchdog
);

HPI_API (hpi_err_t) HPI_WatchdogSetTime(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hWatchdog,
	uint32_t dwTimeMillisec
);

HPI_API (hpi_err_t) HPI_WatchdogPing(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hWatchdog
);
#endif
/*************/
/* OutStream */
/*************/
HPI_API (hpi_err_t) HPI_OutStreamOpen(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	uint16_t wOutStreamIndex,
	hpi_handle_t * phOutStream
);

HPI_API_VOID (hpi_err_t) HPI_OutStreamClose(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream
);

#ifdef HPI_BUILD_INCLUDE_DEPRECATED
HPI_DEPRECATED HPI_API (hpi_err_t) HPI_OutStreamGetInfo(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream,
	uint16_t *pwState,
	uint32_t *pdwBufferSize,
	uint32_t *pdwDataToPlay
);
#endif

HPI_API (hpi_err_t) HPI_OutStreamGetInfoEx(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream,
	uint16_t *pwState,
	uint32_t *pdwBufferSize,
	uint32_t *pdwDataToPlay,
	uint32_t *pdwSamplesPlayed,
	uint32_t *pdwAuxiliaryDataToPlay
);

HPI_API (hpi_err_t) HPI_OutStreamWriteBuf(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream,
	const uint8_t *pbWriteBuf,
	uint32_t dwBytesToWrite,
	const struct hpi_format *pFormat
);

#ifdef HPI_BUILD_INCLUDE_DEPRECATED
HPI_DEPRECATED HPI_API (hpi_err_t) HPI_OutStreamWrite(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream,
	const struct hpi_data *pData
);
#endif

HPI_API (hpi_err_t) HPI_OutStreamStart(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream
);

#ifndef HPI_OS_LINUX_KERNEL
HPI_API (hpi_err_t) HPI_OutStreamWaitStart(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream
);
#endif

HPI_API (hpi_err_t) HPI_OutStreamStop(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream
);

#ifndef HPI_OS_LINUX_KERNEL
HPI_API (hpi_err_t) HPI_OutStreamSinegen(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream
);
#endif

HPI_API (hpi_err_t) HPI_OutStreamReset(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream
);

HPI_API (hpi_err_t) HPI_OutStreamQueryFormat(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream,
	struct hpi_format *pFormat
);

HPI_API (hpi_err_t) HPI_OutStreamSetFormat(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream,
	struct hpi_format *pFormat
);

#ifndef HPI_OS_LINUX_KERNEL
HPI_API (hpi_err_t) HPI_OutStreamSetPunchInOut(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream,
	uint32_t dwPunchInSample,
	uint32_t dwPunchOutSample
);

HPI_API (hpi_err_t) HPI_OutStreamSetVelocity(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream,
	short nVelocity
);

HPI_API (hpi_err_t) HPI_OutStreamAncillaryReset(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream,
	uint16_t wMode
);

HPI_API (hpi_err_t) HPI_OutStreamAncillaryGetInfo(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream,
	uint32_t *pdwFramesAvailable
);

HPI_API (hpi_err_t) HPI_OutStreamAncillaryRead(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream,
	struct hpi_anc_frame *pAncFrameBuffer,
	uint32_t dwAncFrameBufferSizeInBytes,
	uint32_t dwNumberOfAncillaryFramesToRead
);

HPI_API (hpi_err_t) HPI_OutStreamSetTimeScale(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream,
	uint32_t dwTimeScaleX10000
);

HPI_API (hpi_err_t) HPI_OutStreamHostBufferAllocate(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream,
	uint32_t dwSizeInBytes
);

HPI_API (hpi_err_t) HPI_OutStreamHostBufferFree(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream
);
#endif

HPI_API (hpi_err_t) HPI_OutStreamGroupAdd(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream,
	hpi_handle_t hStream
);

HPI_API (hpi_err_t) HPI_OutStreamGroupGetMap(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream,
	uint32_t *pdwOutStreamMap,
	uint32_t *pdwInStreamMap
);

HPI_API (hpi_err_t) HPI_OutStreamGroupReset(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream
);

#ifndef HPI_BUILD_NO_STREAM_WAIT
HPI_API (hpi_err_t) HPI_OutStreamWait(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream,
	uint32_t dwThresholdBytesToPlay
);
#endif

HPI_API (hpi_err_t) HPI_OutStreamGetTimestamp(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hOutStream,
	uint32_t *pdwSample,
	uint32_t *pdwNanosPerSample,
	int64_t *pqwTimestamp
);

/************/
/* InStream */
/************/
HPI_API (hpi_err_t) HPI_InStreamOpen(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	uint16_t wInStreamIndex,
	hpi_handle_t * phInStream
);

HPI_API_VOID (hpi_err_t) HPI_InStreamClose(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream
);

HPI_API (hpi_err_t) HPI_InStreamQueryFormat(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream,
	const struct hpi_format *pFormat
);

HPI_API (hpi_err_t) HPI_InStreamSetFormat(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream,
	const struct hpi_format *pFormat
);

HPI_API (hpi_err_t) HPI_InStreamReadBuf(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream,
	uint8_t *pbReadBuf,
	uint32_t dwBytesToRead
);

#ifdef HPI_BUILD_INCLUDE_DEPRECATED
HPI_DEPRECATED HPI_API (hpi_err_t) HPI_InStreamRead(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream,
	struct hpi_data * pData
);
#endif

HPI_API (hpi_err_t) HPI_InStreamStart(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream
);

HPI_API (hpi_err_t) HPI_InStreamWaitStart(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream
);

HPI_API (hpi_err_t) HPI_InStreamStop(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream
);

HPI_API (hpi_err_t) HPI_InStreamReset(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream
);

#ifdef HPI_BUILD_INCLUDE_DEPRECATED
HPI_DEPRECATED HPI_API (hpi_err_t) HPI_InStreamGetInfo(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream,
	uint16_t *pwState,
	uint32_t *pdwBufferSize,
	uint32_t *pdwDataRecorded
);
#endif

HPI_API (hpi_err_t) HPI_InStreamGetInfoEx(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream,
	uint16_t *pwState,
	uint32_t *pdwBufferSize,
	uint32_t *pdwDataRecorded,
	uint32_t *pdwSamplesRecorded,
	uint32_t *pdwAuxiliaryDataRecorded
);

#ifndef HPI_OS_LINUX_KERNEL
HPI_API (hpi_err_t) HPI_InStreamAncillaryReset(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream,
	uint16_t wBytesPerFrame,
	uint16_t wMode,
	uint16_t wAlignment,
	uint16_t wIdleBit
);

HPI_API (hpi_err_t) HPI_InStreamAncillaryGetInfo(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream,
	uint32_t *pdwFrameSpace
);

HPI_API (hpi_err_t) HPI_InStreamAncillaryWrite(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream,
	const struct hpi_anc_frame *pAncFrameBuffer,
	uint32_t dwAncFrameBufferSizeInBytes,
	uint32_t dwNumberOfAncillaryFramesToWrite
);

HPI_API (hpi_err_t) HPI_InStreamHostBufferAllocate(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream,
	uint32_t dwSizeInBytes
);
#endif

HPI_API (hpi_err_t) HPI_InStreamHostBufferFree(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream
);

HPI_API (hpi_err_t) HPI_InStreamGroupAdd(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream,
	hpi_handle_t hStream
);

HPI_API (hpi_err_t) HPI_InStreamGroupGetMap(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream,
	uint32_t *pdwOutStreamMap,
	uint32_t *pdwInStreamMap
);

HPI_API (hpi_err_t) HPI_InStreamGroupReset(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream
);

#ifndef HPI_BUILD_NO_STREAM_WAIT
HPI_API (hpi_err_t) HPI_InStreamWait(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream,
	uint32_t dwThresholdBytesRecorded
);
#endif

HPI_API (hpi_err_t) HPI_InStreamGetTimestamp(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hInStream,
	uint32_t *pdwSample,
	uint32_t *pdwNanosPerSample,
	int64_t *pqwTimestamp
);

/*********/
/* Mixer */
/*********/
HPI_API (hpi_err_t) HPI_MixerOpen(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	hpi_handle_t * phMixer
);

#ifndef HPI_OS_LINUX_KERNEL
HPI_API_VOID (hpi_err_t) HPI_MixerClose(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hMixer
);
#endif

HPI_API (hpi_err_t) HPI_MixerGetControl(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hMixer,
	uint16_t wSrcNodeType,
	uint16_t wSrcNodeTypeIndex,
	uint16_t wDstNodeType,
	uint16_t wDstNodeTypeIndex,
	uint16_t wControlType,
	hpi_handle_t * phControl
);

HPI_API (hpi_err_t) HPI_MixerGetControlByIndex(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hMixer,
	uint16_t wControlIndex,
	uint16_t *pwSrcNodeType,
	uint16_t *pwSrcNodeIndex,
	uint16_t *pwDstNodeType,
	uint16_t *pwDstNodeIndex,
	uint16_t *pwControlType,
	hpi_handle_t * phControl
);

#ifndef HPI_OS_LINUX_KERNEL
HPI_API (hpi_err_t) HPI_MixerStore(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hMixer,
	enum HPI_MIXER_STORE_COMMAND Command,
	uint16_t wIndex
);
HPI_API(hpi_err_t) HPI_MixerStoreStatus(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hMixer,
	uint16_t *pwControlsToStore
);
#endif
/************/
/* Controls */
/************/
/******************/
/* Volume control */
/******************/
HPI_API (hpi_err_t) HPI_VolumeSetGain(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	short anGain0_01dB[HPI_MAX_CHANNELS]
);

HPI_API (hpi_err_t) HPI_VolumeGetGain(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	short anGain0_01dB_out[HPI_MAX_CHANNELS]
);

HPI_API (hpi_err_t) HPI_VolumeSetMute(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t mute
);

HPI_API (hpi_err_t) HPI_VolumeGetMute(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *mute
);

#define HPI_VolumeGetRange HPI_VolumeQueryRange
HPI_API (hpi_err_t) HPI_VolumeQueryRange(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	short *nMinGain_01dB,
	short *nMaxGain_01dB,
	short *nStepGain_01dB
);

HPI_API (hpi_err_t) HPI_Volume_QueryChannels(
	const hpi_hsubsys_t *phSubSys,
	const hpi_handle_t hControl,
	uint32_t *pChannels
);

#ifndef HPI_OS_LINUX_KERNEL
HPI_API (hpi_err_t) HPI_VolumeAutoFade(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	short anStopGain0_01dB[HPI_MAX_CHANNELS],
	uint32_t wDurationMs
);

HPI_API (hpi_err_t) HPI_VolumeAutoFadeProfile(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	short anStopGain0_01dB[HPI_MAX_CHANNELS],
	uint32_t dwDurationMs,
	uint16_t dwProfile
);

HPI_API (hpi_err_t) HPI_Volume_QueryAutoFadeProfile(
	const hpi_hsubsys_t *phSubSys,
	const hpi_handle_t hControl,
	const uint32_t i,
	uint16_t *wProfile
);
#endif

/*****************/
/* Level control */
/*****************/
HPI_API (hpi_err_t) HPI_LevelQueryRange(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	short *nMinGain_01dB,
	short *nMaxGain_01dB,
	short *nStepGain_01dB
);

HPI_API (hpi_err_t) HPI_LevelSetGain(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	short anGain0_01dB[HPI_MAX_CHANNELS]
);

HPI_API (hpi_err_t) HPI_LevelGetGain(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	short anGain0_01dB_out[HPI_MAX_CHANNELS]
);

/*****************/
/* Meter control */
/*****************/
HPI_API (hpi_err_t) HPI_Meter_QueryChannels(
	const hpi_hsubsys_t *phSubSys,
	const hpi_handle_t hMeter,
	uint32_t *pChannels
);

HPI_API (hpi_err_t) HPI_MeterGetPeak(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	short anPeak0_01dB_out[HPI_MAX_CHANNELS]
);

#ifndef HPI_OS_LINUX_KERNEL
HPI_API (hpi_err_t) HPI_MeterGetRms(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	short anPeak0_01dB_out[HPI_MAX_CHANNELS]
);

HPI_API (hpi_err_t) HPI_MeterSetPeakBallistics(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t nAttack,
	uint16_t nDecay
);

HPI_API (hpi_err_t) HPI_MeterSetRmsBallistics(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t nAttack,
	uint16_t nDecay
);

HPI_API (hpi_err_t) HPI_MeterGetPeakBallistics(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t *nAttack,
	uint16_t *nDecay
);

HPI_API (hpi_err_t) HPI_MeterGetRmsBallistics(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t *nAttack,
	uint16_t *nDecay
);
#endif

/************************/
/* ChannelMode control */
/************************/
HPI_API (hpi_err_t) HPI_ChannelMode_QueryMode(
	const hpi_hsubsys_t *phSubSys,
	const hpi_handle_t hMode,
	const uint32_t dwIndex,
	uint16_t *pwMode
);

HPI_API (hpi_err_t) HPI_ChannelModeSet(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wMode
);

HPI_API (hpi_err_t) HPI_ChannelModeGet(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t *wMode
);

/*****************/
/* Tuner control */
/*****************/
HPI_API (hpi_err_t) HPI_Tuner_QueryBand(
	const hpi_hsubsys_t *phSubSys,
	const hpi_handle_t hTuner,
	const uint32_t dwIndex,
	uint16_t *pwBand
);

HPI_API (hpi_err_t) HPI_Tuner_SetBand(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wBand
);

HPI_API (hpi_err_t) HPI_Tuner_GetBand(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t *pwBand
);


HPI_API (hpi_err_t) HPI_Tuner_QueryFrequency(
	const hpi_hsubsys_t *phSubSys,
	const hpi_handle_t hTuner,
	const uint32_t dwIndex,
	const uint16_t band,
	uint32_t *pdwFreq
);

HPI_API (hpi_err_t) HPI_Tuner_SetFrequency(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t wFreqInkHz
);

HPI_API (hpi_err_t) HPI_Tuner_GetFrequency(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *pwFreqInkHz
);

HPI_API (hpi_err_t) HPI_Tuner_QueryGain(
	const hpi_hsubsys_t *phSubSys,
	const hpi_handle_t hTuner,
	const uint32_t dwIndex,
	uint16_t *pwGain
);

HPI_API (hpi_err_t) HPI_Tuner_SetGain(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	short nGain
);

HPI_API (hpi_err_t) HPI_Tuner_GetGain(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	short *pnGain
);

#ifndef HPI_OS_LINUX_KERNEL
HPI_API (hpi_err_t) HPI_Tuner_GetRFLevel(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	short *pwLevel
);

HPI_API (hpi_err_t) HPI_Tuner_GetRawRFLevel(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	short *pwLevel
);

#ifdef HPI_BUILD_INCLUDE_DEPRECATED
HPI_DEPRECATED HPI_API (hpi_err_t) HPI_Tuner_GetVideoStatus(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t *pwVideoStatus
);
#endif

HPI_API (hpi_err_t) HPI_Tuner_GetStatus(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t *pwStatusMask,
	uint16_t *pwStatus
);

HPI_API (hpi_err_t) HPI_Tuner_SetMode(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t nMode,
	uint32_t nValue
);

HPI_API (hpi_err_t) HPI_Tuner_GetMode(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t nMode,
	uint32_t *pnValue
);

HPI_API (hpi_err_t) HPI_Tuner_GetRDS(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	char *pRdsData
);

HPI_API (hpi_err_t) HPI_Tuner_QueryDeemphasis(
	const hpi_hsubsys_t *phSubSys,
	const hpi_handle_t hTuner,
	const uint32_t dwIndex,
	const uint16_t band,
	uint32_t *pdwDeemphasis
);

HPI_API (hpi_err_t) HPI_Tuner_SetDeemphasis(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwDeemphasis
);
HPI_API (hpi_err_t) HPI_Tuner_GetDeemphasis(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *pdwDeemphasis
);

HPI_API (hpi_err_t) HPI_Tuner_QueryProgram(
	const hpi_hsubsys_t *phSubSys,
	const hpi_handle_t hTuner,
	uint32_t *pbitmapProgram
);

HPI_API (hpi_err_t) HPI_Tuner_SetProgram(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwProgram
);

HPI_API (hpi_err_t) HPI_Tuner_GetProgram(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *pdwProgram
);

HPI_API (hpi_err_t) HPI_Tuner_GetHdRadioDspVersion(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	char *pszDspVersion,
	const uint32_t dwStringSize
);

HPI_API (hpi_err_t) HPI_Tuner_GetHdRadioSdkVersion(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	char *pszSdkVersion,
	const uint32_t dwStringSize
);

HPI_API (hpi_err_t) HPI_Tuner_GetFirmwareVersion(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	char *pszFWVersion,
	const uint32_t dwStringSize
);

HPI_API (hpi_err_t) HPI_Tuner_GetHdRadioSignalQuality(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *pdwQuality
);

HPI_API (hpi_err_t) HPI_Tuner_GetHdRadioSignalBlend(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *pdwBlend
);

HPI_API (hpi_err_t) HPI_Tuner_SetHdRadioSignalBlend(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	const uint32_t dwBlend
);

HPI_API (hpi_err_t) HPI_Tuner_GetDabAudioServiceCount(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *pdwIndex,
	uint32_t *pdwCount
);

HPI_API (hpi_err_t) HPI_Tuner_SetDabAudioService(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	const uint32_t dwIndex
);

HPI_API (hpi_err_t) HPI_Tuner_GetDabAudioServiceName(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	char *pszAudioServiceName,
	const uint32_t nSize,
	const uint32_t nIndex
);

HPI_API (hpi_err_t) HPI_Tuner_GetDabMultiplexName(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	char *pszMultiplexName,
	const uint32_t nSize
);

HPI_API (hpi_err_t) HPI_Tuner_GetDabAudioInfo(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	char *pszAudioInfo,
	const uint32_t nSize
);

HPI_API (hpi_err_t) HPI_Tuner_GetDabServiceInfo(
	hpi_handle_t hControl,
	unsigned int index,
	struct hpi_dab_service_info *service_info
);

HPI_API (hpi_err_t) HPI_Tuner_GetDabComponentInfo(
	hpi_handle_t hControl,
	uint32_t service_id,
	uint16_t component_id,
	struct hpi_dab_component_info *component_info
);

HPI_API (hpi_err_t) HPI_Tuner_SetDabService(
	hpi_handle_t hControl,
	uint32_t service_id,
	uint16_t component_id,
	int stop
);

HPI_API (hpi_err_t) HPI_Tuner_GetDabService(
	hpi_handle_t hControl,
	uint32_t *service_id,
	uint16_t *component_id
);

HPI_API (hpi_err_t) HPI_Tuner_GetDabDataPacket(
	hpi_handle_t hControl,
	char *buffer,
	const size_t buf_len,
	size_t *data_len,
	int *next_poll_interval_ms
);

HPI_API (hpi_err_t) HPI_Tuner_SendMsg(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint8_t *pBuf,
	uint16_t msgSize,
	uint16_t resSize
);

HPI_API (hpi_err_t) HPI_Tuner_SetBackgroundProc(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint8_t enabled
);

/***************/
/* PAD control */
/***************/

HPI_API (hpi_err_t) HPI_PAD_GetChannelName(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	char *pszString,
	const uint32_t dwStringLength
);

HPI_API (hpi_err_t) HPI_PAD_GetArtist(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	char *pszString,
	const uint32_t dwStringLength
);

HPI_API (hpi_err_t) HPI_PAD_GetTitle(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	char *pszString,
	const uint32_t dwStringLength
);

HPI_API (hpi_err_t) HPI_PAD_GetComment(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	char *pszString,
	const uint32_t dwStringLength
);

HPI_API (hpi_err_t) HPI_PAD_GetProgramType(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *pdwPTY
);

HPI_API (hpi_err_t) HPI_PAD_GetRdsPI(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *pdwPI
);

HPI_API (hpi_err_t) HPI_PAD_GetProgramTypeString(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	const uint32_t dwDataType,
	const uint32_t nPTY,
	char *pszString,
	const uint32_t dwStringLength
);
#endif

/****************************/
/* AES/EBU Receiver control */
/****************************/
#ifndef HPI_OS_LINUX_KERNEL
HPI_API (hpi_err_t) HPI_AESEBU_Receiver_QueryFormat(
	const hpi_hsubsys_t *phSubSys,
	const hpi_handle_t hAesRx,
	const uint32_t dwIndex,
	uint16_t *pwFormat
);
#endif
HPI_API (hpi_err_t) HPI_AESEBU_Receiver_SetFormat(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wSource
);

HPI_API (hpi_err_t) HPI_AESEBU_Receiver_GetFormat(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t *pwSource
);

#ifdef HPI_BUILD_INCLUDE_DEPRECATED
HPI_DEPRECATED HPI_API (hpi_err_t) HPI_AESEBU_Receiver_SetSource(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wSource
);

HPI_DEPRECATED HPI_API (hpi_err_t) HPI_AESEBU_Receiver_GetSource(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t
	*pwSource
);
#endif

#ifndef HPI_OS_LINUX_KERNEL
HPI_API (hpi_err_t) HPI_AESEBU_Receiver_GetSampleRate(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *pdwSampleRate
);

HPI_API (hpi_err_t) HPI_AESEBU_Receiver_GetUserData(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wIndex,
	uint16_t *pwData
);

HPI_API (hpi_err_t) HPI_AESEBU_Receiver_GetChannelStatus(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wIndex,
	uint16_t *pwData
);
#endif

HPI_API (hpi_err_t) HPI_AESEBU_Receiver_GetErrorStatus(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t *pwErrorData
);

/*******************************/
/* AES/EBU Transmitter control */
/*******************************/
#ifndef HPI_OS_LINUX_KERNEL
HPI_API (hpi_err_t) HPI_AESEBU_Transmitter_SetSampleRate(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwSampleRate
);

HPI_API (hpi_err_t) HPI_AESEBU_Transmitter_SetUserData(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wIndex,
	uint16_t wData
);

HPI_API (hpi_err_t) HPI_AESEBU_Transmitter_SetChannelStatus(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wIndex,
	uint16_t wData
);

HPI_API (hpi_err_t) HPI_AESEBU_Transmitter_GetChannelStatus(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wIndex,
	uint16_t *pwData
);

HPI_API (hpi_err_t) HPI_AESEBU_Transmitter_QueryFormat(
	const hpi_hsubsys_t *phSubSys,
	const hpi_handle_t hAesTx,
	const uint32_t dwIndex,
	uint16_t *pwFormat
);
#endif
HPI_API (hpi_err_t) HPI_AESEBU_Transmitter_SetFormat(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wOutputFormat
);

HPI_API (hpi_err_t) HPI_AESEBU_Transmitter_GetFormat(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t *pwOutputFormat
);

/***********************/
/* Multiplexer control */
/***********************/
HPI_API (hpi_err_t) HPI_Multiplexer_SetSource(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wSourceNodeType,
	uint16_t wSourceNodeIndex
);

HPI_API (hpi_err_t) HPI_Multiplexer_GetSource(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t *wSourceNodeType,
	uint16_t *wSourceNodeIndex
);

HPI_API (hpi_err_t) HPI_Multiplexer_QuerySource(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t nIndex,
	uint16_t *wSourceNodeType,
	uint16_t *wSourceNodeIndex
);

#ifndef HPI_OS_LINUX_KERNEL
/***************/
/* Vox control */
/***************/
HPI_API (hpi_err_t) HPI_VoxSetThreshold(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	short anGain0_01dB
);

HPI_API (hpi_err_t) HPI_VoxGetThreshold(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	short *anGain0_01dB
);
#endif
#ifndef HPI_OS_LINUX_KERNEL
/*********************/
/* Bitstream control */
/*********************/
HPI_API (hpi_err_t) HPI_Bitstream_SetClockEdge(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wEdgeType
);

HPI_API (hpi_err_t) HPI_Bitstream_SetDataPolarity(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wPolarity
);

HPI_API (hpi_err_t) HPI_Bitstream_GetActivity(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t *pwClkActivity,
	uint16_t *pwDataActivity
);
#endif

/***********************/
/* SampleClock control */
/***********************/

HPI_API (hpi_err_t) HPI_SampleClock_QuerySource(
	const hpi_hsubsys_t *phSubSys,
	const hpi_handle_t hClock,
	const uint32_t dwIndex,
	uint16_t *pwSource
);

HPI_API (hpi_err_t) HPI_SampleClock_SetSource(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wSource
);

HPI_API (hpi_err_t) HPI_SampleClock_GetSource(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t *pwSource
);

HPI_API (hpi_err_t) HPI_SampleClock_QuerySourceIndex(
	const hpi_hsubsys_t *phSubSys,
	const hpi_handle_t hClock,
	const uint32_t dwIndex,
	const uint32_t dwSource,
	uint16_t *pwSourceIndex
);

HPI_API (hpi_err_t) HPI_SampleClock_SetSourceIndex(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wSourceIndex
);

HPI_API (hpi_err_t) HPI_SampleClock_GetSourceIndex(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t *pwSourceIndex
);

#ifdef HPI_BUILD_INCLUDE_DEPRECATED
HPI_DEPRECATED HPI_API (hpi_err_t) HPI_SampleClock_SetSampleRate(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwSampleRate
);
#endif

HPI_API (hpi_err_t) HPI_SampleClock_GetSampleRate(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *pdwSampleRate
);

HPI_API (hpi_err_t) HPI_SampleClock_QueryLocalRate(
	const hpi_hsubsys_t *phSubSys,
	const hpi_handle_t hClock,
	const uint32_t dwIndex,
	uint32_t *pdwSource
);

HPI_API (hpi_err_t) HPI_SampleClock_SetLocalRate(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwSampleRateHz
);

HPI_API (hpi_err_t) HPI_SampleClock_SetLocalRateEx(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwSampleRateHz,
	uint32_t dwSampleRatePPB
);

HPI_API (hpi_err_t) HPI_SampleClock_GetLocalRate(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *pdwSampleRate
);

#ifndef HPI_OS_LINUX_KERNEL
HPI_API (hpi_err_t) HPI_SampleClock_SetAuto(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwEnable
);

HPI_API (hpi_err_t) HPI_SampleClock_GetAuto(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *pdwEnable
);

HPI_API (hpi_err_t) HPI_SampleClock_SetLocalRateLock(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwLock
);

HPI_API (hpi_err_t) HPI_SampleClock_GetLocalRateLock(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *pdwLock
);
#endif
#ifndef HPI_OS_LINUX_KERNEL
/***********************/
/* Microphone control */
/***********************/
HPI_API (hpi_err_t) HPI_Microphone_SetPhantomPower(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wOnOff
);

HPI_API (hpi_err_t) HPI_Microphone_GetPhantomPower(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t *pwOnOff
);
#endif
#ifndef HPI_OS_LINUX_KERNEL
/********************************/
/* Parametric Equalizer control */
/********************************/
HPI_API (hpi_err_t) HPI_ParametricEQ_GetInfo(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t *pwNumberOfBands,
	uint16_t *pwEnabled
);

HPI_API (hpi_err_t) HPI_ParametricEQ_SetState(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wOnOff
);

HPI_API (hpi_err_t) HPI_ParametricEQ_SetBand(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wIndex,
	uint16_t nType,
	uint32_t dwFrequencyHz,
	short nQ100,
	short nGain0_01dB
);

HPI_API (hpi_err_t) HPI_ParametricEQ_GetBand(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wIndex,
	uint16_t *pnType,
	uint32_t *pdwFrequencyHz,
	short *pnQ100,
	short *pnGain0_01dB
);

HPI_API (hpi_err_t) HPI_ParametricEQ_GetCoeffs(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wIndex,
	short coeffs[5]
);
#endif
#ifndef HPI_OS_LINUX_KERNEL
/*******************************/
/* Compressor Expander control */
/*******************************/
#ifdef HPI_BUILD_INCLUDE_DEPRECATED
HPI_DEPRECATED HPI_API (hpi_err_t) HPI_Compander_Set(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t wAttack,
	uint16_t wDecay,
	short wRatio100,
	short nThreshold0_01dB,
	short nMakeupGain0_01dB
);

HPI_DEPRECATED HPI_API (hpi_err_t) HPI_Compander_Get(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint16_t *pwAttack,
	uint16_t *pwDecay,
	short *pwRatio100,
	short *pnThreshold0_01dB,
	short *pnMakeupGain0_01dB
);
#endif
HPI_API (hpi_err_t) HPI_Compander_SetEnable(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwOn
);

HPI_API (hpi_err_t) HPI_Compander_GetEnable(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *pdwOn
);

HPI_API (hpi_err_t) HPI_Compander_SetMakeupGain(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	short nMakeupGain0_01dB
);

HPI_API (hpi_err_t) HPI_Compander_GetMakeupGain(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	short *pnMakeupGain0_01dB
);

HPI_API (hpi_err_t) HPI_Compander_SetAttackTimeConstant(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwIndex,
	uint32_t wAttack
);

HPI_API (hpi_err_t) HPI_Compander_GetAttackTimeConstant(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwIndex,
	uint32_t *pwAttack
);

HPI_API (hpi_err_t) HPI_Compander_SetDecayTimeConstant(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwIndex,
	uint32_t wDecay
);

HPI_API (hpi_err_t) HPI_Compander_GetDecayTimeConstant(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwIndex,
	uint32_t *pwDecay
);

HPI_API (hpi_err_t) HPI_Compander_SetThreshold(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwIndex,
	short nThreshold0_01dB
);

HPI_API (hpi_err_t) HPI_Compander_GetThreshold(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwIndex,
	short *pnThreshold0_01dB
);

HPI_API (hpi_err_t) HPI_Compander_SetRatio(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwIndex,
	uint32_t wRatio100
);

HPI_API (hpi_err_t) HPI_Compander_GetRatio(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwIndex,
	uint32_t *pwRatio100
);
#endif
#ifndef HPI_OS_LINUX_KERNEL
/********************/
/* Cobranet control */
/********************/
HPI_API (hpi_err_t) HPI_Cobranet_HmiWrite(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwHmiAddress,
	uint32_t dwByteCount,
	uint8_t *pbData
);

HPI_API (hpi_err_t) HPI_Cobranet_HmiRead(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwHmiAddress,
	uint32_t dwMaxByteCount,
	uint32_t *pdwByteCount,
	uint8_t *pbData
);

HPI_API (hpi_err_t) HPI_Cobranet_HmiGetStatus(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *pdwStatus,
	uint32_t *pdwReadableSize,
	uint32_t *pdwWriteableSize
);

HPI_API (hpi_err_t) HPI_Cobranet_GetIPaddress(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *pdwIPaddress
);

HPI_API (hpi_err_t) HPI_Cobranet_SetIPaddress(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwIPaddress
);

HPI_API (hpi_err_t) HPI_Cobranet_GetStaticIPaddress(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *pdwIPaddress
);

HPI_API (hpi_err_t) HPI_Cobranet_SetStaticIPaddress(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t dwIPaddress
);

HPI_API (hpi_err_t) HPI_Cobranet_GetMACaddress(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hControl,
	uint32_t *pdwMAC_MSBs,
	uint32_t *pdwMAC_LSBs
);
#endif
#ifndef HPI_OS_LINUX_KERNEL
/*************************/
/* Tone Detector control */
/*************************/
HPI_API (hpi_err_t) HPI_ToneDetector_GetState(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	uint32_t *state
);

HPI_API (hpi_err_t) HPI_ToneDetector_SetEnable(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	uint32_t enable
);

HPI_API (hpi_err_t) HPI_ToneDetector_GetEnable(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	uint32_t *enable
);

HPI_API (hpi_err_t) HPI_ToneDetector_SetEventEnable(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	uint32_t EventEnable
);

HPI_API (hpi_err_t) HPI_ToneDetector_GetEventEnable(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	uint32_t *EventEnable
);

HPI_API (hpi_err_t) HPI_ToneDetector_SetThreshold(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	int threshold
);

HPI_API (hpi_err_t) HPI_ToneDetector_GetThreshold(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	int *threshold
);

HPI_API (hpi_err_t) HPI_ToneDetector_GetFrequency(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	uint32_t nIndex,
	uint32_t *dwFrequency
);
#endif
#ifndef HPI_OS_LINUX_KERNEL
/****************************/
/* Silence Detector control */
/****************************/
HPI_API (hpi_err_t) HPI_SilenceDetector_GetState(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	uint32_t *state
);

HPI_API (hpi_err_t) HPI_SilenceDetector_SetEnable(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	uint32_t enable
);

HPI_API (hpi_err_t) HPI_SilenceDetector_GetEnable(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	uint32_t *enable
);

HPI_API (hpi_err_t) HPI_SilenceDetector_SetEventEnable(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	uint32_t EventEnable
);

HPI_API (hpi_err_t) HPI_SilenceDetector_GetEventEnable(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	uint32_t *EventEnable
);

HPI_API (hpi_err_t) HPI_SilenceDetector_SetDelay(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	uint32_t delay
);

HPI_API (hpi_err_t) HPI_SilenceDetector_GetDelay(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	uint32_t *delay
);

HPI_API (hpi_err_t) HPI_SilenceDetector_SetThreshold(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	int threshold
);

HPI_API (hpi_err_t) HPI_SilenceDetector_GetThreshold(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	int *threshold
);
#endif
#ifndef HPI_OS_LINUX_KERNEL
/*********************/
/* Universal control */
/*********************/

HPI_API (hpi_err_t) HPI_Universal_Info(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	struct hpi_entity **info
);

HPI_API (hpi_err_t) HPI_Universal_Get(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	struct hpi_entity **value
);

HPI_API (hpi_err_t) HPI_Universal_Set(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hC,
	struct hpi_entity *value
);

/*****************/
/* Clock control */
/*****************/
HPI_API (hpi_err_t) HPI_ClockOpen(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	hpi_handle_t * phDspClock
);

HPI_API (hpi_err_t) HPI_ClockSetTime(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hClock,
	uint16_t wHour,
	uint16_t wMinute,
	uint16_t wSecond,
	uint16_t wMilliSecond
);

HPI_API (hpi_err_t) HPI_ClockGetTime(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hClock,
	uint16_t *pwHour,
	uint16_t *pwMinute,
	uint16_t *pwSecond,
	uint16_t *pwMilliSecond
);

/***********/
/* Profile */
/***********/
HPI_API (hpi_err_t) HPI_ProfileOpenAll(
	const hpi_hsubsys_t *phSubSys,
	uint16_t wAdapterIndex,
	uint16_t wProfileIndex,
	hpi_handle_t * phProfile,
	uint16_t *pwMaxProfiles
);

HPI_API (hpi_err_t) HPI_ProfileGet(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hProfile,
	uint16_t wBinIndex,
	uint16_t *pwProfileInterval,
	uint32_t *pdwTotalTickCount,
	uint32_t *pdwCallCount,
	uint32_t *pdwMaxTickCount,
	uint32_t *pdwTicksPerMillisecond
);

#ifdef HPI_BUILD_INCLUDE_DEPRECATED
HPI_DEPRECATED HPI_API (hpi_err_t) HPI_ProfileGetIdleCount(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hProfile,
	uint16_t wIndex,
	uint32_t *pdwCycleCount,
	uint32_t *pdwCount
);
#endif

HPI_API (hpi_err_t) HPI_ProfileStartAll(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hProfile
);

HPI_API (hpi_err_t) HPI_ProfileStopAll(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hProfile
);

HPI_API (hpi_err_t) HPI_ProfileGetName(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hProfile,
	uint16_t wIndex,
	char *szProfileName,
	uint16_t nProfileNameLength
);

HPI_API (hpi_err_t) HPI_ProfileGetUtilization(
	const hpi_hsubsys_t *phSubSys,
	hpi_handle_t hProfile,
	uint32_t *pdwUtilization
);



#endif
/*********************/
/* Utility functions */
/*********************/

#ifndef HPI_OS_LINUX_KERNEL
HPI_API_VOID (void) HPI_GetErrorText(
	hpi_err_t wError,
	char *pszErrorText
);
#endif

HPI_API (hpi_err_t) HPI_FormatCreate(
	struct hpi_format *pFormat,
	uint16_t wChannels,
	uint16_t wFormat,
	uint32_t dwSampleRate,
	uint32_t dwBitRate,
	uint32_t dwAttributes
);

#ifdef HPI_BUILD_INCLUDE_DEPRECATED
HPI_DEPRECATED HPI_API (hpi_err_t) HPI_DataCreate(
	struct hpi_data * pData,
	struct hpi_format *pFormat,
	uint8_t *pbData,
	uint32_t dwDataSize
);
#endif

#if defined(HPI_OS_WIN32_USER) || defined(INCLUDE_WINDOWS_ON_LINUX) || defined(HPI_OS_OSX)
/* Until it's verified, this function is for Windows OSs only */

#if defined HPI_OS_LINUX || defined HPI_OS_OSX
#include <asimmdef.h>
#else
#include <windows.h>
#include <mmreg.h>
#endif

HPI_API (hpi_err_t) HPI_WaveFormatToHpiFormat(
	const PWAVEFORMATEX lpFormatEx,
	struct hpi_format *pHpiFormat
);

HPI_API (hpi_err_t) HPI_HpiFormatToWaveFormat(
	const struct hpi_format *pHpiFormat,
	PWAVEFORMATEX lpFormatEx
);

#endif				/* defined(HPI_OS_WIN32_USER) */

#ifdef __cplusplus
/* *INDENT-OFF* */
}
/* *INDENT-ON* */
#endif
#endif	/*_HPI_H_ */
