/******************************************************************************
 $Header: /Repository/apps/firmdown/firmdown.c,v 1.30 2007/09/07 09:39:18 as-ewb Exp $

  Firmware downloader for ASI2400

 (C) Copyright AudioScience Inc. 2005
*******************************************************************************/
#include <string.h>
#include <stdlib.h>

#include "hpi.h"
#include "hpinet.h"
#include "hpifirmware.h"
#include <getopt.h>

#define MAX_FILES 3
#define MAC_STRING_SIZE 12

#ifndef MAX_PATH
#define MAX_PATH 260
#endif

static HPI_HSUBSYS *phSubSys=0 ;

struct fw_info {
	enum eHpiFirmwareId type;
	char path[MAX_PATH];
	char name[12];
};

static struct fw_info fw[MAX_FILES]= {
	{HPI_FW_UPDATE,     "dsp2400.bin", "Update"},
	{HPI_FW_FACTORY,    "dsp2400.bin", "Factory"},
	{HPI_FW_BOOTLOADER, "boot2400.bin","Bootloader"}
};

/* Option variables */
int image_sel=0;
int num_files=0;
int shown_help=0;
char szFile[MAX_FILES][80]={"","",""};
int adapter_specified=0;
HW16 wAdapterIndex=0;

static struct option long_options[] =
    {
	{"adapter"    , required_argument, 0, 'a'},
        {"factory"      , no_argument,0,'f'},
        {"update"       , no_argument,0,'u'},
        {"bootloader"   , no_argument,0,'b'},
        {"help"         , no_argument,0,'h'},
        {0, 0, 0, 0}
    };

const char* short_options="a:fubh?";

const char* option_help[]=
    {
	"Adapter index",
	"Update factory image (danger!)",
	"Update update image",
	"Update bootloader image (danger!)",
        "Show this text."
    };

void help(char * progname)
{
    int i=0;
    printf("\nUsage - %s [options]"
		" [update image [factory image [bootloader image]]]\n", progname);
    while (long_options[i].name != 0) {
        printf("--%s -%c %s\n",
               long_options[i].name,
               (char)(long_options[i].val),
               option_help[i]);
        i++;
    }
	printf("\n");
}

void parse_options(int argc, char *argv[])
{
    int c;
    /*********** Parse the command line options ***************/
    while (1) {
      //int this_option_optind = optind ? optind : 1;
        int option_index = 0;

        c = getopt_long (argc, argv, short_options,
                         long_options, &option_index);
        if (c == -1)
            break;

        switch (c) {
        case 0:
            printf ("option %s", long_options[option_index].name);
            if (optarg)
                printf (" with arg %s", optarg);
            printf ("\n");
            break;
	case 'a':
		wAdapterIndex=atoi(optarg);
		adapter_specified=1;
		break;

	case 'u':
		image_sel |= 1;
		break;
	case 'f':
		image_sel |= 2;
		break;
	case 'b':
		image_sel |= 4;
		break;
        case '?':
        case 'h':
			shown_help=1;
            help(argv[0]);
			exit(0);
            break;

        default:
            printf ("?? getopt returned character code 0%o ??\n", c);
        }
    }

    if (optind < argc) {
        // printf ("non-option ARGV-elements: ");
        while ((optind < argc) && (num_files < MAX_FILES))  {
            strncpy( fw[num_files].path, argv[optind++], MAX_PATH-1 );
            printf("%s image is %s\n", fw[num_files].name,fw[num_files].path);
            num_files++;
        }
    }

}

/*=================================================================*/
/**

*/
void HPI_PrintError(HW16 wError, HW16 wSpecificError)
{
	char errbuf[256];

	if (wError==0) {

	} else {
		HPI_GetErrorText( wError, errbuf);
				printf(" Error%s, SE=%d\n",errbuf,wSpecificError);	}
}

/*=================================================================*/
/**
Progress callback
*/

void pc(char * state, u_int percent)
{
	printf("%s %d\n",state, percent);
}
/*=================================================================*/
/**

*/
int main(int argc, char* argv[])
{

	int i=0;
	short error=0;
    enum eHpiFirmwareId sel;
	int wNumAdapters;

	printf("FIRMDOWN $Revision: 1.30 $ (c) AudioScience, Inc. 2005\n\n");
	parse_options(argc,argv);

	if (!image_sel && !shown_help)
		printf("To see full option list, use -h option\n\n");

	phSubSys=HPI_SubSysCreate();
	if (!phSubSys) {
		printf("SubSysOpen Failed\n");
		return 1;
	}

	if (!adapter_specified) {
		printf("Choose one of these adapters using -a<index>:\n");

		error = HPI_SubSysGetNumAdapters(
			phSubSys,
			&wNumAdapters );

		if (!error)
		{
			for (i=0;i<wNumAdapters;i++)
			{
				HW32 dwAdapterIndex;
				HW16 wAdapterType;

				error = HPI_SubSysGetAdapter(
						phSubSys, i,
						&dwAdapterIndex, &wAdapterType);
					if (!error)
						printf("\tASI%x index %d\n",wAdapterType,dwAdapterIndex);
			}
		}
		exit (0);
	}

	//========= adapter open ==========
	error=HPI_AdapterOpen(phSubSys,wAdapterIndex);
	HPI_PrintError(error , 0);
	if (error)
		exit(1);

	{ //========= adapter get info ==========
		HW16 wNumOutStreams;
		HW16 wNumInStreams;
		HW16 wVersion;
		HW32 dwSerialNumber;
		HW16 wAdapterType;

		error= HPI_AdapterGetInfo(phSubSys,wAdapterIndex,
			&wNumOutStreams,&wNumInStreams,
			&wVersion,&dwSerialNumber,&wAdapterType);
		HPI_PrintError(error , 0);

		printf("Adapter ID=%4X Index=%d S/N=%d\nHw Version %c%d DSP code version %03d\n",
			   wAdapterType, wAdapterIndex,
			   dwSerialNumber,
			   ((wVersion>>3)&0xf)+'A',    // Hw version major
			   wVersion&0x7,               // Hw version minor
			   ((wVersion>>13)*100)+((wVersion>>7)&0x3f)  // DSP code version
			  );
	}

	for (sel=HPI_FW_UPDATE; sel <= HPI_FW_BOOTLOADER; sel++) {
                unsigned int version, checksum, length;

		HPI_AdapterFirmwareGetInfo(wAdapterIndex, sel,
                  &version, &checksum, &length);

			printf("%10s image, version %3d, checksum %08x, length %d\n",
				fw[sel-1].name,
				version,
				checksum,
				length);
	}


	//========= program selected images to flash ==========
	for (i=0; i<3; i++) {
		if (image_sel & (1 << i)) {
			error = HPI_AdapterFirmwareDownload(wAdapterIndex, fw[i].type, fw[i].path, pc);
				HPI_PrintError(error , 0);
		}
	}

	if (image_sel)
		HPI_AdapterRestart(wAdapterIndex);

	if (error) {
		exit(1);
	}
	exit(0);
}

// $Log: firmdown.c,v $
// Revision 1.30  2007/09/07 09:39:18  as-ewb
// move engine to hpifirmware.[ch]
//
// Revision 1.29  2007/06/20 23:49:36  as-ewb
// exit if error on adapter open
//
// Revision 1.28  2007/06/19 21:47:17  as-ewb
// Add UDP build option
//
// Revision 1.27  2007/05/18 21:13:22  as-age
// Packet debug code correction.
//
// Revision 1.26  2007/05/10 00:37:03  as-age
// Add packet data dumping in verbose (-v 1) mode.
//
// Revision 1.25  2006/11/28 07:53:02  as-ewb
// add MAX_PATH if not defined
//
// Revision 1.24  2006/11/28 04:49:59  as-ewb
// Allow selection of adapter by MAC when more than one adapter present on network.  -a option removed because adapter index potentially can change on each run depending on which adapter answers the broadcast query first.
// Print the MAC addresses of found adapters.
//
// Revision 1.23  2006/11/28 00:38:23  as-ewb
// Add -h option
//
// Revision 1.22  2006/11/27 23:26:38  as-ewb
// Minor change to help text.
// Add -m and -a options for choosing which adapter to program.
// Note selection of what image to program is by a combination of the options -u, -f or -b
//
// Revision 1.21  2006/11/27 22:24:52  as-ewb
// add getopt style options
//
// Revision 1.20  2006/11/16 10:26:24  as-ewb
// Get firmware files from local directory (no path).
// Remove all unused test code.
//
// Revision 1.19  2006/03/14 15:11:06  as-tfe
// Add new parameter to HpiDspCode_Open().
//
// Revision 1.18  2006/02/24 17:08:22  as-age
// Fix missing parameter.
//
// Revision 1.17  2006/02/21 20:26:50  as-age
// Fix for updated tHPIControlCacheValue structure.
//
// Revision 1.16  2006/01/13 01:30:19  as-ewb
// tweaks to make compatible with old and new hpi api.
//
// Revision 1.15  2005/12/14 00:26:18  as-sgt
// SGT - only program if 1 adapter is found on the network (until we get IDs)
//
// Revision 1.14  2005/11/07 23:36:01  as-ewb
// ewb send special "reset" adapter close message when finished if any images have been programmed
//
// Revision 1.13  2005/08/30 21:26:12  as-ewb
// ewb exit if subsysopen fails
//
// Revision 1.12  2005/08/19 15:54:59  as-age
// AGE - add code to read changed control array from a ASI2416. Seems to work.
//
// Revision 1.11  2005/08/10 07:30:57  as-ewb
// ewb select adapter from commandline by index or mac address
//
// Revision 1.10  2005/08/05 20:15:48  as-age
// AGE - add code to dump the returned control array - seems to work !
//
// Revision 1.9  2005/08/05 01:10:57  as-ewb
// ewb use a table for firmware details.
//
// Revision 1.8  2005/08/04 13:45:55  as-age
// AGE - tweaks for modifications to hpinet.h structure defns for DSP compiler support.
// No functional changes.
//
// Revision 1.7  2005/08/04 03:29:13  as-ewb
// ewb update to match hpinet.c.
// Add test code for get control array function.
//
// Revision 1.6  2005/08/03 22:12:39  as-ewb
// EWB tweaks for compiler warnings
//
// Revision 1.5  2005/08/03 08:27:04  as-ewb
// EWB uncomment more code needed for firmware update.
// Uncomment mixer dump call.
//
// Revision 1.4  2005/08/03 08:22:37  as-ewb
// ewb move some stuff to hpinet.c.  add some code to test hpi messages.
//
// Revision 1.3  2005/08/01 09:29:24  as-ewb
// EWB add code to get local MAC address.   Required use of pcap-int.h to
// access "opaque" pcap_t data structure.  Would like a cleaner way to do this.
// Changed default action to do nothing other than display adapter info.
//
// Revision 1.2  2005/07/28 05:12:25  as-ewb
// ewb add commandline parameter to specify which image to download.
// Default or 1 is for update image, 2=factory, 4=bootloader.  Add them to download more than one.
//
// Revision 1.1  2005/07/27 05:13:23  as-ewb
// EWB prototype firmware downloader  for ASI2400.
// Requres winpcap library.
//

