/******************************************************************************
Copyright (C) 1997-2003 AudioScience, Inc. All rights reserved.

This software is provided 'as-is', without any express or implied warranty.
In no event will AudioScience Inc. be held liable for any damages arising
from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
   claim that you wrote the original software. If you use this software
   in a product, an acknowledgment in the product documentation would be
   appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.
3. This copyright notice and list of conditions may not be altered or removed 
   from any source distribution.

AudioScience, Inc. <support@audioscience.com>

( This license is GPL compatible see http://www.gnu.org/licenses/license-list.html#GPLCompatibleLicenses )

  Control tuners on ASI87xx cards

  Usage: asihpitune --help

******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#ifdef HPI_OS_LINUX
#include <unistd.h>
#endif
#include <getopt.h>
#include <hpi.h>
#include <hpirds.h>
#include <hpidebug.h>

#define verbose_printf if (verbose) printf

typedef struct
{
    HW16            wControlType;           // HPI_CONTROL_METER, _VOLUME etc
    HW16            wSrcNodeType;
    HW16            wSrcNodeIndex;
    HW16            wDstNodeType;
    HW16            wDstNodeIndex;
}
hpi_control_t;

HPI_HSUBSYS *hSubSys;   // handle to audio subsystem

static char *hpi_control_strings[] = HPI_CONTROL_TYPE_STRINGS;
static char *hpi_src_strings[] = HPI_SOURCENODE_STRINGS;
static char *hpi_dst_strings[] = HPI_DESTNODE_STRINGS;

#define nanosec 1000000

// local protos
void    HandleError( HW16 wHE );
void    HandleErrorComment( char * comment, HW16 wHE );
int getch(void);
static void asi_sleep(unsigned int milliseconds);

/* NOTE: indices must match HPI defines for each band */
char * band_names[HPI_TUNER_BAND_LAST+1] = {
	"invalid",
	"AM",
	"FM" ,
	"NTSC_M",
	"FM2" ,
	"Aux" ,
	"PAL_BG" ,
	"PAL_I" ,
	"PAL_DK" ,
	"Secam_L" ,
};

/* Option variables */
unsigned int dump_mixer = 0;
HW16		wAdapterIndex=0;
HW16	 nTunerIndex=0;
int nReadLevel=1;
HW16 nBand= 0;
HW32 nTunerFrequency=0;
short wMode=-1;
int query_tuner_caps=0;
int query_tuner_settings=0;
int verbose=0;
int rds=0;
short nMux=-1;


static struct option long_options[] =
    {
        {"adapter"    , required_argument, 0,'a'
        },
        {"band"       , required_argument, 0,'b'},
        {"query-caps" , no_argument      , 0,'c'},
        {"dump-mixer" , no_argument      , 0,'d'},
        {"frequency"  , required_argument, 0,'f'},
        {"mode"       , required_argument, 0,'m'},
        {"rds"        , no_argument      , 0,'r'},
        {"settings"  , no_argument       , 0,'s'},
        {"tuner"      , required_argument, 0,'t'},
        {"verbose"    , no_argument      , 0,'v'},
        {"mux"        , required_argument, 0,'x'},
        {"help"       , no_argument      , 0,'h'},
        {0, 0, 0, 0}
    };

const char* short_options="a:b:cdf:m:rt:st:vx:h";

const char* option_help[]=
    {
        "<adapter number> to test, default is 0.",
        "<band> see below for valid options",
        "Query tuner capabilities.",
        "Dump list of mixer controls.",
        "<freq in kHz> tuner frequency",
        "<mode> Set tuner mode.",
        "Reads RDS data from the tuner.",
        "Read back tuner settings.",
        "<tuner index> to query or set, default is 0.",
        "Enable verbose output.",
        "<n> Set audio out mux to tuner",
        "Show this text."
    };
void help(void)
{
    int i=0;
    printf("\nUsage - asihpitune [options]\n");
    while (long_options[i].name != 0) {
        printf("--%s -%c %s\n",
               long_options[i].name,
               (char)(long_options[i].val),
               option_help[i]);
        i++;
    }
    printf("Valid values for -b option are ");
    for (i=1; i<=HPI_TUNER_BAND_LAST; i++)
        printf("%s, ",band_names[i]);
    printf("\n");
    exit(0);
}

void parse_options(int argc, char *argv[])
{
    int c;
    /*********** Parse the command line options ***************/
    while (1) {
      // int this_option_optind = optind ? optind : 1;
        int option_index = 0;

        c = getopt_long (argc, argv, short_options,
                         long_options, &option_index);
        if (c == -1)
            break;

        switch (c) {
        case 0:
            printf ("option %s", long_options[option_index].name);
            if (optarg)
                printf (" with arg %s", optarg);
            printf ("\n");
            break;
        case 'a':
            wAdapterIndex=atoi(optarg);
            break;
        case 'b':
        {   int i;
            for (i=1; i<=HPI_TUNER_BAND_LAST; i++)
                if (strcmp(band_names[i],optarg)==0)
                    nBand=i;
            if (!nBand) {
                printf("\n****** Invalid band: %s ******\n",optarg);
                help();
            }

            query_tuner_settings=1;
        }
            break;
        case 'c':
            query_tuner_caps=1;
            break;
        case 'd':
            dump_mixer=1;
            break;
        case 'f':
            nTunerFrequency=atoi(optarg);
            query_tuner_settings=1;
            break;
        case 'm':
            wMode=atoi(optarg);
            query_tuner_settings=1;
            break;
        case 'r':
            rds=1;
            break;
        case 's':
            query_tuner_settings=1;
            break;
        case 't':
            nTunerIndex=atoi(optarg);
            break;
        case 'v':
            verbose=1;
            break;
        case 'x':
            nMux=atoi(optarg);
            break;
        case '?':
        case 'h':
            help();
            break;

        default:
            printf ("?? getopt returned character code 0%o ??\n", c);
        }
    }

    if (optind < argc) {
        // printf ("non-option ARGV-elements: ");
        /*
              nNumFiles=0;
              while ((optind < argc) && (nNumFiles < MAX_FILES))  {
                  strcpy( szFile[nNumFiles], argv[optind++] );
                  printf("File %d is %s\n",nNumFiles,szFile[nNumFiles]);
                  nNumFiles++;
              }
        */
    }

}


void print_mixer_controls(HPI_HMIXER  hMixer)
{
    HW16 f;
    HPI_HCONTROL hControl;
    HW32 wHE;
    for (f=0; f < 256; f++) {
        hpi_control_t asihpi_control;

        wHE= HPI_MixerGetControlByIndex(
                 hSubSys,
                 hMixer,
                 f,
                 &asihpi_control.wSrcNodeType,
                 &asihpi_control.wSrcNodeIndex,
                 &asihpi_control.wDstNodeType,
                 &asihpi_control.wDstNodeIndex,
                 &asihpi_control.wControlType,           // HPI_CONTROL_METER, _VOLUME etc
                 &hControl     );
        if (wHE==HPI_ERROR_CONTROL_DISABLED)
		printf("DISABLED ");
	else if (wHE)
            break;

        printf ("HPI Control %d, %s:%s[%d]->%s[%d]\n",
                f,
                hpi_control_strings[asihpi_control.wControlType],
                hpi_src_strings[asihpi_control.wSrcNodeType-HPI_SOURCENODE_BASE],
                asihpi_control.wSrcNodeIndex,
                hpi_dst_strings[asihpi_control.wDstNodeType-HPI_DESTNODE_BASE],
                asihpi_control.wDstNodeIndex
               );
        if (asihpi_control.wControlType==HPI_CONTROL_MULTIPLEXER) {
            HW16 l;
            for (l=0; l<256; l++) {
                wHE=HPI_Multiplexer_QuerySource(hSubSys,  hControl, l,
                                                &asihpi_control.wSrcNodeType,&asihpi_control.wSrcNodeIndex);
                if (!wHE)
                    printf ("\tSource %d %s[%d]\n",
                            l,
                            hpi_src_strings[asihpi_control.wSrcNodeType-HPI_SOURCENODE_BASE],
                            asihpi_control.wSrcNodeIndex
                           );
                else
                    break;
            }
        }
    }
    printf("%d controls found\n\n",f);
}

/************************************** MAIN ***********************/
int main(int argc, char *argv[])
{
    HW16        wHE=0;  // HPI error
    HW32        dwVersion=0;
    HW16        wNumAdapters=0;
    HW16        awAdapterList[20];
    HW16        wListLength=20;
    HW16	wVersion;
    HW32	dwSerialNumber;
    HW16	wType;
    HW16	wNumOutStreams;
    HW16	wNumInStreams;
    int i;

    HPI_HMIXER  hMixer=0;

    parse_options(argc,argv);

    /* */
    verbose_printf("********************************************************************\n");
    verbose_printf("\n** Test HPI using Functions **\n");
    /*
       if (argc>1)
          sscanf(argv[1],"%d",&wAdapterIndex);
       else {
    printf("Usage:\n:Dump the mixer map of the adapter: test8700 <adapter index>\n");
    printf("Show tuner bands: test8700 <adapter index> <tuner index>\n");
    printf("Set a tuner: test8700 <adapter index> <tuner index> <band> <frequency in kHz>\n");

    exit (0);
       }
       */
    /* */
    // open subsystem and find adapters
    verbose_printf("********************************************************************\n");
    verbose_printf("HPI_SubSysCreate\n");
    hSubSys=HPI_SubSysCreate();
    if (hSubSys==NULL) {
        printf("hSubSys==NULL\n");
        exit(1);
    }

    wHE = HPI_SubSysGetVersion( hSubSys, &dwVersion );
    HandleError( wHE );
    verbose_printf("HPI_SubSysGetVersion=%x\n", dwVersion );

    wHE = HPI_SubSysFindAdapters(
              hSubSys,
              &wNumAdapters,
              awAdapterList,
              wListLength
          );
    HandleError( wHE );
    verbose_printf("HPI_SubSysFindAdapters NumberAdapters=%d ",wNumAdapters);
    for(i=0; i<4; i++)
        verbose_printf("%d=%X ", i, awAdapterList[i] );
    verbose_printf("\n");

    wHE = HPI_AdapterClose(
              hSubSys,
              wAdapterIndex
          );
    HandleError( wHE );
    verbose_printf("HPI_AdapterClose \n");

    // open 1st adapter
    wHE = HPI_AdapterOpen(
              hSubSys,
              wAdapterIndex
          );
    HandleError( wHE );
    verbose_printf("HPI_AdapterOpen \n");

    wHE = HPI_AdapterGetInfo(
              hSubSys,
              wAdapterIndex,
              &wNumOutStreams,
              &wNumInStreams,
              &wVersion,
              &dwSerialNumber,
              &wType
          );
    HandleError( wHE );
    verbose_printf("HPI_AdapterGetInfo\n");
    verbose_printf("Adapter ID=%4X Index=%d NumOutStreams=%d NumInStreams=%d S/N=%d\nHw Version %c%d DSP code version %03d\n",
                   wType, wAdapterIndex,
                   wNumOutStreams, wNumInStreams, dwSerialNumber,
                   ((wVersion>>3)&0xf)+'A',    // Hw version major
                   wVersion&0x7,               // Hw version minor
                   ((wVersion>>13)*100)+((wVersion>>7)&0x3f)  // DSP code version
                  );


    // open the mixer of this adapter
    wHE = HPI_MixerOpen(
              hSubSys,
              wAdapterIndex,
              &hMixer
          );
    verbose_printf("HPI_MixerOpen: handle=%X\n", hMixer);
    HandleError( wHE );
    printf("\n");

    /******** Dump mixer ******/
    if (dump_mixer)
        print_mixer_controls(hMixer);

    if ( ((wType & 0x8700) == 0x8700)||((wType & 0x8900) == 0x8900) ) {
		HPI_HCONTROL hControl=0,hMux=0;

        wHE = HPI_MixerGetControl( hSubSys, hMixer, HPI_SOURCENODE_TUNER,nTunerIndex, 0,0,
                                   HPI_CONTROL_TUNER, &hControl);

		wHE=HPI_MixerGetControl( hSubSys, hMixer,  0,0,HPI_DESTNODE_LINEOUT,0,
                                     HPI_CONTROL_MULTIPLEXER, &hMux);


		/******** Mode setting ******/
        if (wMode >= 0) {
            if (wHE==0) {
                HPI_Tuner_SetMode(hSubSys,hControl,HPI_TUNER_MODE_RSS,wMode);
                HandleErrorComment("SetMode", wHE );
            }

        }

        /******** Mux setting ******/
        if (hMux && (nMux >= 0 ) && ( nMux < 8)) {


            HandleErrorComment("Get mux ctrl", wHE );
            wHE = HPI_Multiplexer_SetSource( hSubSys, hMux, HPI_SOURCENODE_TUNER , nMux);
            HandleErrorComment("SetMuxSource", wHE );

        }

        /******** Band setting ******/
        if (nBand) {
            wHE = HPI_Tuner_SetBand( hSubSys, hControl, nBand);
            HandleErrorComment("SetBand", wHE );
        }

        /******** Frequency setting ******/
        if(nTunerFrequency) {
            wHE = HPI_Tuner_SetFrequency( hSubSys, hControl, nTunerFrequency);
            HandleErrorComment("SetFrequency", wHE );
        }
		asi_sleep(nanosec/1000000);

        /******** Query capabilities ******/
        if (query_tuner_caps) {
            HW32 dwIndex,dwSetting;

            wHE = HPI_MixerGetControl( hSubSys, hMixer, HPI_SOURCENODE_TUNER,nTunerIndex, 0,0,
                                       HPI_CONTROL_TUNER, &hControl);
            if (wHE==0) {
                printf("\nTuner %d supports\n",nTunerIndex);
                for (dwIndex=0; dwIndex<100; dwIndex++) {
                    wHE=HPI_ControlQuery(hSubSys, hControl,HPI_TUNER_BAND,
                                         dwIndex,0,&dwSetting);
                    if (wHE!=0)
                        break;
                    if ((dwSetting==0) || (dwSetting > HPI_TUNER_BAND_LAST)) {
                        printf("Unknown band type %d\n",dwSetting);
                        continue;
                    }

                    printf("%i %s : ",dwIndex,band_names[dwSetting]);
                    {
                        HW32 dwStart,dwEnd,dwStep;
                        wHE=HPI_ControlQuery(hSubSys, hControl,HPI_TUNER_FREQ,
                                             0,dwSetting,&dwStart);
                        wHE=HPI_ControlQuery(hSubSys, hControl,HPI_TUNER_FREQ,
                                             1,dwSetting,&dwEnd);
                        wHE=HPI_ControlQuery(hSubSys, hControl,HPI_TUNER_FREQ,
                                             2,dwSetting,&dwStep);
                        printf("from %ikHz to %ikHz step %ikHz\n",dwStart,dwEnd,dwStep);
                    }
                }
            }
        }
        /******** Query settings ******/
        if (query_tuner_settings) {
            HW16 nTunerBand=0;
            short int nRfLevel=0;
            HW32 nTunerFrequency=0;
            HW32 wMode=0;
			HW16 wMuxIndex=0;
			HW16 wStatus=0;
			HW16 wStatusMask=0;


			wHE = HPI_Tuner_GetBand( hSubSys, hControl, &nTunerBand);
			HandleErrorComment("HPI_Tuner_GetBand",wHE);
            wHE = HPI_Tuner_GetFrequency( hSubSys, hControl, &nTunerFrequency);
			HandleErrorComment("HPI_Tuner_GetFrequency",wHE);
            wHE=HPI_Tuner_GetRFLevel(hSubSys, hControl, &nRfLevel);
			HandleErrorComment("HPI_Tuner_GetRFLevel",wHE);
            wHE=HPI_Tuner_GetMode(hSubSys,hControl,HPI_TUNER_MODE_RSS,&wMode);
			if(wHE==HPI_ERROR_INVALID_CONTROL_ATTRIBUTE)
			{
				verbose_printf("Tuner does not support MODE_RSS.\n");
				wMode = 0;
			}
			else
			{
				HandleErrorComment("HPI_Tuner_GetMode",wHE);
			}
			wHE = HPI_Multiplexer_GetSource(hSubSys, hMux,0, &wMuxIndex);
			HandleErrorComment("HPI_Multiplexer_GetSource",wHE);

			wHE = HPI_Tuner_GetStatus(hSubSys, hControl, &wStatusMask, &wStatus);
			HandleErrorComment("HPI_Tuner_GetStatus",wHE);

            printf("Current Settings  %s band, frequency %i. Mode %d.  RF level %3.2fdBuV. Mux=%d\n",
                   band_names[nTunerBand],nTunerFrequency,wMode,nRfLevel/100.0,wMuxIndex);
            printf("Status mask %08X, value %08X\n",wStatusMask,wStatus);
        }
       /******** RDS ******/
        if (rds) {
			unsigned int nRDSpackets=100;
			unsigned int nRDStimeout=10;

			HPI_RDS_HANDLE hRDS;
			char aRDS[12];			// RDS data group of (4 blocks each of 16 bits) plus 4 chars of error information
			enum  eHPI_RDS_errors rds_err;

			printf("Reading %d RDS groups from tuner %d\n",nRDSpackets,nTunerIndex);

			hRDS = HPI_RDS_Create(HPI_RDS_DATATYPE_RDS);
			while(nRDSpackets && --nRDStimeout )
			{
				wHE = HPI_Tuner_GetRDS(hSubSys, hControl, aRDS);
				if(!wHE)
				{
					unsigned short *pRDS = (unsigned short *)&aRDS[0];

					nRDStimeout = 10;

					verbose_printf("\n");
					verbose_printf("RDS block data 0x%04X 0x%04X 0x%04X 0x%04X BER 0x%02x 0x%02x 0x%02x 0x%02x\n",
						pRDS[0],pRDS[1],pRDS[2],pRDS[3],aRDS[8],aRDS[9],aRDS[10],aRDS[11]);
					rds_err = HPI_RDS_AnalyzeGroup(hRDS,aRDS,sizeof(aRDS));
					nRDSpackets--;
					if(rds_err)
					{
						if(rds_err==HPI_RDS_ERROR_UNKNOWN_GROUP)
						{
							printf("HPI_RDS_ERROR_UNKNOWN_GROUP : Group %d , Version %c \n",
								HPI_RDS_Get_GroupType(hRDS),
								HPI_RDS_Get_GroupVersion(hRDS));
						}
						else if(rds_err==HPI_RDS_ERROR_BLOCK_DATA)
						{
							unsigned int eb0,eb1,eb2,eb3;

							HPI_RDS_Get_BlockErrorCounts(hRDS,&eb0,&eb1,&eb2,&eb3);
							printf("HPI_RDS_ERROR_BLOCK_DATA : BLER A%d,B%d,C%d,D%d\n",eb0,eb1,eb2,eb3);
						}
						else
							printf("RDS ERROR : %d\n",(int)rds_err);
					}
					else
					{
						printf("RDS (%d%c) PI:%04X TP:%d",
							HPI_RDS_Get_GroupType(hRDS),
							HPI_RDS_Get_GroupVersion(hRDS),
							HPI_RDS_Get_PI(hRDS),
							HPI_RDS_Get_TP(hRDS)
							);

						if(HPI_RDS_Get_PS_Ready(hRDS))
							printf(", PS:%s", HPI_RDS_Get_PS(hRDS));
						if(HPI_RDS_Get_RT_Ready(hRDS))
						{
							unsigned int i;
							// catch carrige return without line feed
							char *pRT = HPI_RDS_Get_RT(hRDS);

							for(i=0;i<strlen(pRT);i++)
								if(pRT[i]==0x0D)
									pRT[i]='\n';
							printf("\n\tRT:%s", pRT);
						}
						printf("\n");
					}

				} else if (wHE == HPI_ERROR_BUFFER_EMPTY)
				{
					verbose_printf(".");
					asi_sleep(100);
				}
				else
					HandleErrorComment("HPI_Tuner_GetRDS",wHE);
			}
			if(nRDStimeout==0)
				printf("RDS time out - no data received.\n");
			HPI_RDS_Delete(hRDS);
        }
    } else
        printf("Only ASI87xx and ASI89xx adapters are supported/n");

    wHE = HPI_MixerClose(
              hSubSys,
              hMixer );
    verbose_printf("\nHPI_MixerClose\n" );
    HandleError( wHE );

    wHE = HPI_AdapterClose(
              hSubSys,
              wAdapterIndex );
    verbose_printf("HPI_AdapterClose\n" );
    HandleError( wHE );

    wHE = HPI_AdapterClose(
              hSubSys,
              wAdapterIndex );

    HPI_SubSysFree(hSubSys);
    return 0;
}

/****************************** HandleError **********************/
void    HandleError(  HW16 wHE )
{
    char szError[256];
    char nK=0;

    if(wHE) {
        HPI_GetErrorText( wHE, szError );
        printf("ERROR %d %s\n",wHE, szError);
        printf("\tpress Enter to continue, (q,Enter) to exit...\n");
        nK = getch();
        if(nK=='q')
            exit(0);
    }
}

void    HandleErrorComment( char * comment, HW16 wHE )
{
    if (wHE)
        printf("%s ",comment);
    HandleError(wHE);
}


int getch(void)
{
    return getchar();
}

/****************************** asi_sleep **********************/
static void asi_sleep(unsigned int milliseconds)
{
#ifdef HPI_OS_LINUX
    struct timespec req = { milliseconds/1000,(milliseconds%1000)*1000000 };
    nanosleep(&req, 0);
#else
	Sleep(milliseconds);
#endif
}


/*
*/
