/******************************************************************************
Copyright (C) 1997-2003 AudioScience, Inc. All rights reserved.

This software is provided 'as-is', without any express or implied warranty.
In no event will AudioScience Inc. be held liable for any damages arising
from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
   claim that you wrote the original software. If you use this software
   in a product, an acknowledgment in the product documentation would be
   appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.
3. This copyright notice and list of conditions may not be altered or removed 
   from any source distribution.

AudioScience, Inc. <support@audioscience.com>

( This license is GPL compatible see http://www.gnu.org/licenses/license-list.html#GPLCompatibleLicenses )

  Test Hardware Programming Interface (HPI) using HPI functions

  Usage: asihpitest --help

note to cleanup this file, use "astyle --style=linux -s4 asihpitest.c"
******************************************************************************/
/* Some customization defines */
// Print out the complete list of mixer controls
#define DUMP_MIXER_CONTROLS 0
// Use background bus mastering if available
#define USE_BBM 1
// Record instreams to files "recN.pcm"
#define RECORD_TO_FILE 0
// Card polling interval in milliseconds
#define POLL_INTERVAL 50
// Test runtime
#define RUNTIME (15 * SECONDS)
// Interval at which stream state is printed
#define PRINT_INTERVAL (1 * SECONDS)

/* End customization defines */

#define SECONDS (1000/POLL_INTERVAL)
#define MINUTES (60*SECONDS)

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
//#include <unistd.h>
#include <getopt.h>
#include <hpi.h>
#include <hpidebug.h>

typedef struct
{
    HW16            wControlType;           // HPI_CONTROL_METER, _VOLUME etc
    HW16            wSrcNodeType;
    HW16            wSrcNodeIndex;
    HW16            wDstNodeType;
    HW16            wDstNodeIndex;
}
hpi_control_t;

HPI_HSUBSYS *hSubSys;   // handle to audio subsystem

static char *hpi_control_strings[] = HPI_CONTROL_TYPE_STRINGS;
static char *hpi_src_strings[] = HPI_SOURCENODE_STRINGS;
static char *hpi_dst_strings[] = HPI_DESTNODE_STRINGS;

// local protos
void HandleError( HW16 wHE );

// file read/write
void THPI_WavFileOpen( short nIndex, char *pszFile);
short THPI_WavFileRead( short nIndex, HW8 *pbData, long lLength);
void THPI_WavFileClose( short nIndex );
int getch(void);

// global
#define BLOCK_SIZE 32768L //30720L //6144 //12288 //16384 //19200 //9600
HW8     abBuffer[BLOCK_SIZE];
short haveFiles=0;

/* Option variables */
int instream_bbm=USE_BBM;
int outstream_bbm=USE_BBM;
unsigned int record_to_file=RECORD_TO_FILE;
unsigned int runtime = RUNTIME+1;
unsigned int dump_mixer = 0;
int max_instreams=999;
int max_outstreams=999;
HW16        wAdapterIndex=0;
int nNumFiles=0;
unsigned int samplerate=48000;
unsigned int channels=2;
unsigned int format=HPI_FORMAT_PCM16_SIGNED;
unsigned int new_mode = 0;


#define MAX_FILES 8

char szFile[MAX_FILES][80]={"","","",""};

static struct option long_options[] =
    {
        {"adapter" , required_argument, 0, 'a'
        },
        {"dump-mixer"    , no_argument, 0, 'd'},
        {"disable-bbm"   , no_argument, 0, 'b'},
        {"adapter-mode"    , required_argument, 0, 'm'},
        {"samplerate"    , required_argument, 0, 's'},
        {"channels"      , required_argument, 0, 'c'},
        {"format"        , required_argument,0,'g'},
        {"file-record"   , no_argument, 0, 'f'},
        {"max-instreams" , required_argument,0,'i'},
        {"max-outstreams", required_argument,0,'o'},
        {"runtime"       , required_argument, 0, 'r'},
        {"help"          , no_argument,0,'h'},
        {0, 0, 0, 0}
    };

const char* short_options="a:bc:d:fg:hi:do:r:s:?";

const char* option_help[]=
    {
        "<adapter number> to test.",
        "Dump list of mixer controls.",
        "Disable use of background bus mastering.",
        "Set the adapter mode. (need driver reload).",
		"<Hz> samplerate.",
		"<n> channels to play or record.",
		"<f> format index (see hpi.h)",
        "Record audio to disk files.",
        "<n> Max number of instreams to open.",
        "<n> Max number of outstreams to open.",
        "<runtime> in seconds.",
        "Show this text."
    };



#ifdef _MSC_VER
void poll_delay(void)
{
	Sleep(POLL_INTERVAL);
}
#else
#include <time.h>

#define millisec 1000000
struct timespec poll_interval=
    {
        0,POLL_INTERVAL*millisec
    };

void poll_delay(void)
{
	nanosleep(&poll_interval, 0);
}
#endif

void help(void)
{
    int i=0;
    printf("\nUsage - asihpitest [options] [files to play]\n");
    while (long_options[i].name != 0) {
        printf("--%s -%c %s\n",
               long_options[i].name,
               (char)(long_options[i].val),
               option_help[i]);
        i++;
    }
    exit(0);
}

void parse_options(int argc, char *argv[])
{
    int c;
    /*********** Parse the command line options ***************/
    while (1) {
      //int this_option_optind = optind ? optind : 1;
        int option_index = 0;

        c = getopt_long (argc, argv, short_options,
                         long_options, &option_index);
        if (c == -1)
            break;

        switch (c) {
        case 0:
            printf ("option %s", long_options[option_index].name);
            if (optarg)
                printf (" with arg %s", optarg);
            printf ("\n");
            break;
        case 'a':
            wAdapterIndex=atoi(optarg);
            break;
        case 'b':
            instream_bbm=0;
            outstream_bbm=0;
            break;
        case 'c':
            channels=atoi(optarg);
            break;
        case 'd':
            dump_mixer=1;
            break;
        case 'f':
            record_to_file=1;
            break;
        case 'g':
            format=atoi(optarg);
            break;
        case 'i':
            max_instreams=atoi(optarg);
            if (max_instreams < 0) {
                max_instreams = -max_instreams;
                instream_bbm=0;
            }
            break;
        case 'm':
            new_mode=atoi(optarg);
            break;
        case 'o':
            max_outstreams=atoi(optarg);
            if (max_outstreams < 0) {
                max_outstreams = -max_outstreams;
                outstream_bbm=0;
            }
            break;
        case 'r':
            runtime=atoi(optarg);
            runtime *= SECONDS;
            break;
        case 's':
            samplerate=atoi(optarg);
            break;
        case '?':
        case 'h':
            help();
            break;

        default:
            printf ("?? getopt returned character code 0%o ??\n", c);
        }
    }

    if (optind < argc) {
        // printf ("non-option ARGV-elements: ");
        nNumFiles=0;
        while ((optind < argc) && (nNumFiles < MAX_FILES))  {
            strcpy( szFile[nNumFiles], argv[optind++] );
            printf("File %d is %s\n",nNumFiles,szFile[nNumFiles]);
            nNumFiles++;
        }
    }

}


void print_mixer_controls(HPI_HMIXER  hMixer)
{
    int f;
    HPI_HCONTROL hControl;
    HW16 wHE;
    for (f=0; f < 10000; f++) { // put SOME limit to the number
        hpi_control_t asihpi_control;

        wHE= HPI_MixerGetControlByIndex(
                 hSubSys,
                 hMixer,
                 f,
                 &asihpi_control.wSrcNodeType,
                 &asihpi_control.wSrcNodeIndex,
                 &asihpi_control.wDstNodeType,
                 &asihpi_control.wDstNodeIndex,
                 &asihpi_control.wControlType,           // HPI_CONTROL_METER, _VOLUME etc
                 &hControl     );
        if (wHE==HPI_ERROR_CONTROL_DISABLED)
		printf("DISABLED ");
	else if (wHE)
            break;

        printf ("HPI Control %d, %s:%s[%d]->%s[%d]\n",
                f,
                hpi_control_strings[asihpi_control.wControlType],
                hpi_src_strings[asihpi_control.wSrcNodeType-HPI_SOURCENODE_BASE],
                asihpi_control.wSrcNodeIndex,
                hpi_dst_strings[asihpi_control.wDstNodeType-HPI_DESTNODE_BASE],
                asihpi_control.wDstNodeIndex
               );
        if (asihpi_control.wControlType==HPI_CONTROL_MULTIPLEXER) {
            int l;
            for (l=0; l<256; l++) {
                wHE=HPI_Multiplexer_QuerySource(hSubSys,  hControl, l,
                                                &asihpi_control.wSrcNodeType,&asihpi_control.wSrcNodeIndex);
                if (!wHE)
                    printf ("\tSource %d %s[%d]\n",
                            l,
                            hpi_src_strings[asihpi_control.wSrcNodeType-HPI_SOURCENODE_BASE],
                            asihpi_control.wSrcNodeIndex
                           );
                else
                    break;
            }
        }
    }
    printf("%d controls found\n",f);
}
/************************************** MAIN ***********************/
int main(int argc, char *argv[])
{
    HW16        wHE=0;  // HPI error
    HW32        dwVersion=0;
    HW32 	dwDataSizeW,dwDataSizeR=0;
    HPI_FORMAT  FormatW,FormatR;
    HW16        wNumAdapters=0;
    HW16        awAdapterList[HPI_MAX_ADAPTERS];
    HW16        wListLength=20;
    HW16	wVersion;
    HW32	dwSerialNumber;
    HW16	wType;
    HW16	wNumOutStreams;
    HW16	wNumInStreams;
        HW32 current_mode;

    HPI_HOSTREAM    hOutStream[16];
    HPI_HISTREAM    hInStream[16];

    HW32        dwStrBufferSize=0;
    HW32        dwIStrBufferSize=0;

    HPI_HMIXER    hMixer=0;
    HPI_HCONTROL hControl;
    HPI_HCONTROL hMeterControl=0;
    HPI_HCONTROL ahGainControl[4]={0,0,0,0};

    short f;

    HW32  i=0;
    short anGainLog[4][2];

    long testcount=0;
    short gwReads=0;

    //////////////////////////////////////////////////////////////////////////
	// #include "msgsize.c"

    parse_options(argc,argv);

    printf("********************************************************************\n");
    printf("\n** Test HPI using Functions **\n");

    if (nNumFiles == 0 ) {
        haveFiles=0;
        nNumFiles=32;
    } else
        haveFiles=1;

    /////////////////////////////////////////////////////////////////////////////
    // open subsystem and find adapters
    printf("********************************************************************\n");
    printf("HPI_SubSysCreate\n");
    hSubSys=HPI_SubSysCreate();
    if (hSubSys==NULL) {
        printf("hSubSys==NULL\n");
        exit(1);
    }

    wHE = HPI_SubSysGetVersion( hSubSys, &dwVersion );
    HandleError( wHE );
    printf("HPI_SubSysGetVersion=%x\n", dwVersion );

    wHE = HPI_SubSysFindAdapters(
              hSubSys,
              &wNumAdapters,
              awAdapterList,
              wListLength
          );
    HandleError( wHE );
    printf("%li HPI_SubSysFindAdapters found %d adapters\n ",testcount++, wNumAdapters);
    for (i=0; i< HPI_MAX_ADAPTERS; i++) {
		if ( awAdapterList[i])
			printf("%d=%X\n ", i, awAdapterList[i] );
	}

    if (awAdapterList[wAdapterIndex] == 0) {
        printf("No adapter with index %d\n",wAdapterIndex);
        exit(1);
    }

    wHE = HPI_AdapterClose(
              hSubSys,
              wAdapterIndex
          );
    HandleError( wHE );
    printf("HPI_AdapterClose \n");

    ////////////////////////////////////////////////////////////////////////////
    // open 1st adapter
    wHE = HPI_AdapterOpen(
              hSubSys,
              wAdapterIndex
          );
    HandleError( wHE );
    printf("HPI_AdapterOpen \n");

    wHE = HPI_AdapterGetInfo(
              hSubSys,
              wAdapterIndex,
              &wNumOutStreams,
              &wNumInStreams,
              &wVersion,
              &dwSerialNumber,
              &wType
          );
    HandleError( wHE );
    printf("HPI_AdapterGetInfo\n");
    printf("Adapter ID=%4X Index=%d NumOutStreams=%d NumInStreams=%d S/N=%d\nHw Version %c%d DSP code version %03d\n",
           wType, wAdapterIndex,
           wNumOutStreams, wNumInStreams, dwSerialNumber,
           ((wVersion>>3)&0xf)+'A',    // Hw version major
           wVersion&0x7,               // Hw version minor
           ((wVersion>>13)*100)+((wVersion>>7)&0x3f)  // DSP code version
          );

    // apply the max streams options
    if (wNumOutStreams> max_outstreams)
        wNumOutStreams=max_outstreams;
    if (wNumInStreams > max_instreams)
        wNumInStreams=max_instreams;

    if (nNumFiles > wNumOutStreams ) {
        nNumFiles=wNumOutStreams;
    }


    wHE = HPI_AdapterGetMode(hSubSys,
                             wAdapterIndex,
                             &current_mode);


    printf("Adapter current mode %d\n",current_mode);
    if (new_mode && (new_mode != current_mode)) {
        wHE = HPI_AdapterSetMode(hSubSys,
                             wAdapterIndex,
                             new_mode);
        HandleError( wHE );

        printf("Changed adapter mode. Reload driver to activate.\n");
        exit(0);
    }

    ////////////////////////////////////////////////////////////////////////////
    // open the mixer of this adapter
    wHE = HPI_MixerOpen(
              hSubSys,
              wAdapterIndex,
              &hMixer
          );
    printf("HPI_MixerOpen: handle=%08X\n", hMixer);
    HandleError( wHE );

    if (dump_mixer)
        print_mixer_controls(hMixer);

    wHE = HPI_MixerGetControl(hSubSys, hMixer,
                              HPI_SOURCENODE_CLOCK_SOURCE, 0,0,0, HPI_CONTROL_SAMPLECLOCK,
                              &hControl);

    if (!wHE) {
        wHE = HPI_SampleClock_SetSampleRate( hSubSys, hControl, 44100 );
        HandleError( wHE );
    }

    ////////////////////////////////////////////////////////////////////////////
    if (wNumOutStreams) {
        // open the out_streams
        printf("HPI_OutStreamOpen\n");
        for (f=0; f<wNumOutStreams; f++) {
            wHE = HPI_OutStreamOpen(
                      hSubSys,
                      wAdapterIndex,
                      f,
                      &hOutStream[f]
                  );
            printf(" %i:%08X ",f,hOutStream[f]);
            HandleError( wHE );

            if (outstream_bbm) {
                printf("HPI_OutStreamHostBufferAllocate ");
                wHE = HPI_OutStreamHostBufferAllocate( hSubSys, hOutStream[f], BLOCK_SIZE);
                if (wHE==HPI_ERROR_INVALID_FUNC)
                    outstream_bbm=0;
                else
                    HandleError(wHE);
            }

            wHE = HPI_OutStreamReset(hSubSys,hOutStream[f]);
            HandleError( wHE );
            // find out some info about the stream
            if (hOutStream[f]) {
                wHE = HPI_OutStreamGetInfoEx(
                          hSubSys,
                          hOutStream[f],
                          NULL, // state
                          &dwStrBufferSize,
                          NULL,NULL,NULL
                      );
                HandleError( wHE );
                printf(" HPI_OutStreamGetInfo %d: BufferSize=%d\n", f,dwStrBufferSize);
            }
        }
    }

    ////////////////////////////////////////////////////////////////////////////
    // setup format and size of data block
    // we will use to send audio data to out_stream
    // we would normally get the format directly from the file
    // or audio data
    wHE = HPI_FormatCreate(
              &FormatW,
              channels,          // stereo channel
              //HPI_FORMAT_MPEG_L2,
              //HPI_FORMAT_MPEG_L3,
              HPI_FORMAT_PCM16_SIGNED,
              //HPI_FORMAT_PCM8_UNSIGNED,
              //HPI_FORMAT_PCM16_BIGENDIAN,
              samplerate,     //sample rate
              //16000,        //sample rate
              128000L,     //128k bits/sec
              0           // no attributes
          );
    HandleError( wHE );

    dwDataSizeW=BLOCK_SIZE/2;
    ////////////////////////////////////////////////////////////////////////////
    if (wNumInStreams) {
        // if the adapter has instreams then open them
        printf("\nHPI_InStreamOpen ");
        for (f=0; f<wNumInStreams; f++) {

            wHE = HPI_InStreamOpen(
                      hSubSys,
                      wAdapterIndex,
                      f,
                      &hInStream[f]
                  );
            printf(" %i:%08X ",f,hInStream[f]);
            HandleError( wHE );
        }

        if( wNumInStreams) {
            // find out some info about stream 0
            wHE = HPI_InStreamGetInfoEx(
                      hSubSys,
                      hInStream[0],
                      NULL,
                      &dwIStrBufferSize,
                      NULL,NULL,NULL
                  );
            printf("\nHPI_InStreamGetInfo %i: BufferSize=%d\n",0, dwStrBufferSize);
            HandleError( wHE );
            wHE = HPI_FormatCreate(
                      &FormatR,
                      channels,          // stereo channel
                      //HPI_FORMAT_MPEG_L2,
                      //HPI_FORMAT_MPEG_L3,
                      HPI_FORMAT_PCM16_SIGNED,
                      //HPI_FORMAT_PCM8_UNSIGNED,
                      //HPI_FORMAT_PCM16_BIGENDIAN,
                      samplerate,     //sample rate
                      128000L,     //128k bits/sec
                      0           // no attributes
                  );
            HandleError( wHE );
            wHE = HPI_InStreamQueryFormat(hSubSys,hInStream[0],&FormatR);
            HandleError( wHE );

            dwDataSizeR=BLOCK_SIZE/2;

            wHE = HPI_MixerGetControl(
                  hSubSys,
                  hMixer,
                  0,
                  0,
                  HPI_DESTNODE_ISTREAM,
                  0,
                  HPI_CONTROL_METER,
                  &hMeterControl
              );
            printf("HPI_MixerGetControl - PeakMeter: handle=%08X\n", hMeterControl);
            HandleError( wHE );
        }
    }

    ////////////////////////////////////////////////////////////////////////////
    if (wNumOutStreams) {
        printf("********************************************************************\n");
        gwReads=0;
        // open some audio files to play
        printf("Opening files \n");
        for (f=0; f<nNumFiles; f++)
            THPI_WavFileOpen( f, szFile[f] );


        //check first available open stream to make sure we can play this format
        for (f=0; f<nNumFiles; f++)
            if (hOutStream[f]) {
                printf("HPI_OutStreamQueryFormat\n");
                wHE = HPI_OutStreamQueryFormat(
                          hSubSys,
                          hOutStream[f],
                          &FormatW
                      );
                HandleError( wHE );
                break;
            }
        // get mixer meter control on out_stream0
        wHE = HPI_MixerGetControl(
                  hSubSys,
                  hMixer,
                  HPI_SOURCENODE_OSTREAM,
                  0,
                  0,
                  0,
                  HPI_CONTROL_METER,
                  &hMeterControl
              );
        printf("HPI_MixerGetControl - PeakMeter: handle=%08X\n", hMeterControl);
        HandleError( wHE );


        for (f=1; f>=0; f--) {
            // get mixer volume control on the connection
            // between out_stream0 and lineOut0
            wHE = HPI_MixerGetControl(
                      hSubSys,
                      hMixer,
                      HPI_SOURCENODE_OSTREAM,
                      f,
                      HPI_DESTNODE_LINEOUT,
                      f,
                      HPI_CONTROL_VOLUME,
                      &ahGainControl[f]
                  );
            // HandleError( wHE );   Ignore the error, some adapters don't have this control

            if (wHE==0) {
                printf("HPI_MixerGetControl - Volume: handle=%08X\n", ahGainControl[f]);
                // test - set gain on stream
                anGainLog[f][0] = 0; //-4*f;
                anGainLog[f][1] = 0; // -4*f-2;
                wHE = HPI_VolumeSetGain(
                          hSubSys,
                          ahGainControl[f],
                          anGainLog[f]
                      );
                HandleError( wHE );
            }
        }

    }

    {

        HW16        wState=0;
        HW32        dwDataToPlay=0;
        FILE * recfile[8];
        char recname[]="rec0.pcm";

        ///////////////////////////////////////////////////////////////////
        // preload buffer of stream
        if (wNumOutStreams) {
            printf("Preload\n");
            for (f=0; f<nNumFiles; f++) {
                if (hOutStream[f]) {
                    for(i=0; i<((dwStrBufferSize-4)/dwDataSizeW); i++) {
                        //printf("f=%d, i=%d,bs=%d,ds=%d\n",f,i,dwStrBufferSize,DataW.dwDataSize);
                        // out_stream #1
                        THPI_WavFileRead( f, abBuffer , dwDataSizeW );

                        wHE = HPI_OutStreamWriteBuf(
                                  hSubSys,
                                  hOutStream[f],
                                  abBuffer,dwDataSizeW,&FormatW );
                        HandleError( wHE );

                        wHE = HPI_OutStreamGetInfoEx(
                                  hSubSys,
                                  hOutStream[f],
                                  &wState,
                                  NULL,
                                  &dwDataToPlay,
				  NULL,NULL
                              );
                        HandleError( wHE );

                        printf("[%i] D=%03d S=%d   \r",f,dwDataToPlay/1000, wState);

                    }
                    printf("\n");
                }
            }

            ///////////////////////////////////////////////////////////////////
            // start play back

            printf("********************************************************************\n");
            printf("HPI_OutStreamStart ");
            for (f=0; f<nNumFiles; f++) {
                if (hOutStream[f]) {

                    wHE = HPI_OutStreamStart(
                              hSubSys,
                              hOutStream[f]
                          );
                    HandleError( wHE );
                    printf("%i ",f);
                }
            }
            printf ("\n");
        }
        for (f=0; f<wNumInStreams; f++) {
            printf("%d: ",f);

            if (instream_bbm) {
                printf("HPI_InStreamHostBufferAllocate ");
                wHE = HPI_InStreamHostBufferAllocate( hSubSys, hInStream[f], BLOCK_SIZE);
                if (wHE==HPI_ERROR_INVALID_FUNC)
                    instream_bbm=0;
                else
                    HandleError(wHE);
            }

            printf("HPI_InStreamSetFormat ");
            wHE = HPI_InStreamSetFormat(
                      hSubSys,
                      hInStream[f],
                      &FormatR);
            HandleError( wHE );

            printf("HPI_InStreamReset " );
            wHE = HPI_InStreamReset(
                      hSubSys,
                      hInStream[f] );
            HandleError( wHE );


            printf("HPI_InStreamStart \n");
            wHE = HPI_InStreamStart(
                      hSubSys,
                      hInStream[f]);
            HandleError( wHE );

            if (record_to_file) {
				if (szFile[f][0])
					recfile[f]=fopen(szFile[f],"w");
				else {
					recname[3]=f+'0';
					recfile[f]=fopen(recname,"w");
				}
            }
        }

        ///////////////////////////////////////////////////////////////////
        // monitor state of streams and stop when empty
        while(--runtime) {

            int do_print = ((runtime % PRINT_INTERVAL ) == 0);

            poll_delay();

            if (do_print && wNumInStreams)
                printf("IN :");
            for (f=0; f<wNumInStreams; f++) {
                //get state of in stream
                wHE = HPI_InStreamGetInfoEx(
                          hSubSys,
                          hInStream[f],
                          &wState,
                          NULL,
                          &dwDataToPlay,
			  NULL,NULL
                      );
                HandleError( wHE );

                if (do_print)
                    printf("%03d %d ",dwDataToPlay/1000, wState);

                if ( dwDataToPlay >= dwDataSizeR) {
                    dwDataToPlay=dwDataSizeR;

                    wHE = HPI_InStreamReadBuf(
                              hSubSys,
                              hInStream[f],
                              abBuffer,
                              dwDataToPlay );

                    HandleError( wHE );

                    if (record_to_file)
                        fwrite(abBuffer,dwDataToPlay,1,recfile[f]);
                }
            }
            if (do_print && wNumInStreams)
                printf("\n");

            if (do_print && wNumOutStreams)
                printf("OUT:");
            for (f=0; f<nNumFiles; f++) {
                if (hOutStream[f]) {
                    //get state of stream
                    wHE = HPI_OutStreamGetInfoEx(
                              hSubSys,
                              hOutStream[f],
                              &wState,
                              NULL,
                              &dwDataToPlay,
			      NULL,NULL
                          );
                    HandleError( wHE );
                    if (do_print)
                        printf("%03d %d ",dwDataToPlay/1000, wState);

                    if ((dwStrBufferSize-dwDataToPlay) > dwDataSizeW) {
                        THPI_WavFileRead( f, abBuffer, dwDataSizeW );
                        wHE = HPI_OutStreamWriteBuf(
                                  hSubSys,
                                  hOutStream[f],
                                  abBuffer,dwDataSizeW,&FormatW );

                        HandleError( wHE );
                    }
                }
            }
            if (do_print && wNumOutStreams)
                printf("\n");
        }

        if (record_to_file)
            for (f=0; f<wNumInStreams; f++)
                fclose(recfile[f]);

        printf("\n");
    }

    for (f=0; f<nNumFiles; f++)
        THPI_WavFileClose(f);

    if (wNumOutStreams) {
        printf("HPI_OutStreamStop " );
        for (f=0; f<nNumFiles; f++) {
            if (hOutStream[f]) {
                wHE = HPI_OutStreamStop(
                          hSubSys,
                          hOutStream[f] );
                printf("%i ",f );
                HandleError( wHE );
            }
        }

        printf("\nHPI_OutStreamClose ");
        for (f=0; f<wNumOutStreams; f++) {
            if (hOutStream[f]) {
                if (outstream_bbm)
                    wHE=HPI_OutStreamHostBufferFree(
                            hSubSys,
                            hOutStream[f]);

                wHE = HPI_OutStreamClose(
                          hSubSys,
                          hOutStream[f] );
                printf("%i ",f );
                HandleError( wHE );
            }
        }
    }

    if( wNumInStreams) {
        printf("\nHPI_InStreamStop \n" );
        wHE = HPI_InStreamStop(
                  hSubSys,
                  hInStream[0] );
        HandleError( wHE );

        printf("\nHPI_InStreamClose ");
        for (f=0; f<wNumInStreams; f++) {
            if (instream_bbm)
                wHE=HPI_InStreamHostBufferFree(
                        hSubSys,
                        hInStream[f]);

            wHE = HPI_InStreamClose(
                      hSubSys,
                      hInStream[f] );
            printf("%i ",f );
            HandleError( wHE );
        }
    }

    wHE = HPI_MixerClose(
              hSubSys,
              hMixer );
    printf("\nHPI_MixerClose\n" );
    HandleError( wHE );

    wHE = HPI_AdapterClose(
              hSubSys,
              wAdapterIndex );
    printf("HPI_AdapterClose\n" );
    HandleError( wHE );

    HPI_SubSysFree(hSubSys);

    return 0;
}

/******************************THPI_WavFileOpen **********************/
FILE *gpFile[8];

void THPI_WavFileOpen( short nIndex, char *pszFile)
{
    if (haveFiles) {
        printf("%s ", pszFile);
        gpFile[nIndex] = fopen(pszFile,"rb");
        if(!gpFile[nIndex]) {
            printf("****ERROR**** - can't open file\n");
            getch();
            exit(0);
        }
        if(toupper(pszFile[strlen(pszFile)-1]) == 'S')
            fseek(gpFile[nIndex], 512, SEEK_SET);       //.DSS file
        else
            fseek(gpFile[nIndex], 0x50, SEEK_SET);          //.WAV file
    }
}

/******************************THPI_WavFileRead **********************/
short THPI_WavFileRead( short nIndex, HW8 *pbData, long lLength)
{
    long lNumRead;
    long i=0;
    HW16 wSine;

    if (haveFiles) {
        lNumRead = fread( pbData, 1, lLength, gpFile[nIndex] );         //read WAV file
        if( lNumRead != lLength)
            return(1);
        else
            return(0);
    } else {
        // setup sinewave in buffer
        for(i=0; i<lLength/4; i++) {
            wSine = (HW16)(32767*sin(2*3.141592653*i/32));
            pbData[(short)i*4  ] = (HW8)wSine;         //left LSB
            pbData[(short)i*4+1] = (HW8)(wSine>>8);    //left MSB
            pbData[(short)i*4+2] = (HW8)wSine;    //right LSB
            pbData[(short)i*4+3] = (HW8)(wSine>>8);   //right MSB
        }
        return(0);
    }
}

/******************************THPI_WavFileClose **********************/
void THPI_WavFileClose(short nIndex )
{
    if (haveFiles)
        fclose(gpFile[nIndex]);
}


/****************************** HandleError **********************/
void    HandleError( HW16 wHE )
{
    char szError[256];
    char nK=0;

    if(wHE) {
        HPI_GetErrorText( wHE, szError );
        printf("ERROR %d %s\n", wHE, szError);
        printf("press Enter to continue, (q,Enter) to exit...\n");
        nK = getch();
        if(nK=='q')
            exit(0);
    }
}

int getch(void)
{
    return getchar();
}

/*
*/
