/******************************************************************************
Copyright (C) 1997-2003 AudioScience, Inc. All rights reserved.

This software is provided 'as-is', without any express or implied warranty.
In no event will AudioScience Inc. be held liable for any damages arising
from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
   claim that you wrote the original software. If you use this software
   in a product, an acknowledgment in the product documentation would be
   appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.
3. This copyright notice and list of conditions may not be altered or removed 
   from any source distribution.

AudioScience, Inc. <support@audioscience.com>

( This license is GPL compatible see http://www.gnu.org/licenses/license-list.html#GPLCompatibleLicenses )
 
Commandline play/record application using HPI

******************************************************************************/

// Use background bus mastering if available
#define USE_BBM 1
// Card polling interval in milliseconds
#define POLL_INTERVAL 30
// Test runtime
#define RUNTIME (15 * SECONDS)

/* End customization defines */

#define SECONDS (1000/POLL_INTERVAL)
#define MINUTES (60*SECONDS)

#define REC_PROGNAME "asihpirec" 
#define PLAY_PROGNAME "asihpiplay" 

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
//#include <unistd.h>
#include <getopt.h>
#include <hpi.h>
#include <hpidebug.h>

typedef struct
{
    HW16            wControlType;           // HPI_CONTROL_METER, _VOLUME etc
    HW16            wSrcNodeType;
    HW16            wSrcNodeIndex;
    HW16            wDstNodeType;
    HW16            wDstNodeIndex;
}
hpi_control_t;

/*---------------------------------------------------------------------------*/
/*GLOBALS*/
HPI_HSUBSYS *hSubSys;   // handle to audio subsystem
HPI_HMIXER hMixer = 0;
HPI_HOSTREAM hOutStream;
HPI_HISTREAM hInStream;
//HPI_DATA hData;
HPI_FORMAT hFormat;
unsigned int bufsize;

// local protos
void sigintHandler( int sig );
void _HandleError( HW16 wHE, const char *filename, const int linenum );

#define HandleError( wHE ) if((wHE)) _HandleError( (wHE), __FILE__, __LINE__ )

#define HandleErrorAndExit( wHE ) if((wHE)) { _HandleError( (wHE), __FILE__, __LINE__ ); sigintHandler(0); }

// global
#define BLOCK_SIZE 32768L //30720L //6144 //12288 //16384 //19200 //9600
HW8 abBuffer[BLOCK_SIZE];

/* Option variables */
int stream_use_bbm = USE_BBM;
int runtime = -1; //run forever
unsigned int stream_num = 0; // default to stream 0
HW16 wAdapterIndex = 0; // default to adapter 0
unsigned int samplerate = 48000;
unsigned int channels = 2;
unsigned int format = HPI_FORMAT_PCM16_SIGNED;
int isRec = 1;
int looping = 0;
int verbose = 0;

#define MAX_PATH 256
char szFile[MAX_PATH] = "-";
FILE *hFile = NULL;

static struct option long_options[] =
    {
        {"adapter" , required_argument, 0, 'a'},
        {"disable-bbm"   , no_argument, 0, 'b'},
        {"samplerate"    , required_argument, 0, 's'},
        {"channels"      , required_argument, 0, 'c'},
        {"format"        , required_argument,0,'f'},
        {"stream-number" , required_argument,0,'n'},
        {"runtime"       , required_argument, 0, 'r'},
        {"loop"           , no_argument, 0, 'l'},
        {"verbose"        , no_argument, 0, 'v'},
        {"help"          , no_argument,0,'h'},
        {0, 0, 0, 0}
    };

const char* short_options="a:bc:f:hln:r:s:v?";

const char* option_help[]=
    {
        "<adapter number> to test.",
        "Disable use of background bus mastering.",
		"<Hz> samplerate.",
		"<n> channels to play or record.",
		"<f> format index 2=PCM16,4=MP2,5=MP3 14=F32 (see hpi.h)",
        "<n> stream # to open.",
        "<runtime> in seconds.",
        "Loop file (play only).",
	"Verbose info display",
        "Show this text."
    };

#ifdef _MSC_VER
void poll_delay(void)
{
	Sleep(POLL_INTERVAL);
}
#else
#include <time.h>

#define millisec 1000000
struct timespec poll_interval=
    {
        0,POLL_INTERVAL*millisec
    };

/*---------------------------------------------------------------------------*/
void poll_delay(void)
{
	nanosleep(&poll_interval, 0);
}
#endif

/*---------------------------------------------------------------------------*/
void help(void)
{
    int i=0;
    printf("\nUsage - asihpirec [options] [file]\n");
    while (long_options[i].name != 0) {
        printf("--%s -%c %s\n",
               long_options[i].name,
               (char)(long_options[i].val),
               option_help[i]);
        i++;
    }
    exit(0);
}
/*---------------------------------------------------------------------------*/
void parse_options(int argc, char *argv[])
{
    int c;
    /*********** Parse the command line options ***************/
    while (1) {
      //int this_option_optind = optind ? optind : 1;
        int option_index = 0;

        c = getopt_long (argc, argv, short_options,
                         long_options, &option_index);
        if (c == -1)
            break;

        switch (c) {
        case 0:
            printf ("option %s", long_options[option_index].name);
            if (optarg)
                printf (" with arg %s", optarg);
            printf ("\n");
            break;
        case 'a':
            wAdapterIndex = atoi(optarg);
            break;
        case 'b':
            stream_use_bbm = 0;
            break;
        case 'l':
            if (!isRec)
                looping=1;
            break;
        case 'c':
            channels = atoi(optarg);
            break;
        case 's':
            samplerate = atoi(optarg);
            break;
        case 'f':
            format = atoi(optarg);
            break;
        case 'n':
            stream_num = atoi(optarg);
            break;
        case 'r':
            runtime = atoi(optarg);
            runtime *= SECONDS;
            break;
	case 'v':
		verbose=1;
		break;
        case '?':
        case 'h':
            help();
            break;

        default:
            printf ("?? getopt returned character code 0%o ??\n", c);
        }
    }

    if (optind < argc) {
        while ((optind < argc))  {
            strncpy( szFile, argv[optind++], MAX_PATH - 1 );
        }
    }

}

static char * state_names[]={
	"invalid",
	"stopped",
	"play",
	"rec",
	"drained",
	"sine"
};


/*---------------------------------------------------------------------------*/
int doRec( HPI_HSUBSYS * hSubSys, HPI_HISTREAM * hStream, unsigned int bufsize, 
           FILE * hOutFile ) 
{
    HW16 wHE = 0;
    HW16 wState = HPI_STATE_STOPPED;
    HW32 dwDataToPlay,dwSamplesRecorded,dwAuxData;

    //get state of in stream
    wHE = HPI_InStreamGetInfoEx(
                          hSubSys,
                          *hStream,
                          &wState,
                          NULL,
                          &dwDataToPlay,&dwSamplesRecorded,&dwAuxData );
    HandleError( wHE );

    if (verbose) {
	char * sn=state_names[0];
	if (wState <= HPI_STATE_SINEGEN)
		sn=state_names[wState];
    	printf("%s %8d %8d %8d\n",sn,dwSamplesRecorded,dwDataToPlay, dwAuxData);
    }

    if ( dwDataToPlay >= bufsize) {
        wHE = HPI_InStreamReadBuf(hSubSys, *hStream, abBuffer,  bufsize);
        HandleError( wHE );

        if (!wHE ) {
            fwrite(abBuffer, bufsize, 1, hOutFile );
        }
    }
    return ( wHE || wState == HPI_STATE_STOPPED );
}

/*---------------------------------------------------------------------------*/
int doPlay( HPI_HSUBSYS * hSubSys, HPI_HOSTREAM * hStream, 
            HPI_FORMAT * hpiFormat, unsigned int dwBufSizeW, FILE * hInFile ) 
{
    HW16 wHE = 0;
    HW16 wState = HPI_STATE_DRAINED;
    HW32 dwDataToPlay;
    HW32 dwStrBufferSize;
    HW32 dwSamplesPlayed,dwAuxData;

    //get state of stream
    wHE = HPI_OutStreamGetInfoEx(
                              hSubSys,
                              *hStream,
                              &wState,
                              &dwStrBufferSize,
                              &dwDataToPlay,&dwSamplesPlayed,&dwAuxData );
    HandleError( wHE );

    if (verbose) {
	char * sn=state_names[0];
	if (wState <= HPI_STATE_SINEGEN)
		sn=state_names[wState];
    	printf("%s %8d %8d %8d\n",sn,dwSamplesPlayed,dwDataToPlay, dwAuxData);
    }
	
    if ( ( dwStrBufferSize - dwDataToPlay ) > dwBufSizeW ) {
        fread(abBuffer, dwBufSizeW, 1, hInFile );
        //stop playing on end of file
        if ( feof( hInFile ) ) return 1;
	wHE = HPI_OutStreamWriteBuf(hSubSys, *hStream, abBuffer, dwBufSizeW, hpiFormat );
        HandleError( wHE );
    }
    return ( wHE || wState == HPI_STATE_STOPPED );
}
 
/*---------------------------------------------------------------------------*/
void startRec( HPI_HSUBSYS * hSubSys,
                    HW16 wAdapterIndex,
                    unsigned int streamNum,
                    unsigned int format,
                    unsigned int channels,
                    unsigned int samplerate,
                    HPI_HISTREAM * hStream,
                    HPI_FORMAT * hpiFormat,
                    unsigned int * bufSize,
                    FILE * hInFile ) {

    HW16        wHE=0;

    wHE = HPI_InStreamOpen(
                      hSubSys,
                      wAdapterIndex,
                      streamNum,
                      hStream );
    HandleErrorAndExit( wHE );

    wHE = HPI_FormatCreate(
                      hpiFormat,
                      channels,
                      format,
                      samplerate,
                      128000L,
                      0           // no attributes
                  );
    HandleErrorAndExit( wHE );

    *bufSize=BLOCK_SIZE/2;

    wHE = HPI_InStreamQueryFormat( hSubSys, *hStream, hpiFormat );
    HandleErrorAndExit( wHE );

    if (stream_use_bbm) {
        wHE = HPI_InStreamHostBufferAllocate( hSubSys, *hStream, BLOCK_SIZE );
        if ( wHE==HPI_ERROR_INVALID_FUNC )
            stream_use_bbm=0;
        else
            HandleErrorAndExit(wHE);
    }

    wHE = HPI_InStreamReset(
                      hSubSys,
                      *hStream );
    HandleErrorAndExit( wHE );


    wHE = HPI_InStreamSetFormat(
                      hSubSys,
                      *hStream,
                      hpiFormat );
    HandleErrorAndExit( wHE );

    wHE = HPI_InStreamStart(
                      hSubSys,
                      *hStream );
    HandleErrorAndExit( wHE );

} 

/*---------------------------------------------------------------------------*/
void startPlay( HPI_HSUBSYS * hSubSys,
                     HW16 wAdapterIndex,
                     unsigned int streamNum,
                     unsigned int format,
                     unsigned int channels,
                     unsigned int samplerate,
                     HPI_HISTREAM * hStream,
                     HPI_FORMAT * hpiFormat,
                     unsigned int * bufsize,
                     FILE * hOutFile ) {

    HW16        wHE=0;

    wHE = HPI_OutStreamOpen(
                      hSubSys,
                      wAdapterIndex,
                      streamNum,
                      hStream
                  );
    HandleErrorAndExit( wHE );

    if (stream_use_bbm) {
        wHE = HPI_OutStreamHostBufferAllocate( hSubSys, *hStream, BLOCK_SIZE );
        if ( wHE==HPI_ERROR_INVALID_FUNC )
            stream_use_bbm=0;
        else
            HandleErrorAndExit(wHE);
    }

    wHE = HPI_OutStreamReset( hSubSys, *hStream );
    HandleErrorAndExit( wHE );

    // setup format and size of data block
    wHE = HPI_FormatCreate(
              hpiFormat,
              channels,
              format,
              samplerate,     //sample rate
              128000L,     //128k bits/sec
              0           // no attributes
          );
    HandleErrorAndExit( wHE );

    *bufsize = BLOCK_SIZE/2 ;

    //check first available open stream to make sure we can play this format
    wHE = HPI_OutStreamQueryFormat(
                          hSubSys,
                          *hStream,
                          hpiFormat );
    HandleErrorAndExit( wHE );

    //prefetch
    doPlay( hSubSys, hStream, hpiFormat, *bufsize, hFile );

    wHE = HPI_OutStreamStart( hSubSys,
                              *hStream );
    HandleErrorAndExit( wHE );

}

/*---------------------------------------------------------------------------*/
void sigintHandler(int sig) {

    HW16        wHE=0;

    fprintf( stderr, "Terminating.\n" );

    if (!hSubSys)
        return;

    if (!isRec) {

        wHE = HPI_OutStreamStop(
                          hSubSys,
                          hOutStream );
        HandleError( wHE );

        //drain the buffer
        while( !doPlay( hSubSys, &hOutStream, &hFormat, bufsize ,hFile ) ){}

        wHE = HPI_OutStreamClose(
                          hSubSys,
                          hOutStream );
        HandleError( wHE );

        if (stream_use_bbm)
            wHE=HPI_OutStreamHostBufferFree(
                            hSubSys,
                            hOutStream );

    } else {

        wHE = HPI_InStreamStop(
                  hSubSys,
                  hInStream );
        HandleError( wHE );

        //drain the buffer
        while( !doRec( hSubSys, &hInStream, bufsize, hFile ) ){}

        wHE = HPI_InStreamClose(
                      hSubSys,
                      hInStream );
        HandleError( wHE );

        if (stream_use_bbm)
            wHE=HPI_InStreamHostBufferFree(
                        hSubSys,
                        hInStream);
    }

    if (hFile)
        fclose(hFile);

    if (hMixer) {
        wHE = HPI_MixerClose(
              hSubSys,
              hMixer );
        HandleError( wHE );
    } 

    wHE = HPI_AdapterClose(
              hSubSys,
              wAdapterIndex );

    HPI_SubSysFree(hSubSys);
    exit(0);
}

/************************************** MAIN ***********************/
int main(int argc, char *argv[])
{
    HW16        wHE=0;  // HPI error
    HW32        dwVersion=0;
 
    HW16        wNumAdapters=0;
    HW16        awAdapterList[HPI_MAX_ADAPTERS];
    HW16        wListLength=20;
    HW16	wVersion;
    HW32	dwSerialNumber;
    HW16	wType;
    HW16	wNumOutStreams;
    HW16	wNumInStreams;
    HPI_HCONTROL hControl;

    if ( strlen( argv[0] ) >= strlen( REC_PROGNAME ) && strncmp(argv[0], REC_PROGNAME ,strlen( REC_PROGNAME )) == 0 )
        isRec = 1;
    else if ( strlen( argv[0] ) >= strlen( PLAY_PROGNAME ) && strncmp(argv[0], PLAY_PROGNAME,strlen( PLAY_PROGNAME ) ) == 0 )
        isRec= 0;
    else {
        fprintf( stderr, "call this executable either %s or %s", REC_PROGNAME, PLAY_PROGNAME );
        exit(1);
    }
    
    parse_options(argc,argv);

    signal( SIGINT, sigintHandler );

    if (isRec) {
        if ( strcmp("-", szFile) == 0 )
            hFile = fdopen(fileno(stdout),"wb");
        else
            hFile = fopen( szFile, "w+b" );
    } else {
        if ( strcmp("-", szFile) == 0 )
            hFile = fdopen(fileno(stdin),"rb");
        else
            hFile = fopen( szFile, "rb" );
    }

    // open subsystem and find adapters
    hSubSys=HPI_SubSysCreate();
    if (hSubSys==NULL) {
        fprintf(stderr,"hSubSys==NULL\n");
        exit(1);
    }

    wHE = HPI_SubSysGetVersion( hSubSys, &dwVersion );
    HandleErrorAndExit( wHE );
    fprintf(stderr,"SubSys version=%x\n", dwVersion );

    wHE = HPI_SubSysFindAdapters(
              hSubSys,
              &wNumAdapters,
              awAdapterList,
              wListLength
          );
    HandleErrorAndExit( wHE );
    fprintf(stderr,"Found %d adapters\n", wNumAdapters);

    if (awAdapterList[wAdapterIndex] == 0) {
        fprintf(stderr,"No adapter with index %d\n",wAdapterIndex);
        exit(1);
    }

    wHE = HPI_AdapterOpen(
              hSubSys,
              wAdapterIndex
          );
    HandleErrorAndExit( wHE );

    wHE = HPI_AdapterGetInfo(
              hSubSys,
              wAdapterIndex,
              &wNumOutStreams,
              &wNumInStreams,
              &wVersion,
              &dwSerialNumber,
              &wType
          );
    HandleErrorAndExit( wHE );

    fprintf(stderr,"Using adapter ID=%4X Index=%d NumOutStreams=%d NumInStreams=%d S/N=%d\nHw Version %c%d DSP code version %03d\n",
           wType, wAdapterIndex,
           wNumOutStreams, wNumInStreams, dwSerialNumber,
           ((wVersion>>3)&0xf)+'A',    // Hw version major
           wVersion&0x7,               // Hw version minor
           ((wVersion>>13)*100)+((wVersion>>7)&0x3f)  // DSP code version
          );
/*
    //HPI doesn't come back with an error if we try to use a non existing stream, it crashes instead
    if ( ( !isRec && stream_num >= wNumOutStreams ) || ( isRec && stream_num >= wNumInStreams ) )
        HandleErrorAndExit( HPI_ERROR_INVALID_STREAM );
*/
    // open the mixer of this adapter
    wHE = HPI_MixerOpen(
              hSubSys,
              wAdapterIndex,
              &hMixer
          );

    HandleErrorAndExit( wHE );

    wHE = HPI_MixerGetControl(hSubSys, hMixer,
                              HPI_SOURCENODE_CLOCK_SOURCE, 0,0,0, HPI_CONTROL_SAMPLECLOCK,
                              &hControl);

//    if (!wHE) {
//        wHE = HPI_SampleClock_SetSampleRate( hSubSys, hControl, 44100 );
//        HandleErrorAndExit( wHE );
//    }

    if (isRec) {
        fprintf( stderr,"Recording from stream %d, format %d, channels %d, samplerate %d\n",
                        stream_num, format, channels, samplerate );
        startRec( hSubSys, wAdapterIndex, stream_num, format, channels, 
                  samplerate, &hInStream, &hFormat, &bufsize, hFile );
    } else {
        fprintf( stderr,"Playing to stream %d, format %d, channels %d, samplerate %d\n",
                        stream_num, format, channels, samplerate );
        if (looping)
            fprintf(stderr,"Looping forever ctrl-C to exit\n");
        startPlay( hSubSys, wAdapterIndex, stream_num, format, channels, 
                   samplerate, &hOutStream, &hFormat, &bufsize, hFile );
    }

    while(1) {
        
        if ( runtime != -1 )
            if (! --runtime )
                break;

        poll_delay();
        if (isRec) {
            if ( doRec( hSubSys, &hInStream, bufsize, hFile ) )
                break;
        } else {
            if( doPlay( hSubSys, &hOutStream, &hFormat, bufsize, hFile ) ) {
                if (looping) {
                    rewind(hFile);
                    continue;
                }
                break;
            }
        }
    }

    sigintHandler(0);
    return 0;
}

/****************************** HandleError **********************/
void _HandleError( HW16 wHE, const char *filename, const int linenum ) {
    char szError[256];

    HPI_GetErrorText( wHE, szError );
    fprintf(stderr,"%s %d HPI ERROR %d %s\n", filename, linenum, wHE, szError);
}

/* 
*/
