/******************************************************************************
$Header: /Repository/drv/hpi/hpiudp.c,v 1.28 2007/09/11 18:18:29 as-age Exp $

UDP HPI NETWORK implementation (Windows)

See
<http://www.sockets.com/winsock.htm>

Sequence of operations:
On SubSysOpen() a message is broadcast to all ASI2416 adapters. Any responses received are
put into the hpi_adapter structure, which is a linked list of adapters.

On AdapterOpen() the specific socket is created and "bound" to the adapter. On AdapterCloser()
the socket is released.

Notes - this module should in theory operate for adapter indexes > 16 with a few minor changes.

(C) Copyright AudioScience Inc. 2006

*******************************************************************************/

#ifdef HPI_INCLUDE_UDP

#define SOURCEFILE_NAME "hpiudp.c"

#ifndef WIN32_LEAN_AND_MEAN
#	define WIN32_LEAN_AND_MEAN
#endif

#include <stdio.h>
#include <stdlib.h>

#ifdef HPI_OS_WIN32_USER
#	include <Winsock2.h>
#   define UDP_CACHING
#else
#	undef UDP_CACHING
#	include <sys/types.h>
#	include <sys/socket.h>  /* u_int, u_char etc. */
#	include <arpa/inet.h>  /* u_int, u_char etc. */
#	include <sys/time.h>
#	include <resolv.h>
#	include <string.h>
#	include <errno.h>
#	include <unistd.h>

#	define SOCKET int
#	define TCHAR char
#	define SOCKADDR struct sockaddr
#	define TEXT(T) T

#	define WSAETIMEDOUT EAGAIN
#	define INVALID_SOCKET -1
#	define SOCKET_ERROR -1

#	define closesocket close
#	define WSACleanup()
#	define WSAGetLastError() errno

#endif

#include <hpi.h>
#include <hpinet.h>
#include <hpidebug.h>
#include <hpicmn.h>


typedef void HPI_message_function(HPI_MESSAGE *phm, HPI_RESPONSE *phr );
struct hpi_adapter;

#ifdef UDP_CACHING
struct hpi_adapter_context{
	unsigned int nControls;
	unsigned int timeStamp_ms;
	HPI_CONTROL_DEFN controlDef[1024];
	tHPIControlCacheSingle controlCache[1024];
};
#endif

struct hpi_adapter{
	HPI_message_function * hpi_message;
	HPI_RESPONSE bhr;
	HW32 adapterType;
	HW16 adapterIndex;
	struct in_addr addr;
	unsigned int adapterIsOpen;
	unsigned int mixerIsOpen;
	SOCKET sock;
#ifdef UDP_CACHING
	HANDLE hFileMappedContext;
	struct hpi_adapter_context *cache;
#endif
	struct hpi_adapter *next; /* can make a linked list of them */
};

struct asi_pkt {
	unsigned char ASIpktID;
	unsigned char ASIpktVersion;
};

#define DISCOVERY_TIMEOUT_SECONDS 1
#define DISCOVERY_TIMEOUT_MS      0

/* Flash erase takes several seconds before returning response */
#define NET_TIMEOUT_SECONDS 10
#define NET_TIMEOUT_MS      0

/** global: List of adapters known by this HPI */
struct hpi_adapter *first_adapter=0;
static int gnSocketCleanupRequired=0;

/** Local prototypes */

// HPI type functions.
HW16 SubsysOpen(void );
HW16 SubsysClose(void);
static void SubsysFindAdapters(HPI_MESSAGE *phm, HPI_RESPONSE *phr);
static int SubsysGetNumAdapters(void);
static int SubsysGetAdapter(int nIndex, HW16 *pwAdapterIndex, HW16 *pwAdapterType);
static void AdapterOpen(HPI_MESSAGE *phm, HPI_RESPONSE *phr);
static void AdapterClose(HPI_MESSAGE *phm, HPI_RESPONSE *phr);

// utility functions
int HPINET_checkReceivePacket(char *szData,	HPI_RESPONSE *phr);
void HPINET_createTransmitPacket(char *szData, HPI_MESSAGE *phm);
int HPINET_CheckSockError(const int nRet, const TCHAR *szFn, HPI_RESPONSE *phr);
static struct hpi_adapter *findAdapter(const HW16 wAdapterIndex);
int CountAdapters(void);
#ifdef UDP_CACHING
static int UpdateControlCache(struct hpi_adapter *pA);
#endif

void HPI_NET(HPI_MESSAGE *phm, HPI_RESPONSE *phr );
/*=================================================================*/
/* Global variables
*/
char szIP[64]="";

/*=================================================================*/
/** Discover all ASI products on the metwork.
  */
int HPINET_MessageDiscovery(void)
{
	HPI_MESSAGE hm;
	HPI_RESPONSE hr;
	HPI_MESSAGE *phm=&hm;
	HPI_RESPONSE *phr=&hr;
	struct sockaddr_in destAdr;
	char szBuffer[HPINET_ASI_DATA_SIZE+2];	// +2 to account for asi_pkt space.
	int num_adapters=0;
	int intBool;
	int status;
	int nSent,nSize;
	unsigned int nFromLen;
#ifdef HPI_OS_LINUX
	struct timeval timeout = {
		.tv_sec = DISCOVERY_TIMEOUT_SECONDS,
		.tv_usec = DISCOVERY_TIMEOUT_MS * 1000,
	};
#else
	int timeout = DISCOVERY_TIMEOUT_SECONDS * 1000 + DISCOVERY_TIMEOUT_MS;
#endif

	SOCKET sock;

    /* Create socket for sending/receiving datagrams */
	HPI_DEBUG_LOG0(DEBUG,TEXT("Opening socket\n"));
	sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
	if(sock==INVALID_SOCKET)
	{
		HPI_DEBUG_LOG0(ERROR,TEXT("Error opening socket\n"));
		gnSocketCleanupRequired=0;
		WSACleanup();			// terminate use of WS2_32.DLL
		return -1;
	}
	/* Can we bind the socket to an IP address ? */
	//if(0)
	if(strlen(szIP))
	{
		/* bind it */
		struct sockaddr_in service;
		service.sin_family = AF_INET;
                    service.sin_addr.s_addr = inet_addr(szIP);
		service.sin_port = htons(HPI_ETHERNET_UDP_PORT);
		status=bind(sock, (SOCKADDR*) &service, sizeof(service));
		HPINET_CheckSockError(status, TEXT("bind() \n"), phr);
	}

	HPI_InitMessage( &hm, HPI_OBJ_ADAPTER, HPI_ADAPTER_GET_INFO );

	HPINET_createTransmitPacket(szBuffer,phm);

	// Set up the destination address structure
	destAdr.sin_family = AF_INET;
	destAdr.sin_port = htons(HPI_ETHERNET_UDP_PORT);
	destAdr.sin_addr.s_addr = INADDR_BROADCAST;

	// set broadcast permission
	intBool=1;
	status=setsockopt(sock, SOL_SOCKET, SO_BROADCAST, (void *)&intBool,
			  sizeof(intBool));
	HPINET_CheckSockError(status, TEXT("setsockopt SO_BROADCAST\n"), phr);
	// set send timeout option
	status=setsockopt(sock, SOL_SOCKET, SO_SNDTIMEO, (void *)&timeout, sizeof(timeout));
	HPINET_CheckSockError(status, TEXT("setsockopt SO_SNDTIMEO\n"), phr);
	// set receive timeout option
	status=setsockopt(sock, SOL_SOCKET, SO_RCVTIMEO, (void *)&timeout, sizeof(timeout));
	HPINET_CheckSockError(status, TEXT("setsockopt SO_RCVTIMEO"), phr);

	// Write the packet
	HPI_DEBUG_LOG0(DEBUG,TEXT("HPINET_MessageDiscovery:Send msg using sendto()\n"));
	nSent = sendto( sock,				// SOCKET s
		        (char *)szBuffer,		// const char* buf
				phm->wSize+sizeof(struct asi_pkt),				// int length
				0,					// int flags
				(SOCKADDR *) &destAdr,			// const struct sockaddr* to socket address, seems weird to typecast it, but MS does exactly this in their example.
				(int)sizeof(destAdr));	// int total length
	if(HPINET_CheckSockError(nSent, TEXT("sendto"),phr))
		return -1;

	while(1)
	{
		int serr;
		struct hpi_adapter * pA;

		nFromLen = sizeof(destAdr);
		nSize = recvfrom(sock,					// SOCKET s
					(char *)szBuffer,			// char * buffer
					sizeof(szBuffer),			// int length
					0,							// int flags
					(SOCKADDR *)&destAdr,		// struct sockaddr* will hold source addr upon return
					&nFromLen);					// int * in,out size in byes of destAdr.
		HPI_DEBUG_LOG0(DEBUG,TEXT("HPINET_MessageDiscovery:Get msg using recvfrom()\n"));

		// exit the while loop after we get the first timeout.
		serr = WSAGetLastError();
		if(serr==WSAETIMEDOUT)
		{
			HPI_DEBUG_LOG0(DEBUG,TEXT("HPINET_MessageDiscovery: TIMEOUT\n"));
			break;
		}

		if(HPINET_CheckSockError(nSize, TEXT("recfrom"), phr))
			return -1;

		if(HPINET_checkReceivePacket(szBuffer, phr))
			return -1;

		if((phr->u.a.wAdapterIndex==0)&&(phr->u.a.wAdapterType==0))
		{
			HPI_DEBUG_LOG2(INFO,TEXT("HPINET_MessageDiscovery ERROR: Found adapter %d with address %x\n"),phr->u.a.wAdapterIndex,destAdr.sin_addr.s_addr);
			continue;
		}

		pA = malloc( sizeof(struct hpi_adapter) );
		if(first_adapter==0)
			first_adapter = pA;
		else
		{
			struct hpi_adapter * pAlast;

			// find the last adapter in the list.
			pAlast = first_adapter;
			while(pAlast->next){pAlast = pAlast->next;}
			// add pA to the list
			pAlast->next = pA;
		}

		memset(pA,0,sizeof(struct hpi_adapter));
		HPI_DEBUG_LOG2(INFO,TEXT("HPINET_MessageDiscovery: Found adapter %d with address %x\n"),phr->u.a.wAdapterIndex,destAdr.sin_addr.s_addr);
		pA->adapterIndex=phr->u.a.wAdapterIndex;
		pA->adapterType=phr->u.a.wAdapterType;
		pA->addr.s_addr = destAdr.sin_addr.s_addr;
		pA->adapterIsOpen=0;
		pA->hpi_message=HPI_NET;
		pA->bhr=*phr;
		phr->wError=0;
		num_adapters++;
	}

	closesocket(sock);

	if(num_adapters==0)
		return -1;

	return 0;
}

/*=================================================================*/
/** Send an HPI message via UDP and get a response back from the ASI2416.
  *
  */
int HPINET_Message(
		struct hpi_adapter *pA, ///< Adapter structure
		HPI_MESSAGE *phm,		///< Message to send
		HPI_RESPONSE *phr		///< array of responses received
		)
{
	int nSent,nSize;
	char szBuffer[HPINET_ASI_DATA_SIZE+2];	// +2 to account for asi_pkt space.
	struct sockaddr_in destAdr;

	//HPI_DEBUG_LOG0(DEBUG,TEXT("HPINET_Message\n"));

	HPINET_createTransmitPacket(szBuffer,phm);

	// Set up the destination address structure
	destAdr.sin_family = AF_INET;
	destAdr.sin_port = htons(HPI_ETHERNET_UDP_PORT);
	destAdr.sin_addr.s_addr = pA->addr.s_addr;

#ifdef UDP_CACHING
	// If this is the first message and the control cache has not been setup, we do it here.
	if(pA->cache==NULL)
	{
		HPI_MESSAGE hm;
		HPI_RESPONSEX hr;
		int nRet;
		int i=0;

		// alloc a control cache
		pA->cache = malloc( sizeof(struct hpi_adapter_context));
		memset(pA->cache, 0, sizeof(struct hpi_adapter_context) );
		// Now fill controls.
		// 1. Call MixerOpen()
		HPI_InitMessage( &hm,  HPI_OBJ_MIXER, HPI_MIXER_OPEN );
	    hm.wAdapterIndex = pA->adapterIndex;
		nRet = HPINET_Message(pA,&hm,(HPI_RESPONSE *)&hr);
		if(nRet!=0)
			return nRet;
		// 2. Call MixerGetControlArrayByIndex() to load all the available controls.
		while(1)
		{
			HPI_InitMessage( &hm,  HPI_OBJ_MIXER, HPI_MIXER_GET_CONTROL_ARRAY_BY_INDEX );
		    hm.wAdapterIndex = pA->adapterIndex;
			hm.u.mx.gcabi.wStartingIndex = (HW16)pA->cache->nControls;
			hm.u.mx.gcabi.dwLengthInBytes = HPINET_RESP_SIZEOF_DATA;
			nRet = HPINET_Message(pA,&hm,(HPI_RESPONSE *)&hr);
			if(nRet!=0)
				return nRet;
			if(hr.resp.wError==0)
			{

				memcpy(&pA->cache->controlDef[pA->cache->nControls], &(((unsigned char *)&hr)[sizeof(HPI_RESPONSE)]), hr.resp.u.mx.gcabi.dwBytesReturned );
    			pA->cache->nControls += (HW16)(hr.resp.u.mx.gcabi.dwBytesReturned / sizeof(HPI_CONTROL_DEFN));
				HPI_DEBUG_LOG1(INFO,TEXT("MixerGetControlArrayByIndex() total %d controls\n"), pA->cache->nControls);
				if(hr.resp.u.mx.gcabi.dwBytesReturned/sizeof(HPI_CONTROL_DEFN) <
					HPINET_RESP_SIZEOF_DATA/sizeof(HPI_CONTROL_DEFN) )
					break;
			}
			else
				break;
		}
		// 3. Load the control cache for the first time and timestamp it.
		// First reset the cache.
		HPI_InitMessage( &hm,  HPI_OBJ_MIXER, HPI_MIXER_GET_CONTROL_MULTIPLE_VALUES );
		hm.wAdapterIndex = pA->adapterIndex;
		hm.u.mx.gcabi.wStartingIndex = (HW16)i;
		hm.u.mx.gcabi.dwLengthInBytes = HPINET_RESP_SIZEOF_DATA;
		hm.u.mx.gcabi.wFlags = HPI_MIXER_GET_CONTROL_MULTIPLE_RESET;
		nRet = HPINET_Message(pA,&hm,(HPI_RESPONSE *)&hr);
		nRet=UpdateControlCache(pA);
		if(nRet!=0)
			return nRet;
	}

	if((phm->wFunction==HPI_MIXER_OPEN)&&pA->mixerIsOpen)
	{
        HPI_InitResponse(phr,HPI_OBJ_MIXER,HPI_MIXER_OPEN,0);
		return 0;
	}

	if((phm->wFunction==HPI_MIXER_GET_CONTROL_BY_INDEX)&&(pA->cache!=NULL))
	{
		unsigned int i=phm->u.m.wControlIndex;
        HPI_InitResponse(phr,HPI_OBJ_MIXER,HPI_MIXER_GET_CONTROL_BY_INDEX,0);
		if(i<pA->cache->nControls)
		{
			if(pA->cache->controlDef[i].wSrcNodeType>=HPI_SOURCENODE_BASE)
				pA->cache->controlDef[i].wSrcNodeType -= HPI_SOURCENODE_BASE;
			if(pA->cache->controlDef[i].wDestNodeType>=HPI_DESTNODE_BASE)
				pA->cache->controlDef[i].wDestNodeType -= HPI_DESTNODE_BASE;
			phr->u.m.wSrcNodeType = pA->cache->controlDef[i].wSrcNodeType;
			phr->u.m.wSrcNodeIndex = pA->cache->controlDef[i].wSrcNodeIndex;
			phr->u.m.wDstNodeType = pA->cache->controlDef[i].wDestNodeType;
			phr->u.m.wDstNodeIndex = pA->cache->controlDef[i].wDestNodeIndex;
			phr->u.m.wControlIndex = pA->cache->controlDef[i].wType; // actually type !
			//HPI_DEBUG_LOG1(INFO,TEXT("HPI_MIXER_GET_CONTROL_BY_INDEX cached control # %d\n"), i);
		}
		else
			phr->wError = HPI_ERROR_INVALID_OBJ_INDEX;
		return 0;
	}
	// handle HPI_ControlQuery() for volume and meter channels
	if((phm->wFunction==HPI_CONTROL_GET_INFO)&&(pA->cache!=NULL))
	{
		unsigned int i=phm->u.c.wControlIndex;
        HPI_InitResponse(phr,HPI_OBJ_CONTROL,HPI_CONTROL_GET_INFO,0);
		if(i<pA->cache->nControls)
		{
			if(
				((pA->cache->controlDef[i].wType==HPI_CONTROL_METER)&&(phm->u.c.wAttribute==HPI_METER_NUM_CHANNELS))
				||
				((pA->cache->controlDef[i].wType==HPI_CONTROL_VOLUME)&&(phm->u.c.wAttribute==HPI_VOLUME_NUM_CHANNELS))
				)
			{
				unsigned int nChannels = pA->cache->controlDef[i].wChannels;
				if(nChannels==0)
					nChannels=2;
				phr->u.c.dwParam1 = nChannels;
				return 0;
			}
		}
	}

	if((phm->wFunction==HPI_CONTROL_GET_STATE)&&(pA->cache!=NULL))
	{
		UpdateControlCache(pA);
		if (CheckControlCache(&pA->cache->controlCache[phm->u.c.wControlIndex],phm, phr)==1)
			return 0;
	}
	// the following for debug only
	if((phm->wFunction==HPI_CONTROL_GET_INFO)&&(pA->cache!=NULL))
	{
		HPI_DEBUG_LOG3(INFO,TEXT("HPI_MIXER_GET_CONTROL_BY_INDEX control type %d, Attrib %d, index %d\n"),
			pA->cache->controlDef[phm->u.c.wControlIndex].wType,
			phm->u.c.wAttribute,
			phm->u.c.dwParam1);
	}

#endif

	HPI_DEBUG_MESSAGE(phm);

	// Write the packet
	nSent = sendto( pA->sock,				// SOCKET s
		        (char *)szBuffer,		// const char* buf
				phm->wSize+sizeof(struct asi_pkt),				// int length
				0,					// int flags
				(SOCKADDR *) &destAdr,			// const struct sockaddr* to socket address, seems weird to typecast it, but MS does exactly this in their example.
				(int)sizeof(destAdr));	// int total length
	if(HPINET_CheckSockError(nSent, TEXT("sendto"), phr))
		return -1;

	nSize = recvfrom(pA->sock,					// SOCKET s
				(char *)szBuffer,			// char * buffer
				sizeof(szBuffer),			// int length
				0,							// int flags
				NULL,		// struct sockaddr* will hold source addr upon return
				NULL);					// int * in,out size in byes of destAdr.

	if(HPINET_CheckSockError(nSize, TEXT("recvfrom"), phr))
		return -1;

	HPINET_checkReceivePacket(szBuffer, phr);

	if(phm->wFunction==HPI_MIXER_OPEN)
		pA->mixerIsOpen=1;

	return 0;
}

#ifdef UDP_CACHING
/*=================================================================*/
/** Update control cache
  *
  */
static int UpdateControlCache(struct hpi_adapter *pA)
{
	HPI_MESSAGE		hm;
	HPI_RESPONSEX	hr;
	unsigned int 	nMoreToDo=1;
	int				i=0;
	int				nRet=0;
	unsigned int	nCurrent_ms;

	nCurrent_ms =  GetTickCount();

	// if time delta is < 50ms don't do anything.
	if((pA->cache->timeStamp_ms!=0)&&(nCurrent_ms-pA->cache->timeStamp_ms < 50))
		nMoreToDo = 0;
	// if we are going to update cache, update the timestamp
	if(nMoreToDo)
		pA->cache->timeStamp_ms = nCurrent_ms;


	while(nMoreToDo)
	{
		HPI_InitMessage( &hm,  HPI_OBJ_MIXER, HPI_MIXER_GET_CONTROL_MULTIPLE_VALUES );
		hm.wAdapterIndex = pA->adapterIndex;
		hm.u.mx.gcabi.wStartingIndex = (HW16)i;
		hm.u.mx.gcabi.dwLengthInBytes = HPINET_RESP_SIZEOF_DATA;
		hm.u.mx.gcabi.wFlags = HPI_MIXER_GET_CONTROL_MULTIPLE_CHANGED;
		nRet = HPINET_Message(pA,&hm,(HPI_RESPONSE *)&hr);
		if(nRet!=0)
			return nRet;
		if(hr.resp.wError==0)
		{
			int j;
			tHPIControlCacheSingle *c = (tHPIControlCacheSingle *)&( ((HPI_RESPONSEX *)&hr)->data[0] ) ;
			int n = (int)hr.resp.u.mx.gcabi.dwBytesReturned/sizeof(tHPIControlCacheSingle);
			HPI_DEBUG_LOG1(INFO,TEXT("Control cache update %d controls\n"), n);
			for(j=0;j<n;j++)
				pA->cache->controlCache[c[j].ControlIndex] = c[j];
			nMoreToDo = hr.resp.u.mx.gcabi.wMoreToDo;
			i += n;
		}
		else
			break;
	}
	return nRet;
}
#endif

/*=================================================================*/
/** Basic message processing function.
  *
  */
void HPI_NET(HPI_MESSAGE *phm, HPI_RESPONSE *phr )
{
	struct hpi_adapter *pA;

	pA=findAdapter(phm->wAdapterIndex);

	if(pA) {
		//HPI_DEBUG_LOG1(DEBUG,TEXT("Adapter %d found\n"), phm->wAdapterIndex);
		HPINET_Message(pA,phm,phr);
	} else
		HPI_DEBUG_LOG1(WARNING,TEXT("Adapter %d not found\n"), phm->wAdapterIndex);

}
/*=================================================================*/
/** Send extended HPI message.
  *
  * Extended message may include an appended data element.
  * and/or have a non-default timeout.
  *
  * phr->wSize must be set before calling this function, to indicate the actual
  * size available for the returned response.
  * If it is  zero, response size defaults to sizeof(HPI_RESPONSE)
  */
void HPI_MessageEx(HPI_MESSAGEX *phm, HPI_RESPONSEX *phr ,
		unsigned int timeout)
{
	//HPI_DEBUG_LOG0(DEBUG,TEXT("HPI_MessageEx\n"));
	HPI_NET((HPI_MESSAGE *)phm,(HPI_RESPONSE *)phr);
}
/*=================================================================*/
/** Main entry point into this module.
  */
#ifdef HPI_MULTIINTERFACE
void HPI_MessageUdp(HPI_MESSAGE *phm, HPI_RESPONSE *phr )
#else
void HPI_Message(HPI_MESSAGE *phm, HPI_RESPONSE *phr )
#endif
{
    phr->wSize=0;

	HPI_DEBUG_MESSAGE(phm);

    // check for various global messages
    if(phm->wObject == HPI_OBJ_SUBSYSTEM)
    {
        // subsys message get sent to more than one hpi, may not be filled in by any of them
        // so need to create a default response here
		// HPI_SUBSYS_CREATE_ADAPTER is a special case where the response from previous hpi
		// calls is passed in to facilitate duplicate adapter index detection, so don't init
		if( phm->wObject == HPI_OBJ_SUBSYSTEM && phm->wFunction == HPI_SUBSYS_CREATE_ADAPTER)
			phr->wSize = HPI_RESPONSE_FIXED_SIZE + sizeof(HPI_SUBSYS_RES);
		else
			HPI_InitResponse(phr, phm->wObject, phm->wFunction, HPI_ERROR_INVALID_OBJ );

        switch(phm->wFunction)
        {
            // subsys open - init the debug function
        case HPI_SUBSYS_OPEN:
			phr->wError=SubsysOpen();
            return;

        case HPI_SUBSYS_FIND_ADAPTERS:
            SubsysFindAdapters(phm, phr);
            return;

        case HPI_SUBSYS_CLOSE:
			phr->wError=SubsysClose();
            return;

            // version message - if so then leave
        case HPI_SUBSYS_GET_VERSION:
			HPI_InitResponse( phr, HPI_OBJ_SUBSYSTEM, HPI_SUBSYS_GET_VERSION, 0 );
			phr->u.s.dwVersion = HPI_VER>>8;	// return major.minor
			phr->u.s.dwData = HPI_VER;			// return major.minor.release
            return;

		case HPI_SUBSYS_GET_NUM_ADAPTERS:
            HPI_InitResponse(phr,HPI_OBJ_SUBSYSTEM,HPI_SUBSYS_GET_NUM_ADAPTERS,0);
			phr->u.s.wNumAdapters = (HW16)SubsysGetNumAdapters();
			return;

		case HPI_SUBSYS_GET_ADAPTER:
            HPI_InitResponse(phr,HPI_OBJ_SUBSYSTEM,HPI_SUBSYS_GET_ADAPTER,0);
			if(!SubsysGetAdapter(phm->wAdapterIndex,&phr->u.s.wAdapterIndex,&phr->u.s.awAdapterList[0]))
				phr->wError = HPI_ERROR_INVALID_OBJ_INDEX;
			return;

		default:
            break;
        }
    }
	else if (phm->wObject == HPI_OBJ_ADAPTER)
	{
        switch(phm->wFunction)
		{
        case HPI_ADAPTER_OPEN:
			AdapterOpen(phm, phr);
            return;

        case HPI_ADAPTER_CLOSE:
			AdapterClose(phm, phr);
            return;

		default:
            break;
        }
	}

	HPI_NET(phm,phr);

    if (phr->wSize==0)
    {	// This should not happen. If it does this is a coding error.
        // However, still fill in enough of the response to return an error to help debugging.
		phr->wError = HPI_ERROR_PROCESSING_MESSAGE;
        phr->wSize = HPI_RESPONSE_FIXED_SIZE;
    }

}
/*=================================================================*/
/** Back door method for setting an IP address. How do we do this the
correct way through HPI?.
  */
void HPIUDP_IP_Set(const char *szIP2Use)
{
	strncpy(szIP,szIP2Use,sizeof(szIP));
}
/*=================================================================*/
/** Local version of subsys open just initializes the winsock2 interface.
  */
HW16 SubsysOpen(void )
{
	int err;
#ifdef HPI_OS_WIN32_USER
	WSADATA wsaData;
	int iResult=0;
	DWORD dwTickCountMs = GetTickCount();
#endif

#ifdef HPI_OS_WIN32_USER

	// don't do anything for the first 10 seconds.
	HPI_DEBUG_LOG1(DEBUG,TEXT("GetTickCount(): %d\n"),GetTickCount());
	if(dwTickCountMs<10000)
	{
		HPI_DEBUG_LOG0(DEBUG,TEXT("dwTickCountMs<10000\n"));
		return HPI_ERROR_INVALID_OBJ;
	}
	// Initialize Winsock under windows.
	iResult = WSAStartup(MAKEWORD(2,2), &wsaData);
	if (iResult != 0)
	{
		HPI_DEBUG_LOG1(ERROR, TEXT("WSAStartup failed: %d\n"),iResult);
		return HPI_ERROR_INVALID_OBJ;
	}
	else
		gnSocketCleanupRequired = 1;
#endif


	// find all the adapters out there....
	err= HPINET_MessageDiscovery();

	return (HW16)err;
}
/*=================================================================*/
/** Close the subsystem just cleans up open adpaters and terminates winsock2.
*/
HW16 SubsysClose(void)
{
	struct hpi_adapter *pA=first_adapter;
	struct hpi_adapter *next;

	while(pA)
	{
		next = pA->next;

		if(pA->adapterIsOpen)
		{
			/* close the socket */
		}
#ifdef UDP_CACHING
		if(pA->cache)
			free(pA->cache);
#endif
                /* add close adapter here */
		free(pA);
		pA=next;
	}
	if(gnSocketCleanupRequired)
	{
		gnSocketCleanupRequired=0;
		WSACleanup();			// terminate use of WS2_32.DLL
	}
	return 0;
}

/*=================================================================*/
/** Return number of adapters.
*/
int SubsysGetNumAdapters(void)
{
	int nCount = 0;
	struct hpi_adapter *pA=first_adapter;

	while(pA)
	{
		++nCount;
		pA = pA->next;
	}
	return nCount;
}

/*=================================================================*/
/** Give an interator index, return an adapter index and type.
*/
int SubsysGetAdapter(int nIterIndex, HW16 *pwAdapterIndex, HW16 *pwAdapterType)
{
	int nCount = 0;
	int nFound = 0;

	struct hpi_adapter *pA=first_adapter;

	while(pA)
	{
		if(nCount==nIterIndex)
		{
			*pwAdapterIndex = (HW16)pA->adapterIndex;
			*pwAdapterType = (HW16)pA->adapterType;
			nFound = 1;
			break;
		}
		++nCount;
		pA = pA->next;
	}
	return nFound;
}

/*=================================================================*/
/** Function for printing a MAC address - unused ?

*/
void PrintMac(HPI_ETHERNET_MAC_ADR	mac)
{
	int i;
	for (i=0; i<5; i++)
		printf("%02X:",mac[i]);
	printf("%02X",mac[5]);
}


/*=================================================================*/
/** Finds all the adapters we konw about. For the moment they are
  * not ordered correctly.
  */
static void SubsysFindAdapters(HPI_MESSAGE *phm, HPI_RESPONSE *phr)
{
	int n=0;
	struct hpi_adapter *pA=first_adapter;

	HPI_InitResponse( phr, HPI_OBJ_SUBSYSTEM, HPI_OBJ_SUBSYSTEM, 0);
	while(pA)
	{
		phr->u.s.awAdapterList[n] = (HW16)pA->adapterType;
		n++;
		pA = pA->next;

	}
	phr->u.s.wNumAdapters = (HW16)n;
}

/*=================================================================*/
/** Open an adapter. This creates a socket for talking to the specific adapter.
  *
  */
static void AdapterOpen(HPI_MESSAGE *phm, HPI_RESPONSE *phr)
{
	struct hpi_adapter *pA;
#ifdef HPI_OS_LINUX
	struct timeval timeout = {
		.tv_sec = NET_TIMEOUT_SECONDS,
		.tv_usec = NET_TIMEOUT_MS * 1000,
	};
#else
	int timeout = NET_TIMEOUT_SECONDS * 1000 + NET_TIMEOUT_MS;
#endif

	pA=findAdapter(phm->wAdapterIndex);

	if(pA)
	{
		/* Create socket for sending/receiving datagrams */
		pA->sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
		if(pA->sock==INVALID_SOCKET)
		{
			HPI_DEBUG_LOG0(ERROR,TEXT("Error opening socket\n"));
			phr->wError = HPI_ERROR_NETWORK_TIMEOUT;
			return;
		}

		setsockopt(pA->sock, SOL_SOCKET, SO_RCVTIMEO, (void *)&timeout, sizeof(timeout));
		setsockopt(pA->sock, SOL_SOCKET, SO_SNDTIMEO, (void *)&timeout, sizeof(timeout));

		HPINET_Message(pA, phm, phr);
		pA->adapterIsOpen=1;

#if 0
//#ifdef UDP_CACHING
		{
			PSECURITY_DESCRIPTOR    pSD;
			SECURITY_ATTRIBUTES     sa;
			TCHAR szFilename[64];
			// Create cache for list of controls and control settings.
			/* security stuff copied from WINNT SDK */
			pSD = (PSECURITY_DESCRIPTOR) LocalAlloc( LPTR,
				SECURITY_DESCRIPTOR_MIN_LENGTH);
			if (pSD == NULL)
				return;
			if (!InitializeSecurityDescriptor(pSD, SECURITY_DESCRIPTOR_REVISION))
				return;
			// Add a NULL DACL to the security descriptor..
			if (!SetSecurityDescriptorDacl(pSD, TRUE, (PACL) NULL, FALSE))
			{
				LocalFree((HLOCAL)pSD);
				return;
			}
			sa.nLength = sizeof(sa);
			sa.lpSecurityDescriptor = pSD;
			sa.bInheritHandle = TRUE;

			_stprintf(szFilename,TEXT("AsiUdpMapping_%d"),pA->adapterIndex);
			pA->hFileMappedContext =
				CreateFileMapping(
					INVALID_HANDLE_VALUE,
					&sa,
					PAGE_READWRITE,
					0,
					sizeof(struct hpi_adapter_context),
					szFilename);
			if(pA->hFileMappedContext)
			{
				int nZero=0;

				// if CreateFileMapping() returned ERROR_SUCCESS, this is the first open that
				// has occurred, so we need to set all the memory to zero.
				if( GetLastError()==ERROR_SUCCESS )
					nZero=1;

				pA->cache = MapViewOfFile(pA->hFileMappedContext, FILE_MAP_ALL_ACCESS,0,0,0);
				if(nZero)
					memset(pA->cache, 0, sizeof(struct hpi_adapter_context) );
			}
		LocalFree((HLOCAL)pSD);
		}

#endif

	} else {
		phr->wError = HPI_ERROR_INVALID_OBJ_INDEX;
		phr->wSize = HPI_RESPONSE_FIXED_SIZE;
	}
}

/*=================================================================*/
/** Close an adapter.
  *
  */
static void AdapterClose(HPI_MESSAGE *phm, HPI_RESPONSE *phr)
{
	struct hpi_adapter *pA;

	pA=findAdapter(phm->wAdapterIndex);
	if(pA)
	{
		HPINET_Message(pA, phm, phr);
		closesocket(pA->sock);
		pA->adapterIsOpen=0;
	}
}

/*=================================================================*/
/** Given an index, this function returns a pointer to the adapter structure.
  *
  */
struct hpi_adapter *findAdapter(const HW16 wAdapterIndex)
{
	struct hpi_adapter *a;

	if(first_adapter==0)
		return 0;

	a=first_adapter;
	while(a)
	{
		if(a->adapterIndex == wAdapterIndex)
			break;
		a = a->next;
	}

	return a;
}

/*=================================================================*/
/** DriverOpen
  *
  */
#ifdef HPI_MULTIINTERFACE
HW16 HPI_DriverOpenUdp(void)
#else
HW16 HPI_DriverOpen(void)
#endif
{
	//HPI_DebugLevelSet(HPI_DEBUG_LEVEL_VERBOSE);
	return 1;
	//return 0;
}

/*=================================================================*/
/** DriverClose
  *
  */
#ifdef HPI_MULTIINTERFACE
void HPI_DriverCloseUdp(void)
#else
void HPI_DriverClose(void)
#endif
{

}
/*=================================================================*/
/** A utility function to put ASI packet specifc information at the
  * front of the phm message and return the whole lot in szData ready
  * for sending.
  * \return Returns 0 on succes, otherwise an error.
  */
void HPINET_createTransmitPacket(
			char *szData,		///< The passed in data buffer is filled with correct packet information on return.
			HPI_MESSAGE *phm	///< The HPI message structure that is to be transmitted.
			)
{
	struct asi_pkt *asi_info;

	// create the buffer to send
	asi_info = (struct asi_pkt *)&szData[0];
	asi_info->ASIpktID = HPI_ETHERNET_PACKET_ID;
	asi_info->ASIpktVersion = HPI_ETHERNET_PACKET_HOSTED_VIA_HMI_V1;
	memcpy(&szData[2], phm, phm->wSize);			// copy the message across
}

/*=================================================================*/
/** A utility function to check a received UDP packet to make
  * sure it came from an ASI client. It also copies the received data into
  * a response structure.
  * \return Returns 0 on succes, otherwise an error.
  */
int HPINET_checkReceivePacket(
				char *szData,		///< The raw data received.
				HPI_RESPONSE *phr	///< The response structure that will contain the response on return.
				)
{
	struct asi_pkt *asi_info;
	HPI_RESPONSE *phr_local = (HPI_RESPONSE *)&szData[2];

	if (phr_local->wSize > sizeof(HPI_RESPONSEX) ) {
			phr->wError= HPI_ERROR_INVALID_DATASIZE;
			phr->wSize = HPI_RESPONSE_FIXED_SIZE;
			return -1;
	}

	// check the return packet type
	// create the buffer to send
	asi_info = (struct asi_pkt *)&szData[0];
	if( (asi_info->ASIpktID != HPI_ETHERNET_PACKET_ID) || (asi_info->ASIpktVersion != HPI_ETHERNET_PACKET_HOSTED_VIA_HMI_V1))
	{
		phr->wError = HPI_ERROR_NETWORK_TIMEOUT;
		return -1;
	}
	memcpy(phr,&szData[2], phr_local->wSize);			// copy the response across
	if(phr->wError)
	{
		HPI_DEBUG_LOG1(DEBUG,TEXT("HPI Error %d\n"),phr->wError);
	}
	return 0;
}
/*=================================================================*/
/** Checks for a socket error code and logs a message.
  * \return Returns 0 on succes, otherwise an error.
  */
int HPINET_CheckSockError(const int nRet,		///< status return from the winsock2 function
						  const TCHAR *szFn,     ///< string to be embedded in the error output. Typically something like "sndto", ie the function that generated the error.
						  HPI_RESPONSE *phr		///< the HPI response structure
						  )
{
#ifdef HPI_OS_LINUX
	if(nRet==SOCKET_ERROR)
	{
		int serr = errno;
		char * errstr = strerror(serr);

		HPI_DEBUG_LOG2(DEBUG,TEXT("Error, %s() gave %s.\n"),szFn,errstr);
		phr->wError = HPI_ERROR_NETWORK_TIMEOUT;
		HPI_DEBUG_LOG1(DEBUG,TEXT("HPI Error %d\n"),phr->wError);
		return -1;
	}
#else
	if(nRet==SOCKET_ERROR)
	{
		int serr = WSAGetLastError();

		if(serr==WSAETIMEDOUT)
			HPI_DEBUG_LOG1(DEBUG,TEXT("WinSock error, %s() gave WSAETIMEDOUT > TIMEOUT."),szFn);
		else if (serr==WSAENOTSOCK)
			HPI_DEBUG_LOG1(DEBUG,TEXT("WinSock error, %s() gave WSAENOTSOCK > NOT A SOCKET."),szFn);
		else
			HPI_DEBUG_LOG2(DEBUG,TEXT("WinSock error, %s() gave %d"),szFn,serr);
		phr->wError = HPI_ERROR_NETWORK_TIMEOUT;
		HPI_DEBUG_LOG1(DEBUG,TEXT(" HPI Error %d\n"),phr->wError);
		return -1;
	}
#endif
	return 0;
}
#endif // HPI_INCLUDE_UDP

// $Log: hpiudp.c,v $
// Revision 1.28  2007/09/11 18:18:29  as-age
// Add channels query.
//
// Revision 1.27  2007/09/07 09:40:08  as-ewb
// linux bring net timeout back to 10sec
//
// Revision 1.26  2007/08/30 01:10:09  as-ewb
// add inet.h for inet_addr function
//
// Revision 1.25  2007/08/28 15:06:56  as-age
// Turn on UDP caching. Fix memory leak.
//
// Revision 1.24  2007/08/27 18:16:54  as-age
// Allow  user to select host adapter. Needs more testing and refinement.. Also bind() seems to cause a
// rouge response.
//
// Revision 1.23  2007/08/22 14:10:23  as-age
// Turn off caching again for the moment.
//
// Revision 1.22  2007/08/14 15:14:20  as-age
// Turn on UDP control caching.
//
// Revision 1.21  2007/08/13 20:17:00  as-age
// Tweak to get it to compile.
//
// Revision 1.20  2007/08/13 18:28:13  as-age
// fixes for asihpi32 compilation
//
// Revision 1.19  2007/08/13 15:39:45  as-age
// Control caching seems to work ok, just need to test it.
//
// Revision 1.18  2007/06/20 23:53:10  as-ewb
// add timeout to adapter socket.  AdapterOpen return error if not found.  Change levels on some log messages. Default log level now INFO
//
// Revision 1.17  2007/06/20 15:18:36  as-age
// Correct a couple more "\n"'s.
//
// Revision 1.16  2007/06/19 17:12:10  as-age
// Get it to compile under Windows again ! Hopefully Linux is still ok ?
//
// Revision 1.15  2007/06/19 00:44:39  as-ewb
// make it work on Linux. Add CR to debug texts
//
// Revision 1.14  2007/06/18 14:56:29  as-age
// Use hpidebug.c fns for debugging.
//
// Revision 1.13  2007/06/15 20:38:57  as-age
// Support compile with debug off.
//
// Revision 1.12  2007/06/15 20:23:02  as-age
// A way point check in. This version seems to run, but still need to pull out older debug stuff.
//
// Revision 1.11  2007/06/07 13:58:14  as-age
// File is skipped unless HPI_INCLUDE_UDP is defined.
//
// Revision 1.10  2007/06/06 17:27:25  as-age
// Update error handling so that it compiles using sources file in asihpi32 dir.
//
// Revision 1.9  2007/05/17 21:06:56  as-age
// Fix SubsysGetAdapter() so that ASI2416 indexes are used correctly. These are the
// "big" indexes, not 0,1,2 anymore.
//
// Revision 1.8  2007/04/12 14:42:01  as-tfe
// Add a send timeout to the UDP socket.
//
// Revision 1.7  2007/03/29 13:13:06  as-tfe
// Added messages HPI_SUBSYS_GET_NUM_ADAPTERS and
// HPI_SUBSYS_GET_ADAPTER.  Network adapters now use index returned
// from HPI_ADAPTER_GET_INFO.
//
// Revision 1.6  2007/03/27 21:01:44  as-tfe
// Added code to multiplex between hpioct32 and hpiudp for hybrid build of asictrl.
//
// Revision 1.5  2007/02/08 17:30:54  as-age
// Improve HPI error logging.
//
// Revision 1.4  2007/02/06 19:33:32  as-age
// Rewrite discovery to support multiple ASI2416s. Works in NY for 2 ASI2416s.
//
// Revision 1.3  2007/01/31 19:14:07  as-age
// Major cleanup. This version works with a single adapter. Need to add code for multiple
// adapters. Don't know what changes Firmdown might require to work with this.
//
// Revision 1.2  2007/01/30 20:10:22  as-age
// This version is hardcoded to a specific IP address, but at least it comes up ok.
//
// Revision 1.1  2007/01/23 15:58:37  as-age
// First check in. This code does not currently work.
//
