/**************************************************************************
$Header: /Repository/drv/hpi/hpirds.c,v 1.21 2007/06/04 18:29:23 as-age Exp $

(c) AudioScience, Inc 2007

***************************************************************************/


#include <stdlib.h>
#include <string.h>

#include <hpirds.h>

/** 

\param fieldPI PI - Program identification. PI is a two byte number which identifies the country, 
coverage area and service. It can be used by the control microprocessor but is not normally 
intended for display. A change in PI code causes the initialisation of all RDS data as it 
indicates that the radio has been retuned. This application also facilitates the display 
of the current PI code.

\param fieldPTY PTY - Program type. PTY is a 5-bit number which indicates the current program 
type. At present 16 of these types are defined. Examples include "no programme type", "Current affairs" and 'Pop music",


*/
struct HPI_RDS_DATA_PRESERVE{
	enum eHPI_RDS_type eType;				///< Type of the bitstream to process.

	unsigned int  flagRTflag;			///< RT text A/B flag. 0=A, 1=B.
	unsigned int  flagRTcountfromzero[2];  ///< Indicates the current string has been built from index 0 - used to get rid of partial strings.
	unsigned int  uRTcount[2];				///< Count of received RT chars.
	unsigned int  uRTValidCount[2][64];		///< RT - Count of valid characters. Separate counts for A and B. A is index 0, B index 1. 
	unsigned int  uRTValidTreshold;		///< RT - Threshold for valid characters.
	char fieldRTbuild[2][64];	///< RT - Build Radio text. Separate build locations for A and B. A is index 0, B index 1. 

	char lastGroup[8];			///< Last block returned from hardware.
	char lastBLER[4];			///< Last block error rate returned from hardware.
};

struct  HPI_RDS_DATA{

	unsigned int  fieldGroup;			///< Number of the last group to be analyzed.
	char fieldGroupVersion;	///< Version (either 'A' or 'B) of the last group to be analyzed.
	unsigned int  flagPSready;			///< Indicates PS text ready.
	unsigned int  flagRTready;			///< Indicates RT text ready.
	unsigned int  uPScount;				///< Count of received PS chars.
	
	// RDS specific fields follow.
	unsigned short fieldPI;		///< PI - Program identification.
	unsigned char fieldPTY;		///< PTY - Program type
	char fieldPS[9];	///< PS - Program service name. The name of the station. Extra char for null termination.
	char fieldRT[65];	///< RT - Radio text - up to 64 characters. Extra char for null termination.
	char fieldPSbuild[8];	///< PS - Build Program service name.
								///< CT - clock time and date
								///< AF - alternative frequencies
	unsigned int flagTA;		///< TA - Traffic annoucement flag. Is set during traffic announcemnet.
	unsigned int flagTP;		///< TP - Traffic program flag. 1 indicates this transmitter carries traffic announcements.
	unsigned int flagMS;		///< MS - Music/speech switch. Single bit indicationg speech or music.
	unsigned int fieldDI;		///< DI - decoder identification. 4 bits indicating mono,stereo, binaural).
	unsigned int fieldPIN[2];	///< PIN - program item number.
	///< EON - enhanced other networks
	///< TDC - transparent data channel
	///< INH - In-house data

	struct HPI_RDS_DATA_PRESERVE d;
};

static char *szERROR = "HANDLE ERROR";

#define MAX_PTY 32
static char *szRDS_PTY_RDS[2][MAX_PTY] = {{
"No program type or undefined",
"News",
"Information",
"Sports",
"Talk",
"Rock",
"Classic Rock",
"Adult Hits",
"Soft Rock",
"Top 40",
"Country",
"Oldies",
"Soft",
"Nostalgia",
"Classical",
"Rhythm and Blues",
"Soft Rhythm and Blues",
"Language",
"Religious Music",
"Religious Talk",
"Personality",
"Public",
"College",
"Unassigned",
"Unassigned",
"Unassigned",
"Unassigned",
"Unassigned",
"Weather",
"Emergency Test",
"Emergency"
},
{"No program type or undefined",
"News",
"Current affairs",
"Information",
"Sport",
"Education",
"Drama",
"Culture",
"Science",
"Varied",
"Pop Music",
"Rock Music",
"M.O.R. Music",
"Light classical",
"Serious classical",
"Other Music",
"Weather",
"Finance",
"Children's programmes",
"Social Affairs",
"Religion",
"Phone In",
"Travel",
"Leisure",
"Jazz Music",
"Country Music",
"National Music",
"Oldies Music",
"Folk Music",
"Documentary",
"Alarm Test",
"Alarm"
}};



static void resetRDS(struct HPI_RDS_DATA *pD);
static void processBlock1(struct HPI_RDS_DATA *pD);
static void processBlock2(struct HPI_RDS_DATA *pD);
static void processGroup0A(struct HPI_RDS_DATA *pD);
static void processGroup0B(struct HPI_RDS_DATA *pD);
static void processGroup2A(struct HPI_RDS_DATA *pD);

/*--------------------------------------------------------------------------------------*/
HPI_RDS_HANDLE HPI_API HPI_RDS_Create(enum eHPI_RDS_type eType)
{
	struct HPI_RDS_DATA *h=malloc(sizeof(struct HPI_RDS_DATA));

	resetRDS(h);
	h->d.eType = eType;

	h->d.uRTValidTreshold = 1;	// default RT threshold for valid characters.

	return (HPI_RDS_HANDLE)h;
}

/*--------------------------------------------------------------------------------------*/
void HPI_API HPI_RDS_Delete(
				   HPI_RDS_HANDLE h	
				   )
{
	if(h!=NULL)
		free(h);
}
/*--------------------------------------------------------------------------------------*/
void HPI_API HPI_RDS_Clear(
				   HPI_RDS_HANDLE h	
				   )
{
	struct  HPI_RDS_DATA *pD=(struct  HPI_RDS_DATA *)h;
	enum eHPI_RDS_type eType = pD->d.eType;
	unsigned int uThreshold = pD->d.uRTValidTreshold;
	resetRDS(h);
	pD->d.eType = eType;
	pD->d.uRTValidTreshold = uThreshold;

}
/*--------------------------------------------------------------------------------------*/
enum eHPI_RDS_errors HPI_API HPI_RDS_AnalyzeGroup(
			HPI_RDS_HANDLE h,
			const char *pData,
			const unsigned int nSize
			)
{
	struct  HPI_RDS_DATA *pD=(struct  HPI_RDS_DATA *)h;

	if(h==NULL)
		return HPI_ERROR_INVALID_OBJ;
	if(nSize!=12)
		return HPI_RDS_ERROR_INVALID_DATASIZE;

	// copy the blcok into the reference section
	memcpy(pD->d.lastGroup,pData,8);
	memcpy(pD->d.lastBLER,&pData[8],4);
	if( (pD->d.lastBLER[0] > 0) || 
            (pD->d.lastBLER[1] > 0) )
		return HPI_RDS_ERROR_BLOCK_DATA;
	processBlock1(pD);
	processBlock2(pD);
	if( (pD->d.lastBLER[2] > 2) || 
            (pD->d.lastBLER[3] > 2) )
		return HPI_RDS_ERROR_BLOCK_DATA;
	if(      (pD->fieldGroup==0) && pD->fieldGroupVersion=='A')	// group 0A
		processGroup0A(pD);
	else if( (pD->fieldGroup==0) && pD->fieldGroupVersion=='B')	// group 0B
		processGroup0B(pD);
	else if( (pD->fieldGroup==2) && pD->fieldGroupVersion=='A')	// group 2A
		processGroup2A(pD);
	else
		return HPI_RDS_ERROR_UNKNOWN_GROUP;

	return HPI_RDS_ERROR_NOERROR;
}
/*--------------------------------------------------------------------------------------*/
enum eHPI_RDS_errors HPI_API HPI_RDS_Get_BlockErrorCounts(
			HPI_RDS_HANDLE h,
			unsigned int *uBlock0,
			unsigned int *uBlock1,
			unsigned int *uBlock2,
			unsigned int *uBlock3
			)
{
	struct  HPI_RDS_DATA *pD=(struct  HPI_RDS_DATA *)h;

	if(h==NULL)
		return HPI_ERROR_INVALID_OBJ;
	if(uBlock0)
		*uBlock0 = (unsigned char)pD->d.lastBLER[0];
	if(uBlock1)
		*uBlock1 = (unsigned char)pD->d.lastBLER[1];
	if(uBlock2)
		*uBlock2 = (unsigned char)pD->d.lastBLER[2];
	if(uBlock3)
		*uBlock3 = (unsigned char)pD->d.lastBLER[3];

	return 0;
}

/*--------------------------------------------------------------------------------------*/
unsigned int  HPI_API HPI_RDS_Get_GroupType(HPI_RDS_HANDLE h)
{
	struct  HPI_RDS_DATA *pD=(struct  HPI_RDS_DATA *)h;
	if(h==NULL)
		return HPI_ERROR_INVALID_OBJ;
	return pD->fieldGroup;
}
/*--------------------------------------------------------------------------------------*/
char HPI_API HPI_RDS_Get_GroupVersion(HPI_RDS_HANDLE h)
{
	struct  HPI_RDS_DATA *pD=(struct  HPI_RDS_DATA *)h;
	if(h==NULL)
		return HPI_ERROR_INVALID_OBJ;
	return pD->fieldGroupVersion;
}
/*--------------------------------------------------------------------------------------*/
unsigned int HPI_API HPI_RDS_Get_PS_Ready(
					HPI_RDS_HANDLE h		///< A previously allocated HPIRDS handle.
					)
{
	struct  HPI_RDS_DATA *pD=(struct  HPI_RDS_DATA *)h;
	if(h==NULL)
		return HPI_ERROR_INVALID_OBJ;
	return pD->flagPSready;
}
/*--------------------------------------------------------------------------------------*/
HPI_RDS_STRING HPI_API HPI_RDS_Get_PS(HPI_RDS_HANDLE h)
{
	struct  HPI_RDS_DATA *pD=(struct  HPI_RDS_DATA *)h;
	if(h==NULL)
		return szERROR;
	pD->flagPSready = 0;
	return pD->fieldPS;
}
/*--------------------------------------------------------------------------------------*/
unsigned int HPI_API HPI_RDS_Get_RT_Ready(
					HPI_RDS_HANDLE h		///< A previously allocated HPIRDS handle.
					)
{
	struct  HPI_RDS_DATA *pD=(struct  HPI_RDS_DATA *)h;
	if(h==NULL)
		return HPI_ERROR_INVALID_OBJ;
	return pD->flagRTready;
}
/*--------------------------------------------------------------------------------------*/
void HPI_API HPI_RDS_Set_RT_Threshold(
					HPI_RDS_HANDLE h,
					unsigned int nCount
					)
{
	struct  HPI_RDS_DATA *pD=(struct  HPI_RDS_DATA *)h;
	pD->d.uRTValidTreshold = nCount;
}

/*--------------------------------------------------------------------------------------*/
HPI_RDS_STRING HPI_API HPI_RDS_Get_RT(
					HPI_RDS_HANDLE h		///< A previously allocated HPIRDS handle.
					)
{
	struct  HPI_RDS_DATA *pD=(struct  HPI_RDS_DATA *)h;
	if(h==NULL)
		return szERROR;
	pD->flagRTready = 0;
	return pD->fieldRT;
}
/*--------------------------------------------------------------------------------------*/
unsigned short HPI_API HPI_RDS_Get_PI(
					HPI_RDS_HANDLE h		///< A previously allocated HPIRDS handle.
					)
{
	struct  HPI_RDS_DATA *pD=(struct  HPI_RDS_DATA *)h;
	if(h==NULL)
		return HPI_ERROR_INVALID_OBJ;
	return pD->fieldPI;
}
/*--------------------------------------------------------------------------------------*/
unsigned char HPI_API HPI_RDS_Get_PTY(
					HPI_RDS_HANDLE h		///< A previously allocated HPIRDS handle.
					)
{
	struct  HPI_RDS_DATA *pD=(struct  HPI_RDS_DATA *)h;
	if(h==NULL)
		return HPI_ERROR_INVALID_OBJ;
	return pD->fieldPTY;
}
/*--------------------------------------------------------------------------------------*/
HPI_RDS_STRING HPI_API HPI_RDS_Get_PTY_Text(
					HPI_RDS_HANDLE h		///< A previously allocated HPIRDS handle.
					)
{
	struct  HPI_RDS_DATA *pD=(struct  HPI_RDS_DATA *)h;
	if(h==NULL)
		return szERROR;

	if (pD->fieldPTY < MAX_PTY)
		return szRDS_PTY_RDS[pD->d.eType][pD->fieldPTY];
	else
		return szRDS_PTY_RDS[pD->d.eType][0];
}
/*--------------------------------------------------------------------------------------*/
unsigned int HPI_API HPI_RDS_Get_TP(
					HPI_RDS_HANDLE h		///< A previously allocated HPIRDS handle.
					)
{
	struct  HPI_RDS_DATA *pD=(struct  HPI_RDS_DATA *)h;
	if(h==NULL)
		return HPI_ERROR_INVALID_OBJ;
	return pD->flagTP;
}


/**************************** PRIVATE *************************************/

/*--------------------------------------------------------------------------------------*/
/** Reset the RDS structure
  */
static void resetRDS(struct HPI_RDS_DATA *pD)
{
	enum eHPI_RDS_type eType=pD->d.eType;
	memset(pD,0,sizeof(struct HPI_RDS_DATA));
	pD->d.eType = eType;

/*
	memset(pD->fieldPS,' ',8);
	memset(pD->fieldPSbuild,' ',8);
	memset(pD->fieldRT,' ',64);
	memset(pD->fieldRTbuild,' ',64);
*/
}

/*--------------------------------------------------------------------------------------*/
/** Process block 1 is common processing that extracts the PI code.
  */
static void processBlock1(struct HPI_RDS_DATA *pD)
{
	unsigned short *pB= (unsigned short *)pD->d.lastGroup;
	unsigned short uPI;
	struct HPI_RDS_DATA_PRESERVE localCopy;

	// ---------------------  PI --------------------------
	// if PI has changed, we need to reset some things
 	uPI = pB[0];
	if(uPI != pD->fieldPI)
	{
		// Save some values for later.
		localCopy = pD->d;

		// can't just reset everything because we will lose the data we are processing !
		resetRDS(pD);

		// restore saved values that we need later.
		pD->d = localCopy;
		pD->fieldPI = uPI;
	}
}

/*--------------------------------------------------------------------------------------*/
/** Process block 2 is common processing that extracts common portions of Block 2.
  * The portions extracted are:<br>
  * Group number<br>
  * Group version (sometimes called group type) - 'A' or 'B'<br>
  * PTY code.<br>
  *
  */
static void processBlock2(struct HPI_RDS_DATA *pD)
{
	unsigned short *pB=(unsigned short *)pD->d.lastGroup;

	// --------------------- Group number and type --------------------------
	pD->fieldGroup = pB[1]>>12;
	if ((pB[1]>>11)&1)
		pD->fieldGroupVersion = 'B';
	else
		pD->fieldGroupVersion = 'A';

	pD->fieldPTY = (pB[1] >> 5) & 0x1f;

	// --------------------- TP --------------------------
	pD->flagTP = (pB[1] >> 10 ) & 1;

}


/*--------------------------------------------------------------------------------------*/
/** Process group 0A
  */
static void processGroup0A(struct HPI_RDS_DATA *pD)
{
	int nPSindex;
	unsigned short *pB=(unsigned short *)pD->d.lastGroup;

	nPSindex = (pB[1] & 0x3)*2;
	pD->fieldPSbuild[nPSindex] = pB[3] >> 8;
	pD->fieldPSbuild[nPSindex+1] = pB[3] & 0xff;
	pD->uPScount += 2;
	/*
	Note: Case has been observed where radio stations transmit multiple strings in
	the PS field (which is not allowed by the spec.). For WXXI in Rochester NY, this
	causes an unusual wrap around of counters, indicating that the transmission of
	multiple strings is completed. This could be detected by checking for the
	case where (pPSindex==6) && (pD->uPScount>8).
	*/
	if(nPSindex==6)
	{
		if(pD->uPScount>=8)
		{
			if(strncmp(pD->fieldPS,pD->fieldPSbuild,8)!=0)	// are the strings different ?
			{
				memcpy(pD->fieldPS,pD->fieldPSbuild,8);
				pD->flagPSready = 1;
			}
		}
		pD->uPScount=0;
	}
/*
	printf("nPSindex = %d, uPScount = %d, chars (%c%c) (0x%02X,0x%02X)\n",
		nPSindex,
		pD->uPScount,
		pD->fieldPSbuild[nPSindex],
		pD->fieldPSbuild[nPSindex+1],
		pD->fieldPSbuild[nPSindex],
		pD->fieldPSbuild[nPSindex+1]
		);
*/

}
/** Process group 0B
  */
static void processGroup0B(struct HPI_RDS_DATA *pD)
{
	// Use 0A for the moment.
	processGroup0A(pD);
}


/*--------------------------------------------------------------------------------------*/
/** Process group 2A
  * 
  * This routine is complicated by the fact that we want to support transmitters
  * that do not completely follow the RDS specification.
  *
  * There are basically 3 possible paths below that could produce a RT output:
  *
  * 1) The A/B flag has toggled and required counts have been exceed. In this case
  * we are currently processing a different flag than the one with output RT.
  *
  * 2) The A/B has not toggled but the required counts have been exceeded. In this
  * case we are processing the same flag as the one with the output RT.
  *
  * In both cases counting should have been from index 0.
  */
static void processGroup2A(struct HPI_RDS_DATA *pD)
{
	int nRTindex;
	unsigned short *pB=(unsigned short *)pD->d.lastGroup;
	unsigned int nRTtextFlag;
	char bytes[4];
	int i;
	unsigned int uUpdateRT=1;
	int nCompletedRTFlag=-1;		// -1 indicates no completed RT.
	int nCompletedCount=0;			// number of completed characters.

	nRTindex = (pB[1] & 0xf)*4;
	// check the RT text flag
	nRTtextFlag = (pB[1] >> 4) & 1;

	// We have an A/B toggle, or starting from 0 agian - so record which data set we are going to check.
	if(nRTtextFlag != pD->d.flagRTflag)
		nCompletedRTFlag = pD->d.flagRTflag;
	else
		if (nRTindex==0)
			nCompletedRTFlag = nRTtextFlag;

	pD->d.flagRTflag = nRTtextFlag;

	//  Check that we counted from 0.
	if(nCompletedRTFlag != -1)
		if(!pD->d.flagRTcountfromzero[nCompletedRTFlag])
			nCompletedRTFlag = -1;
		
	if(nCompletedRTFlag != -1)
		nCompletedCount = pD->d.uRTcount[nCompletedRTFlag];

	// Finsh checking for a completed RT before we process the new data.
	if(nCompletedRTFlag != -1)
	{
		for(i=0;i<nCompletedCount;i++)
		{
			if(pD->d.uRTValidCount[nCompletedRTFlag][i]<pD->d.uRTValidTreshold)
			{
				uUpdateRT=0;
				break;
			}
			if(pD->d.fieldRTbuild[nCompletedRTFlag][i] == 0x0d)
				break;
		}

		// clear the RT buffer on flag change from A to B
		if(uUpdateRT)
		{
			memset(pD->fieldRT,0,64);
			for(i=0;i<nCompletedCount;i++)
			{
				if(pD->d.fieldRTbuild[nCompletedRTFlag][i] == 0x0d)
					break;
				else
				{
					pD->fieldRT[i] = pD->d.fieldRTbuild[nCompletedRTFlag][i];
					pD->d.uRTValidCount[nCompletedRTFlag][i]--;
				}
			}
			pD->flagRTready = 1;
			pD->d.flagRTcountfromzero[nCompletedRTFlag] = 0;
			pD->d.uRTcount[nCompletedRTFlag]=0;
		}
	}




	if(nRTindex==0)
	{
		pD->d.flagRTcountfromzero[nRTtextFlag]=1;
		pD->d.uRTcount[nRTtextFlag]=0;
	}
	bytes[0] = pB[2] >> 8;
	bytes[1] = pB[2] & 0xff;
	bytes[2] = pB[3] >> 8;
	bytes[3] = pB[3] & 0xff;

	for(i=0;i<4;i++)
	{
		if(bytes[i]==pD->d.fieldRTbuild[nRTtextFlag][nRTindex+i])
		{
			if(pD->d.uRTValidCount[nRTtextFlag][nRTindex+i]<pD->d.uRTValidTreshold)
				pD->d.uRTValidCount[nRTtextFlag][nRTindex+i]++;
		}
		else
			pD->d.uRTValidCount[nRTtextFlag][nRTindex+i]=1;
		pD->d.fieldRTbuild[nRTtextFlag][nRTindex+i]=bytes[i];
	}

	pD->d.uRTcount[nRTtextFlag] += 4;

/* *
	printf("[%d] RTIndex = %d, d.uRTcount = %d, chars (%c%c%c%c) (0x%02X,0x%02X,0x%02X,0x%02X)\n",
		nRTtextFlag,
		nRTindex,
		pD->d.uRTcount[nRTtextFlag],
		pD->d.fieldRTbuild[nRTtextFlag][nRTindex],
		pD->d.fieldRTbuild[nRTtextFlag][nRTindex+1],
		pD->d.fieldRTbuild[nRTtextFlag][nRTindex+2],
		pD->d.fieldRTbuild[nRTtextFlag][nRTindex+3],
		pD->d.fieldRTbuild[nRTtextFlag][nRTindex],
		pD->d.fieldRTbuild[nRTtextFlag][nRTindex+1],
		pD->d.fieldRTbuild[nRTtextFlag][nRTindex+2],
		pD->d.fieldRTbuild[nRTtextFlag][nRTindex+3]
		);
* */

}

/*
$Log: hpirds.c,v $
Revision 1.21  2007/06/04 18:29:23  as-age
Only flag a new PS if the text has changed.

Revision 1.20  2007/05/09 19:37:31  as-age
Improved RT detection.

Revision 1.19  2007/04/24 18:58:57  as-age
RT string updates. We now count correct chars across all strings.

Revision 1.18  2007/04/13 20:25:42  as-age
Fix for PTY.

Revision 1.17  2007/04/11 19:52:22  as-age
When PI changes, take care not to clobber the current block when zeroing the RDS data
structure.

Revision 1.16  2007/03/29 02:48:37  as-ewb
bler contains error counts. up to 5 have been corrected by hardware

Revision 1.15  2007/03/27 20:29:26  as-age
Add block error reporting and error checking in the analysis section.

Revision 1.14  2007/03/26 20:09:06  as-age
Add support for returning and using RDS block error codes.

Revision 1.13  2007/03/23 15:13:35  as-age
Have to typedef char * return for DLL compile. SWIG still seems to run ok.

Revision 1.12  2007/03/22 21:59:05  as-ewb
add RDBS strings to  HPI_RDS_Get_PTY_Text. Preserve type across reset

Revision 1.11  2007/03/22 21:17:00  as-ewb
change string returns to char *

Revision 1.10  2007/03/21 02:39:14  as-ewb
avoid compiler warnings

Revision 1.9  2007/03/19 19:05:41  as-age
Check for null handles.

Revision 1.8  2007/03/15 18:50:44  as-age
Add code for building PS and RT fields in the background and flagging when there
is a new string available.

Revision 1.7  2007/03/13 19:51:51  as-age
Add group 0B. Fix PS string indexing.

Revision 1.6  2007/02/27 13:46:13  as-age
Rename some functions.

Revision 1.5  2007/02/09 19:57:37  as-age
Minor type changes. Doc updates.

Revision 1.4  2007/02/09 17:07:35  as-age
Add HPIRDS calls to asihpi32.dll.

Revision 1.3  2007/02/07 19:31:48  as-age
More documentation and add PTY and TP fields.

Revision 1.2  2007/02/06 17:55:06  as-age
Documentation updates.

Revision 1.1  2007/02/05 19:09:01  as-age
HPIRDS first HPI RDS module checkin.

Revision 1.1  2007/02/05 18:34:02  as-age
HPIRDS user level code for processing RDS data.

*/
