/******************************************************************************
 $Header: /Repository/drv/hpi/hpinet.c,v 1.18 2006/12/05 01:36:34 as-ewb Exp $

 HPI NETWORK implementation

 (C) Copyright AudioScience Inc. 2005
*******************************************************************************/

#define WIN32_LEAN_AND_MEAN
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <hpi.h>
#include <hpidspcd.h>
#include <hpinet.h>

#ifndef HPI_OS_LINUX

#ifndef _SS_PAD1SIZE
#define _SS_PAD1SIZE 6
#define _SS_PAD2SIZE 112 

/** IPv6 generalised sockaddr not in older winsock2.h */
typedef struct sockaddr_storage {
  short ss_family;
  char __ss_pad1[_SS_PAD1SIZE];
  __int64 __ss_align;
  char __ss_pad2[_SS_PAD2SIZE];
} SOCKADDR_STORAGE,
*PSOCKADDR_STORAGE;
#endif
#endif

/* avoid warning about double definition */
#define HAVE_SNPRINTF
#define HAVE_VSNPRINTF

#ifdef HPI_OS_LINUX
#include <pcap.h>
#define PCAP_SEND_PACKET_TYPE const u_char *

#else

#include <pcap-int.h>
#include <ntddndis.h>

#define PCAP_SEND_PACKET_TYPE  char *
#endif

typedef void HPI_message_function(HPI_MESSAGE *phm, HPI_RESPONSE *phr );
struct hpi_adapter;

typedef struct {
	HPI_message_function * hpi_message;
	HW32 adapterType;
	HW16 adapterIndex;
	//HW64 adapterUniqueId; // big enough for MAC address
	HPI_ETHERNET_MAC_ADR adapterMAC;
	HW16 adapterIsOpen;
	HW16 lastError;
	struct hpi_adapter * pNext; /* can make a linked list of them */
} hpi_adapter;


#define MAX_BROADCAST_RESPONSES 10
#define NET_TIMEOUT_MS 10

int hpinet_pcap_inum=-1;

/** global: MAC address of ethernet adapter used by hpinet */
HPI_ETHERNET_MAC_ADR gMyMAC ={0,0,0,0,0,0};

/** global: handle to WinPcap device being used */
pcap_t *fp=0;

/** global: List of adapters known by this HPI */
hpi_adapter adapters[MAX_BROADCAST_RESPONSES];

static int num_adapters=0;

/** Local prototypes */
HW16 SubsysOpen(void );
HW16 SubsysClose(void);
void SubsysFindAdapters(HPI_MESSAGE *phm, HPI_RESPONSE *phr);

/*=================================================================*/
HPINET_RESP gResp;

void packet_callback(unsigned char *Args,
                      const struct pcap_pkthdr* Pkthdr,
                      unsigned char *Packet)
{
#if 0
	int i;

	fprintf(stderr, "[%d]",Pkthdr->caplen); fflush(stderr);
	for (i=0; i<Pkthdr->caplen; i++)
		printf("%2hhx ",Packet[i]);
	printf("\n");
#endif
	memcpy(&gResp,Packet,sizeof(gResp));
}

/*=================================================================*/
/**Send an HPI message encapsulated in Cobranet protocol

phr->wSize must be set before calling this function, to indicate the actual
size available for the returned response.
If it is  zero, response size defaults to sizeof(HPI_MESSAGE)
*/
void HPINET_Message(
		HPI_ETHERNET_MAC_ADR msgDestMAC, ///< recipient address
		HPI_MESSAGE *phm, ///< message to send
		HPI_RESPONSE *phr, ///< response received
		unsigned int timeout
		)
{
	HPINET_MSG msg;
	int nSize;
	int packets=0;

	memcpy(msg.hdr.destMAC,msgDestMAC,sizeof(msg.hdr.destMAC));
	memcpy(msg.hdr.srcMAC,gMyMAC,sizeof(msg.hdr.srcMAC));

	msg.hdr.ProtocolLS = 0x19;
	msg.hdr.ProtocolMS = 0x88;
	msg.hdr.ASIpktID = HPI_ETHERNET_PACKET_ID;
	msg.hdr.ASIpktVersion = HPI_ETHERNET_PACKET_HOSTED_VIA_HMI_V1;

	nSize = sizeof(HPINET_HEADER)+phm->wSize;
	memcpy(&msg.HPImsg,phm,nSize);

	// Write the packet
	if (pcap_sendpacket(fp, (PCAP_SEND_PACKET_TYPE)&msg, nSize) != 0)
	{
		fprintf(stderr,"\nError sending the packet: %s\n", pcap_geterr(fp));
		phr->wError= HPI_ERROR_NETWORK_TIMEOUT;
		phr->wSize = HPI_RESPONSE_FIXED_SIZE;
		return;
	}
#ifdef HPI_OS_LINUX
	{ // read the responses
		int count=(timeout+NET_TIMEOUT_MS-1)/NET_TIMEOUT_MS;
		fd_set            fd_wait;
		struct timeval    st;
		int t;

		for(; ((count > 0) && !packets) ; count--)
		{	/* If you include STDIN_FILENO, be sure to read from it when you get
			traffic from it, or select will return every time it is called,
			since there will still be traffic waiting there. */
			
			FD_ZERO(&fd_wait);
			/*FD_SET(STDIN_FILENO, &fd_wait);*/
			FD_SET(pcap_fileno(fp), &fd_wait);
			
			st.tv_sec  = 0;
			st.tv_usec = NET_TIMEOUT_MS*1000;   /* 10000000 = 1 second */
			
			t=select(FD_SETSIZE, &fd_wait, NULL, NULL, &st);
			
			switch(t)
			{
			case -1:      /* Something went really wrong */
				fprintf(stderr, "select() returned -1, quitting\n\n");
				exit(1);
			
			case  0:      /* We timed out, no trafffic */
				//fprintf(stderr, ".");
				break;
			
			default:      /* We got traffic */
				pcap_dispatch(fp, 1, (void *) packet_callback, NULL);
				packets++;
			}
		} /* End of for() */
#else
	{ // read the responses
		struct pcap_pkthdr *header;
		u_char *pkt_data;
		int count=(timeout+NET_TIMEOUT_MS-1)/NET_TIMEOUT_MS;
		int res;

		while(count-- && ((res = pcap_next_ex(fp, &header, &pkt_data)) >= 0)){
			if(res == 0)
				/* Timeout elapsed */
				continue;
			else {
				memcpy(&gResp,pkt_data,sizeof(gResp));

/*
	for(i=0;i<6;i++)
	{
		if (msgDestMAC[i]  != resp.hdr.srcMAC[i])
			continue;
	}

*/
				packets++;
				break;
			}
		}
#endif
	}

	if (!packets) {
		phr->wError = HPI_ERROR_NETWORK_TIMEOUT;
		phr->wSize = HPI_RESPONSE_FIXED_SIZE;
		return;
	}
	

	if (!phr->wSize)
		phr->wSize=sizeof(*phr);

	if (gResp.HPIresp.wSize > phr->wSize) {
			phr->wError= HPI_ERROR_INVALID_DATASIZE;
			phr->wSize = HPI_RESPONSE_FIXED_SIZE;
			return;
	}

	memcpy(phr,&gResp.HPIresp,gResp.HPIresp.wSize);
}

/** Broadcast an HPI message encapsulated in Cobranet protocol

  The message is broadcast using the global source address gMyMACto
  the all ones broadcast address.

  Any responses and corresponding mac addresses are stored in supplied arrays.

  Errors are returnd in phm->wError
  If no messages are received, error HPI_ERROR_NETWORK_TIMEOUT is filled in the
  first response structure, but *num_packets is set to zero.

*/
void HPINET_BroadcastMessage(
		HPI_ETHERNET_MAC_ADR *respSrcMAC, ///< Array of mac addresses of responders
		HPI_MESSAGE *phm, ///< Message to send
		HPI_RESPONSE *phr, ///< array of responses received
		unsigned int max_responses, ///< size of mac and response arrays
		unsigned int *num_packets, ///< number of responses received
		unsigned int timeout ///< to wait for responses in milliseconds
		)
{
	HPINET_MSG msg;
	int nSize = HPI_ETHERNET_HEADER_SIZE+phm->wSize;
	unsigned int packets=0;

	memset(&msg,0,sizeof(HPINET_MSG));
	memset(msg.hdr.destMAC,0xFF,sizeof(msg.hdr.destMAC)); // all F broadcast address
	memcpy(msg.hdr.srcMAC,gMyMAC,sizeof(msg.hdr.srcMAC));

	msg.hdr.ProtocolLS = 0x19;
	msg.hdr.ProtocolMS = 0x88;
	msg.hdr.ASIpktID = HPI_ETHERNET_PACKET_ID;
	msg.hdr.ASIpktVersion = HPI_ETHERNET_PACKET_HOSTED_VIA_HMI_V1;

	memcpy(&msg.HPImsg,phm,phm->wSize);

	// Write the packet
	if (pcap_sendpacket(fp, (PCAP_SEND_PACKET_TYPE)&msg, nSize) != 0)
	{
		fprintf(stderr,"\nError sending the packet: %s\n", pcap_geterr(fp));
			*num_packets=0;
			phr->wError = HPI_ERROR_NETWORK_TIMEOUT;
			phr->wSize = HPI_RESPONSE_FIXED_SIZE;
		return;
	}

	{ // read the responses
#ifdef HPI_OS_LINUX
		int count=(timeout+NET_TIMEOUT_MS-1)/NET_TIMEOUT_MS;
		fd_set            fd_wait;
		struct timeval    st;
		int t;
		packets=0;

		for(;count > 0 ; count--)
		{
			FD_ZERO(&fd_wait);
			FD_SET(pcap_fileno(fp), &fd_wait);
			
			st.tv_sec  = 0;
			st.tv_usec = NET_TIMEOUT_MS*1000;   /* 10000000 = 1 second */
			
			t=select(FD_SETSIZE, &fd_wait, NULL, NULL, &st);
			
			switch(t)
			{
			case -1:      /* Something went really wrong */
					fprintf(stderr, "select() returned -1, quitting\n\n");
					exit(1);
			
			case  0:      /* We timed out, no trafffic */
					//fprintf(stderr, ".");
					break;
			
			default:      /* We got traffic */
				pcap_dispatch(fp, 1, (void *) packet_callback, NULL);
				packets++;
				memcpy(phr,&gResp.HPIresp,gResp.HPIresp.wSize);
				memcpy(respSrcMAC,gResp.hdr.srcMAC,6);
				phr++;
				respSrcMAC++;
			}
		} /* End of for(;;) */
#else
		struct pcap_pkthdr *header;
		u_char *pkt_data;
		int count=(timeout+NET_TIMEOUT_MS-1)/NET_TIMEOUT_MS;
		int res;
		packets=0;

		while(count-- &&
			 (packets <= max_responses) &&
			 ((res = pcap_next_ex(fp, &header, &pkt_data)) >= 0))
		{
			if(res == 0)
				/* Timeout elapsed */
				continue;
			else {
				packets++;
				memcpy(&gResp,pkt_data,sizeof(gResp));
				memcpy(phr,&gResp.HPIresp,gResp.HPIresp.wSize);
				memcpy(respSrcMAC,gResp.hdr.srcMAC,6);
				phr++;
				respSrcMAC++;
			}
		}

#endif
		*num_packets=packets;
		if (!packets) {
			phr->wError = HPI_ERROR_NETWORK_TIMEOUT;
			phr->wSize = HPI_RESPONSE_FIXED_SIZE;
			//fprintf(stderr,"\nNo responses to broadcast packet: %s\n", pcap_geterr(fp));

			return;
		} else {
			//fprintf(stderr,"\n%d responses to broadcast packet\n",packets);
		}
			
	}
}

/** Send extended HPI message
Extended message may include an appended data element.
and/or have a non-default timeout.

phr->wSize must be set before calling this function, to indicate the actual
size available for the returned response.
If it is  zero, response size defaults to sizeof(HPI_MESSAGE)
*/
void HPI_MessageEx(HPI_MESSAGEX *phm, HPI_RESPONSEX *phr ,
		unsigned int timeout)
{
	HPINET_Message(
		adapters[phm->msg.wAdapterIndex].adapterMAC,
		(HPI_MESSAGE *)phm,(HPI_RESPONSE *)phr,timeout);
}

void HPI_NET(HPI_MESSAGE *phm, HPI_RESPONSE *phr )
{
	HPINET_Message(
		adapters[phm->wAdapterIndex].adapterMAC,
		phm,phr,100);
}

/*=================================================================*/
/**

*/
void HPI_Message(HPI_MESSAGE *phm, HPI_RESPONSE *phr )
{
    phr->wSize=0;

    // check for various global messages
    if(phm->wObject == HPI_OBJ_SUBSYSTEM)
    {
        // subsys message get sent to more than one hpi, may not be filled in by any of them
        // so need to create a default response here
		// HPI_SUBSYS_CREATE_ADAPTER is a special case where the response from previous hpi
		// calls is passed in to facilitate duplicate adapter index detection, so don't init
		if( phm->wObject == HPI_OBJ_SUBSYSTEM && phm->wFunction == HPI_SUBSYS_CREATE_ADAPTER)
			phr->wSize = HPI_RESPONSE_FIXED_SIZE + sizeof(HPI_SUBSYS_RES);
		else
			HPI_InitResponse(phr, phm->wObject, phm->wFunction, HPI_ERROR_INVALID_OBJ );

        switch(phm->wFunction)
        {
            // subsys open - init the debug function
        case HPI_SUBSYS_OPEN:
			phr->wError=SubsysOpen();
            return;

        case HPI_SUBSYS_FIND_ADAPTERS:
            SubsysFindAdapters(phm, phr);
            return;

        case HPI_SUBSYS_CLOSE:
			phr->wError=SubsysClose();
            return;

            // version message - if so then leave
        case HPI_SUBSYS_GET_VERSION:
            HPI_InitResponse(phr, HPI_OBJ_SUBSYSTEM, HPI_SUBSYS_GET_VERSION,0);
            phr->u.s.dwVersion = HPI_VER;
            return;

        default:
            break;
        }
    }

	if (phm->wAdapterIndex >= num_adapters) {
		phr->wError = HPI_ERROR_INVALID_OBJ_INDEX;
		phr->wSize = HPI_RESPONSE_FIXED_SIZE;
		return;
	}

	adapters[phm->wAdapterIndex].hpi_message(phm,phr);

    if (phr->wSize==0)
    {	// This should not happen. If it does this is a coding error.
        // However, still fill in enough of the response to return an error to help debugging.
		phr->wError = HPI_ERROR_PROCESSING_MESSAGE;
        phr->wSize = HPI_RESPONSE_FIXED_SIZE;
    }

}

/*=================================================================*/
/**  Find winpcap devices and open one for use.

  Get the MAC address of the device, and store it in gMyMAC

  Install a packet filter to only receive cobranet packets that are also
  HPI responses

*/
HW16 SubsysOpen(void )
{
 // Find and open the interface
	pcap_if_t *d;
	pcap_if_t *alldevs;
	int i=0;
	char errbuf[PCAP_ERRBUF_SIZE];

	/* Retrieve the device list on the local machine */
//	if (pcap_findalldevs_ex(PCAP_SRC_IF_STRING, NULL, &alldevs, errbuf) == -1)
	if (pcap_findalldevs(&alldevs, errbuf) == -1)
	{
		fprintf(stderr,"Error in pcap_findalldevs: %s\n", errbuf);
		return (-1);
	}

    /* Print the list */
	printf("Available interfaces\n");
    for(d=alldevs; d; d=d->next)
    {
        printf("%d. %s", ++i, d->name);
        if (d->description)
            printf(" (%s)\n", d->description);
        else
            printf(" (No description available)\n");
    }

    if(i==0)
    {
        printf("\nNo interfaces found! Make sure WinPcap is installed.\n");
        return -1;
    }

	/* -ve inum  is default, means find first interface with mac */
	if (hpinet_pcap_inum >=0) {
		/* specifying -i0 causes prompt */
		if (!hpinet_pcap_inum) {
			printf("Enter the interface number (1-%d):",i);
			scanf("%d", &hpinet_pcap_inum);
		}

		if(hpinet_pcap_inum < 1 || hpinet_pcap_inum > i)
		{
			printf("\nInterface number %d out of range.\n",hpinet_pcap_inum );
			/* Free the device list */
			pcap_freealldevs(alldevs);
			return -1;
		}

		/* Jump to the selected adapter */
		for(d=alldevs, i=0; i< hpinet_pcap_inum-1 ;d=d->next, i++);

	} else {
		/* set up to search through all */
		d=alldevs;
		i=0;
	}
	
	/* search for the first adapter that can be opened, and a MAC address retrieved */
	for( ; d ;d=d->next,i++)

	{
		/* Open the device */
		fp=pcap_open_live(d->name,          // name of the device
				1600,            // portion of the packet to capture
				// 65536 guarantees that the whole packet will be captured on all the link layers
				0,	// normal mode
				NET_TIMEOUT_MS,             // read timeout milliseconds
				//NULL,             // authentication on the remote machine
				errbuf            // error buffer
				);
		if (fp==NULL)
		{
			fprintf(stderr,"\nUnable to open the adapter. %s is not supported by WinPcap\n", d->name);
			continue;
		}

#ifdef HPI_OS_LINUX
	if(pcap_setnonblock(fp, 1, errbuf) == 1)
	{
	fprintf(stderr, "Could not set device \"%s\" to non-blocking: %s\n", d->name, 
	errbuf);
	exit(1);
	}

{
/* get the MAC address for the adapter */
#include <linux/if_ether.h> // for ETH_P_ALL
#include <sys/ioctl.h> //for ioctl
#include <net/if.h> // for ifreq
#include <arpa/inet.h> // for htons

	struct ifreq ifr;
	
	int s; /*socketdescriptor*/
	int result;
	
	s = socket(AF_PACKET, SOCK_RAW, htons(ETH_P_ALL));
	if (s < 0) { printf("Socket error %d\n",s); exit(1); }
	
	{ /* print the list of interfaces and MAC addresses */
		int index,i;
		for (index=1; index<10; index++) {
			ifr.ifr_ifindex=index;
			result=ioctl(s,SIOCGIFNAME,&ifr);
			if (result==0) {
				for (i=0; i<6; i++) ifr.ifr_hwaddr.sa_data[i]=0;
				if (strcmp(ifr.ifr_name,d->name)==0) {
					result=ioctl(s,SIOCGIFHWADDR,&ifr);
					if (result==0) {
						printf("  Interface %d name %s. ",index,ifr.ifr_name);
						printf("  MAC address ");
						for (i=0; i<6; i++) {
							printf("%2.2hhx:",ifr.ifr_hwaddr.sa_data[i]);
							gMyMAC[i]=ifr.ifr_hwaddr.sa_data[i];
						}
			
						printf("\n");
						break;
					}
				}
			}
			else
				break;
		}
	}
break;
}
#else
		{ /* Get the MAC address of the selected ethernet adapter */
			PPACKET_OID_DATA  OidData;
			BOOLEAN		Status;

			OidData = malloc(6 + sizeof(PACKET_OID_DATA));
			if (OidData == NULL)
			{
				printf("error allocating memory!\n");
				pcap_close(fp);
				return -1;
			}

			//
			// Retrieve the adapter MAC querying the NIC driver
			//

			OidData->Oid = OID_802_3_CURRENT_ADDRESS;

			OidData->Length = 6;
			ZeroMemory(OidData->Data, 6);

			Status = PacketRequest(fp->adapter, FALSE, OidData);
			if(Status)
			{
				printf("The MAC address of adapter %d is %.2x:%.2x:%.2x:%.2x:%.2x:%.2x\n",
					i+1,
					(PCHAR)(OidData->Data)[0],
					(PCHAR)(OidData->Data)[1],
					(PCHAR)(OidData->Data)[2],
					(PCHAR)(OidData->Data)[3],
					(PCHAR)(OidData->Data)[4],
					(PCHAR)(OidData->Data)[5]);

				memcpy(gMyMAC,OidData->Data,6);
				break;
			}
			else
			{
				pcap_close(fp);
				if (hpinet_pcap_inum >= 0) {
					/* specific adapter selected had no mac address */
					pcap_close(fp);
					d=NULL;
					break;
				} else { /* keep searching for adapter that has a mac address */
					continue;
				}
			}
		}

#endif
	}
	if (!d) {
	        /* Free the device list */
		printf("Error retrieving the MAC address of adapter %d!\n",i+1);
        pcap_freealldevs(alldevs);
        return -1;
	}

    printf("\nlistening on adapter %d\n",i+1);
    /* At this point, we don't need any more the device list. Free it */
    pcap_freealldevs(alldevs);
	//? phSubSys=(HPI_HSUBSYS *)fp;

	{ // Compile and install the packet filter, only receive HPI responses
		u_int netmask=0xffffff;
		struct bpf_program fcode;

		/* Packet structure[sizes in bytes]
		   Ethernet dest mac [6]
		   Ethernet src mac  [6]
		   Cobranet protocol [2] = 0x8819
		   ASI CN packet ID  [1] = 0x85
		   ASI packet version[1] = 0x21
		   HPI msg/resp size [2]
		   HPI type Response [2] = 2
		*/
//		char filter_expression[]="ether[12:4] = 0x88198521 and ether[18]=2";
		char filter_expression[]="ether proto 0x8819 and ether[14]=0x85 and ether[18]=2";

		if (pcap_compile(fp, &fcode,filter_expression, 1, netmask) < 0)
		{
			fprintf(stderr,"\nUnable to compile the packet filter. Check the syntax of (%s).\n",filter_expression);
			return -1;
		}

		if (pcap_setfilter(fp, &fcode) < 0)
		{
			fprintf(stderr,"\nError setting the filter.\n");
			return -1;
		}
	}

	memset(adapters,0,sizeof(adapters));
	return 0;
}
/*=================================================================*/
/**

*/
HW16 SubsysClose(void)
{
	pcap_close(fp);
	return 0;
}

/*=================================================================*/
/**

*/
void PrintMac(HPI_ETHERNET_MAC_ADR	mac)
{
	int i;
	for (i=0; i<5; i++)
		printf("%02X:",mac[i]);
	printf("%02X",mac[5]);
}


HW16 SubsysGetAdapterIndexForMac(HPI_ETHERNET_MAC_ADR mac, HW16 * pIndex)
{
	int i;
	int s=sizeof(HPI_ETHERNET_MAC_ADR);
	for (i=0; i<MAX_BROADCAST_RESPONSES; i++)
		if (0==memcmp(mac,adapters[i].adapterMAC,s)) {
			*pIndex = i;
			return 0;
		}
	return 1;
}

HW16 SubsysGetAdapterIndexForMacString(const char * smac, HW16 * pIndex)
{
	HPI_ETHERNET_MAC_ADR mac;
	int x,i;

	for (i=0; i<6; i++) {
		sscanf(&smac[i*2],"%2x",&x);
		mac[i]=x;
	}

	return SubsysGetAdapterIndexForMac(mac,pIndex);
}


HPI_ERR AdapterGetMACaddress(HW16 adapterIndex,
				unsigned char ** pMAC
				)
{
	if (adapterIndex >= num_adapters) {
		pMAC=NULL;
		return HPI_ERROR_BAD_ADAPTER_NUMBER;
	}

	*pMAC= &adapters[adapterIndex].adapterMAC[0];
	return 0;
}


/*=================================================================*/
/**

*/
void SubsysFindAdapters(HPI_MESSAGE *phm, HPI_RESPONSE *phr)
{ // Find adapter(s)
	HPI_ETHERNET_MAC_ADR respSrcMACs[MAX_BROADCAST_RESPONSES];
	HPI_MESSAGE hm;
	HPI_RESPONSE bhr[MAX_BROADCAST_RESPONSES];
	unsigned int num_responses=0;
	unsigned int i;
	char errbuf[128];

	HPI_InitMessage( &hm,  HPI_OBJ_ADAPTER, HPI_ADAPTER_GET_INFO);

	HPINET_BroadcastMessage(
		respSrcMACs,
		&hm,
		bhr,
		MAX_BROADCAST_RESPONSES,
		&num_responses,
		2000u);

//	printf("Maybe Found %d adapters\n",num_responses);
	for (i=0; i<num_responses; i++) {
		HPI_RESPONSE * hr=&bhr[i];
		hpi_adapter * pA = &adapters[i];

//		printf("Response %d\n",i);
		if (hr->wError==0) {
#if 0
			printf("\tAdapter type ASI%4X\n",hr->u.a.wAdapterType);
			printf("\tAdapter serial number %d\n",hr->u.a.dwSerialNumber);
			printf("\tAdapter version %d\n",hr->u.a.wVersion);
			printf("\tAdapter mac address ");
			PrintMac(respSrcMACs[i]);
			printf("\n");
#endif
			pA->adapterIndex=num_adapters;
			pA->adapterType=hr->u.a.wAdapterType;
			memcpy(&pA->adapterMAC,respSrcMACs[i],sizeof(HPI_ETHERNET_MAC_ADR));
			pA->adapterIsOpen=0;
			pA->hpi_message=HPI_NET;
			pA->lastError=0;
			phr->u.s.awAdapterList[num_adapters]=hr->u.a.wAdapterType;
			phr->wError=0;
			num_adapters++;
		} else {
			HPI_GetErrorText( hr->wError, errbuf);
			printf("\tError%s\n",errbuf);
			pA->adapterIndex=0;
			pA->lastError=hr->wError;
		}
	}
	phr->u.s.wNumAdapters = num_adapters;
}

HW16 HPI_DriverOpen(void)
{
	return 1;
}

void HPI_DriverClose(void)
{

}

// $Log: hpinet.c,v $
// Revision 1.18  2006/12/05 01:36:34  as-ewb
// Check for valid adapterindex in HPI_Message
//
// Revision 1.17  2006/11/30 11:18:56  as-ewb
// correct adapter count from findadapters
//
// Revision 1.16  2006/11/30 04:11:41  as-ewb
// remove unneeded headers
//
// Revision 1.15  2006/11/30 01:49:01  as-ewb
// Fix windows compile errors and warnings
//
// Revision 1.14  2006/11/29 23:48:37  as-ewb
// merge linux and windows versions. Relax pcap filter for responses
//
// Revision 1.2  2006/11/28 07:52:11  as-ewb
// update interface selection, add AdapterGetMACaddress
//
// Revision 1.1  2006/11/27 22:26:03  as-ewb
// linux versin of hpinet.c, maybe able to merge later
//
// Revision 1.11  2006/10/17 15:09:27  as-age
// Use define for ASI pkt ID.
//
// Revision 1.10  2006/01/25 16:01:22  as-tfe
// Merged.
//
// Revision 1.7.2.2  2005/12/07 02:45:49  as-ewb
// Removed phSubsys from HPI_Message etc.
//
// Revision 1.7.2.1  2005/11/29 02:04:10  as-dxb
// Bringing the refactoring branch up to date with the cvs HEAD:
// copied hpinet.c and hpinet.h from HEAD
//
// Revision 1.9  2005/10/24 21:17:24  as-ewb
// FindAdapters return error=0 if at least one adapter found without error
//
// Revision 1.8  2005/10/13 21:25:03  as-ewb
// EWB increase HPI_NET() message timeout from 1 to 100ms. Sometimes dsp can't respond so fast.
//
// Revision 1.7  2005/08/30 21:28:51  as-ewb
// ewb To allow net interface to be selected at compile time, enable conditional block at line 359 and set hpinet_pcap_inum
//
// Revision 1.6  2005/08/10 07:29:50  as-ewb
// ewb add GetAdapterIndexForMac functions
//
// Revision 1.5  2005/08/05 01:08:45  as-ewb
// ewb comment out some printfs
//
// Revision 1.4  2005/08/04 13:45:23  as-age
// AGE - tweaks for modifications to hpinet.h structure defns for DSP compiler support.
// No functional changes.
//
// Revision 1.3  2005/08/04 03:25:38  as-ewb
// ewb Move some defs to hpinet.h.  Update message functions so extra data is passed appended to message/response.
// Open first adapter for which a mac address can be retrieved.
//
