/******************************************************************************
 $Header: /Repository/drv/hpi/hpifirmware.c,v 1.1 2007/09/10 20:30:23 as-ewb Exp $

  Firmware downloader for ASI2400

 (C) Copyright AudioScience Inc. 2005
*******************************************************************************/
#include <string.h>
#include <stdlib.h>

#include <hpi.h>
#include <hpios.h>  // for debug
#include <hpidspcd.h>
#include <hpidebug.h>
#include <hpinet.h>

#include <hpifirmware.h>

#define BOOTLOADER_OFFSET 0
#define FACTORY_OFFSET 0x08000
#define UPDATE_OFFSET  0x80000

struct fw_info {
	enum eHpiFirmwareId type;
	int offset;
	char name[12];
};

#define MAX_FILES 3
static struct fw_info fw[MAX_FILES]= {
	{HPI_FW_UPDATE,    UPDATE_OFFSET    , "Update"},
	{HPI_FW_FACTORY,   FACTORY_OFFSET   , "Factory"},
	{HPI_FW_BOOTLOADER,BOOTLOADER_OFFSET, "Bootloader"}
};

/*=================================================================*/
#ifdef _MSC_VER
static void delay(int millisecs)
{
	Sleep(millisecs);
}
#else
#include <time.h>

/*---------------------------------------------------------------------------*/
static void delay(long millisecs)
{
	struct timespec poll_interval={0,millisecs*1000000l};
	nanosleep(&poll_interval, 0);
}
#endif

/* max 999 */
#define MSG_DELAY 10
/*=================================================================*/
/**

*/
static HW32 calc_checksum(HW32 checksum, int count, HW32 * pData)
{	int i;
	for (i=0; i< count; i++)
		checksum ^= pData[i];

	return checksum;
}

/*=================================================================*/
/**

*/
static void null_progress_callback(char * state, u_int percent)
{
}

/** number of words in code tailer */
#define TAILERSIZE 3
/*=================================================================*/
/**

*/
HPI_ERR HPI_AdapterFirmwareDownload(HW16 wAdapterIndex, enum eHpiFirmwareId fwid,
                           const char *fwpath, progress_callback pc)
{
	short error;
	DSP_CODE DspCode;
	HW16 nBootLoadFamily=0x2400;
	HW32 dwLength;
	u_int checksum;
	u_int total_checksum=0;
	u_int dwStartOffset;
	u_int sequence=0;
	u_int progress=0;
	u_int p_inc;

	delay(MSG_DELAY);

	if (!pc)
		pc = &null_progress_callback;

	HpiOs_SetDspCodePath(fwpath);

	if ((fwid > HPI_FW_BOOTLOADER) || (fwid < HPI_FW_UPDATE))
		return HPI_ERROR_INVALID_OBJ_INDEX;

	dwStartOffset = fw[fwid-1].offset;
	printf("Downloading %s image\n", fw[fwid-1].name);

	if ((error = HpiDspCode_Open(nBootLoadFamily, &DspCode, NULL)) !=  0)
		return( error);

	if (fwid==HPI_FW_BOOTLOADER) {
		/* Bootloader needs to be raw code image, so discard
		   the three header words.
		   Check length of first block is same as total file length-
		   overhead, to make sure there is only one block
	    */
		HW32 dwAddress,dwType;

		HpiDspCode_ReadWord(&DspCode,&dwLength);
		HpiDspCode_ReadWord(&DspCode,&dwAddress);
		HpiDspCode_ReadWord(&DspCode,&dwType);

		if (dwLength != (DspCode.dwBlockLength - 4)) {
			return HPI_ERROR_DSP_FILE_FORMAT;
		}
		dwLength=DspCode.dwBlockLength-4+TAILERSIZE;
	} else {
		dwLength=DspCode.dwBlockLength+TAILERSIZE;
	}

#define WORDS_PER_BLOCK 256
	p_inc= 10000*WORDS_PER_BLOCK/dwLength;
	pc("Erasing",0);

	{
		HPI_MESSAGE hm;
		HPI_RESPONSE hr;
		HPI_InitMessage(&hm, HPI_OBJ_ADAPTER, HPI_ADAPTER_START_FLASH);
		hm.wAdapterIndex=wAdapterIndex;
		hm.u.ax.start_flash.dwKey=0x0a51;
		hm.u.ax.start_flash.dwOffset=dwStartOffset;
		hm.u.ax.start_flash.dwLength=DspCode.dwBlockLength+TAILERSIZE;

		hr.wError=0;
		hr.wSize=sizeof(hr);

		HPI_MessageEx((HPI_MESSAGEX *)&hm,(HPI_RESPONSEX *)&hr,10000); // wait up to 10 sec for flash erase
		error=hr.wError;
		delay(MSG_DELAY);
	}

	while ((DspCode.dwWordCount < DspCode.dwBlockLength) && ! error)
	{
		HPI_MESSAGEX hm;
		HPI_RESPONSE hr;
		HW16 wLength;
		HW32 *pdwCode;
		HW32 dwRemain=DspCode.dwBlockLength-DspCode.dwWordCount;
		int retries;

		wLength = WORDS_PER_BLOCK;
		if (wLength > dwRemain)
			wLength = (unsigned short)dwRemain;

		error=HpiDspCode_ReadBlock(wLength,&DspCode,&pdwCode);
		if (error)
			break;

		checksum = calc_checksum(0, wLength, pdwCode);
		total_checksum ^= checksum;

		HPI_InitMessage( (HPI_MESSAGE *)&hm,  HPI_OBJ_ADAPTER, HPI_ADAPTER_PROGRAM_FLASH);
		hm.msg.wAdapterIndex=wAdapterIndex;
		hm.msg.u.ax.program_flash.wSequence=sequence;
		hm.msg.u.ax.program_flash.dwChecksum=checksum;
		hm.msg.u.ax.program_flash.wLength=wLength;
		hm.msg.wSize=sizeof(HPI_MESSAGE)+wLength*sizeof(HW32);

		if(0)
		{
			int i;

			printf("Packet[%d]\n",sequence);
			printf("Checksum = 0x%08X, Length = 0x%X\n",checksum,wLength);
			for(i=0;i<wLength;i++)
			{
				printf("0x%08X ",pdwCode[i]);
				if( (i&3)==3)
					printf("\n");
			}
			printf("\n");
		}

		memcpy(&hm.data,pdwCode,wLength*sizeof(HW32));

		retries=2;

		while (retries) {
			hr.wError=0;
			hr.wSize=sizeof(hr);
			HPI_MessageEx(&hm,(HPI_RESPONSEX*)&hr,2000);
			delay(MSG_DELAY);

			retries--;
			error=hr.wError;

			if (!error)
				break;
		}

		sequence++;
		progress += p_inc;
		pc("Programming",progress/100);
	}

	if (!error) {

		HPI_MESSAGEX hm;
		HPI_RESPONSE hr;
		HW32 * tailer = (HW32 *)&hm.data;

		tailer[0]=total_checksum;
		tailer[1]=DspCode.dwBlockLength;
		tailer[2]=DspCode.dwVersion;

		HPI_InitMessage( (HPI_MESSAGE *)&hm,  HPI_OBJ_ADAPTER, HPI_ADAPTER_PROGRAM_FLASH);
		hm.msg.wAdapterIndex=wAdapterIndex;
		hm.msg.u.ax.program_flash.wSequence=sequence++;
		hm.msg.u.ax.program_flash.dwChecksum=calc_checksum(0, TAILERSIZE, tailer);
		hm.msg.u.ax.program_flash.wLength=TAILERSIZE;
		hm.msg.wSize=sizeof(HPI_MESSAGE)+TAILERSIZE*sizeof(HW32);

		hr.wSize=sizeof(hr);

		HPI_MessageEx(&hm,(HPI_RESPONSEX*)&hr,2000);
		delay(MSG_DELAY);

		error=hr.wError;
	}
	HpiDspCode_Close(&DspCode);
	pc("Done",100);
	return error;
}

HPI_ERR HPI_FileFirmwareGetInfo(const char * fwpath, unsigned int * adapter,
                  unsigned int *version, unsigned int *checksum, unsigned int *length)
{
	DSP_CODE DspCode;
	HPI_ERR error;

	HpiOs_SetDspCodePath(fwpath);

	error = HpiDspCode_Open(0x2400, &DspCode, NULL);
	if (error)
		return error;

	*version = DspCode.dwVersion;
	*checksum = DspCode.dwCrc;
	*length = DspCode.dwBlockLength;
	*adapter = DspCode.nAdapter;
	HpiDspCode_Close(&DspCode);
	return 0;
}

HPI_ERR HPI_AdapterFirmwareGetInfo(HW16 adapterIndex,enum eHpiFirmwareId fwid,
                  unsigned int *version, unsigned int *checksum, unsigned int *length)
{
    HPI_MESSAGE hm;
    HPI_RESPONSE hr;

	if ((fwid > HPI_FW_BOOTLOADER) || (fwid < HPI_FW_UPDATE))
		return HPI_ERROR_INVALID_OBJ_INDEX;

    HPI_InitMessage( &hm,  HPI_OBJ_ADAPTER, HPI_ADAPTER_QUERY_FLASH);
    hm.wAdapterIndex=adapterIndex;
    hm.u.ax.query_flash.dwOffset=fw[fwid-1].offset;

    HPI_MessageEx((HPI_MESSAGEX*)&hm,(HPI_RESPONSEX *)&hr,1000);

    *version = hr.u.ax.query_flash.dwVersion;
	*checksum = hr.u.ax.query_flash.dwChecksum;
	*length = hr.u.ax.query_flash.dwLength;

    return hr.wError;
}

HPI_ERR HPI_AdapterRestart(HW16 adapterIndex)
{
	HPI_MESSAGE hm;
	HPI_RESPONSE hr;
	// Send a special close message to the adapter to reset it
	HPI_InitMessage( &hm,  HPI_OBJ_ADAPTER, HPI_ADAPTER_CLOSE);
	hm.wAdapterIndex = adapterIndex;
	hm.u.a.wObjectType = 0xdead;
	hm.u.a.wAssertId = 0x0FF;
	HPI_MessageEx((HPI_MESSAGEX*)&hm, (HPI_RESPONSEX *)&hr, 10);
	return hr.wError;
}
