/******************************************************************************
Copyright (C) 1997-2003 AudioScience, Inc. All rights reserved.

This software is provided 'as-is', without any express or implied warranty.
In no event will AudioScience Inc. be held liable for any damages arising
from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
   claim that you wrote the original software. If you use this software
   in a product, an acknowledgment in the product documentation would be
   appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.
3. This copyright notice and list of conditions may not be altered or removed 
   from any source distribution.

AudioScience, Inc. <support@audioscience.com>

( This license is GPL compatible see http://www.gnu.org/licenses/license-list.html#GPLCompatibleLicenses )

 Hardware Programming Interface (HPI) header
 The HPI is a low-level hardware abstraction layer to all
 AudioScience digital audio adapters

 You must define one operating systems that the HPI is to be compiled under
 HPI_OS_WIN16         16bit Windows
 HPI_OS_WIN32_USER    32bit Windows
 HPI_OS_DSP_563XX     DSP563XX environment
 HPI_OS_DSP_C6000     DSP TI C6000 environment
 HPI_OS_WDM           Windows WDM kernel driver
 HPI_OS_LINUX         Linux kernel driver

(C) Copyright AudioScience Inc. 1998-2007
******************************************************************************/
#ifndef _HPI_H_
#define _HPI_H_

/* HPI Version
If HPI_VER_MINOR is odd then its a development release not intended for the public
If HPI_VER_MINOR is even then is a release version
i.e 3.05.02 is a development version
*/
#define HPI_VERSION_CONSTRUCTOR(maj,min,rel)  (((maj) <<16 ) + ((min) << 8) + (rel))

#define HPI_VER_MAJOR(v) ((v)>>16)
#define HPI_VER_MINOR(v) (((v)>>8) & 0xFF )
#define HPI_VER_RELEASE(v) ((v) & 0xFF )

/* Use single digits for versions less that 10 to avoid octal. */
#define HPI_VER HPI_VERSION_CONSTRUCTOR( 3L, 8, 6 )


#ifdef _DOX_ONLY_
/*****************************************************************************/
/** \defgroup compile_time_defines HPI compile time defines

This section descibes the usage of HPI defines to set the target compile environment.
The definitions are used in the build environment to control how HPI is built.
They are NOT predefined in hpi.h!
\{
*/
/** \defgroup os_type_defines HPI_OS_XXX build environment defines

Define exactly one of these depending on which OS you are compiling for.
Should we also include DSP_53XXX, C6000, WDM etc?

\{
*/
/**Define when creating a 16 bit Windows user application. */
#define HPI_OS_WIN16
/** Define when creating a 32 bit Windows user application. */
#define HPI_OS_WIN32_USER
/** Define when creating a Linux application or driver.*/
#define HPI_OS_LINUX
/* Should we also include DSP_53XXX, C6000, WDM etc? */
/**\}*/
/** \def HPI_64BIT
Define this when building a 64 bit application. When not defined a 32 bit environment will be assumed. Currently only supported under Linux (autodetected)
*/
#define HPI_64BIT

/** Define this to remove public definition of deprecated functions and defines.
Use this to reveal where the deprecated functionality is used
*/
#define HPI_EXCLUDE_DEPRECATED
/** \defgroup hpi_dll_defines HPI DLL function attributes
DLL environment defines.
\{
*/
#define HPIDLL_IMPORTS /**< Define when building an application that links to ASIHPI32.LIB (imports HPI functions from ASIHPI32.DLL). */
#define HPIDLL_EXPORTS /**< Define when creating an application that uses the HPI DLL. */
#define HPIDLL_STATIC /**< Define when building an application that compiles in HPIFUNC.C and does not use ASIHPI32.DLL */
/**\}*/

/** \defgroup hpi_dspcode_defines HPI DSP code loading method
Define exactly one of these to select how the DSP code is supplied to the adapter.

End users writing applications that use the HPI interface do not have to use any of the below
defines. They are only necessary for building drivers and 16 bit Windows applications.
\{
*/
#define HPI_DSPCODE_ARRAY /**< DSP code is supplied as a series of arrays that are linked into the driver/application. Only valid when \ref HPI_OS_WIN16 is defined. */
#define HPI_DSPCODE_FILE /**< DSP code is supplied as a file that is opened and read from by the driver. */
#define HPI_DSPCODE_FIRMWARE /**<  DSP code is read using the hotplug firmware loader module. Only valid when \ref HPI_OS_LINUX is defined. */
/**\}*/
/**\}*/
/*****************************************************************************/
#endif  /* #ifdef _DOX_ONLY_ */

#ifdef __cplusplus
extern "C" {
#endif

#define HPI_MAX_ADAPTER_MEM_SPACES (2) /**< maximum number of memory regions mapped to an adapter */

#include "hpios.h"

/* A few convenience macros */
#ifndef DEPRECATED
#define DEPRECATED
#endif

#ifndef ARRAY_SIZE
#define ARRAY_SIZE(A) (sizeof(A)/sizeof((A)[0]))
#endif

#ifndef __stringify
#define __stringify_1(x) #x
#define __stringify(x) __stringify_1(x)
#endif

#define HPI_UNUSED(param) (void)param

/******************************************************************************/
/******************************************************************************/
/********                     HPI API DEFINITIONS                     *********/
/******************************************************************************/
/******************************************************************************/

/* //////////////////////////////////////////////////////////////////////// */
/* BASIC TYPES */
/* HW8, HW16, HW32 MUST BE DEFINED IN HPIOS_xxx.H */

/* ////////////////////////////////////////////////////////////////////// */
/** \addtogroup hpi_defines HPI constant definitions
\{
*/

/*******************************************/
/** \defgroup adapter_ids Adapter types/product ids
\{
*/

#define HPI_ADAPTER_EVM6701             0x1002          /**< TI's C6701 EVM has this ID */
#define HPI_ADAPTER_DSP56301            0x1801          /**< DSP56301 rev A has this ID */
#define HPI_ADAPTER_PCI2040             0xAC60          /**< TI's PCI2040 PCI I/F chip has this ID */
#define HPI_ADAPTER_DSP6205             0xA106          /**< TI's C6205 PCI interface has this ID */

#define HPI_ADAPTER_FAMILY_MASK         0xff00          /**< First 2 hex digits define the adapter family */

/*#define HPI_ADAPTER_ASI1101             0x1101*/          /**< OBSOLETE - ASI1101 - 1 Stream MPEG playback */
/*#define HPI_ADAPTER_ASI1201             0x1201*/          /**< OBSOLETE - ASI1201 - 2 Stream MPEG playback */

#define HPI_ADAPTER_FAMILY_ASI1700		0x1700
#define HPI_ADAPTER_ASI1711             0x1711          /**< ASI1711 - Quad FM+RDS tuner module */
#define HPI_ADAPTER_ASI1721             0x1721          /**< ASI1721 - Quad AM/FM+RDS tuner module */
#define HPI_ADAPTER_ASI1731             0x1731          /**< ASI1731 - Quad TV tuner module */

/*#define HPI_ADAPTER_ASI2214             0x2214*/          /**< ASI2214 - USB 2.0 1xanalog in, 4 x analog out, 1 x AES in/out */

#define HPI_ADAPTER_ASI2416             0x2416          /**< ASI2416 - CobraNet peripheral */

/*#define HPI_ADAPTER_ASI4030             0x4030*/          /**< ASI4030 = PSI30 = OEM 3 Stereo playback */

#define HPI_ADAPTER_ASI4111             0x4111          /**< 2 play-1out, 1 rec PCM, MPEG*/
#define HPI_ADAPTER_ASI4113             0x4113          /**< 4 play-3out, 1 rec PCM, MPEG*/
#define HPI_ADAPTER_ASI4215             0x4215          /**< 4 play-5out, 1 rec, PCM, MPEG*/

#define HPI_ADAPTER_FAMILY_ASI4300		0x4300
#define HPI_ADAPTER_ASI4311             0x4311          /**< 4 play-1out, 1 rec-1in, PCM, MPEG*/
#define HPI_ADAPTER_ASI4312             0x4312          /**< 4 play-2out, 1 rec-1in, PCM, MPEG*/
#define HPI_ADAPTER_ASI4332             0x4332          /**< 4 play-2out, 1 rec-3in, PCM, MPEG*/
#define HPI_ADAPTER_ASI4334             0x4334          /**< 4 play-4out, 1 rec-3in, PCM, MPEG*/
#define HPI_ADAPTER_ASI4335             0x4335          /**< 4 play-4out, 1 rec-3in, PCM, MPEG, 8-relay, 16-opto*/
#define HPI_ADAPTER_ASI4336             0x4336          /**< 4 play-4out, 1 rec-3in, PCM, MPEG, 8-relay, 16-opto, RS422*/
#define HPI_ADAPTER_ASI4342             0x4342          /**< (ASI4312 with MP3) 4 play-2out, 1 rec-1in, PCM, MPEG-L2, MP3 */
#define HPI_ADAPTER_ASI4344             0x4344          /**< (ASI4334 with MP3)4 play-4out, 1 rec-3in, PCM, MPEG-L2, MP3 */
#define HPI_ADAPTER_ASI4346             0x4346          /**< (ASI4336 with MP3)4 play-4out, 1 rec-3in, PCM, MPEG-L2, MP3, 8-relay, 16-opto, RS422*/

/*#define HPI_ADAPTER_ASI4401             0x4401*/          /**< OEM 2 play, PCM mono/stereo, 44.1kHz*/

/*#define HPI_ADAPTER_ASI4501             0x4501*/          /**< OBSOLETE - OEM 4 play, 1 rec PCM, MPEG*/
/*#define HPI_ADAPTER_ASI4502             0x4502*/          /**< OBSOLETE - OEM 1 play, 1 rec PCM, MPEG*/
/*#define HPI_ADAPTER_ASI4503             0x4503*/          /**< OBSOLETE - OEM 4 play PCM, MPEG*/

/*#define HPI_ADAPTER_ASI4601             0x4601*/          /**< OEM 4 play PCM, MPEG & 1 record with AES-18 */
/*#define HPI_ADAPTER_ASI4701             0x4701*/          /**< OEM 24 mono play PCM with 512MB RAM */

#define HPI_ADAPTER_FAMILY_ASI5000		0x5000
#define HPI_ADAPTER_ASI5001             0x5001          /**< ASI5001 OEM, PCM only, 4 in, 1 out analog */
#define HPI_ADAPTER_ASI5002             0x5002          /**< ASI5002 OEM, PCM only, 4 in, 1 out analog and digital */
#define HPI_ADAPTER_ASI5020             0x5020          /**< ASI5020 PCM only, 2 analog only in/out */
#define HPI_ADAPTER_ASI5044             0x5044          /**< ASI5044 PCM only, 4 analog and digital in/out */
#define HPI_ADAPTER_ASI5041             0x5041          /**< ASI5041 PCM only, 4 digital only in/out */
#define HPI_ADAPTER_ASI5042             0x5042          /**< ASI5042 PCM only, 4 analog only in/out */

#define HPI_ADAPTER_FAMILY_ASI5100		0x5100
#define HPI_ADAPTER_ASI5101             0x5101          /**< ASI5101 OEM is ASI5111 with no mic. */
#define HPI_ADAPTER_ASI5111             0x5111          /**< ASI5111 PCM only */

#define HPI_ADAPTER_ASI6101             0x6101          /**< ASI6101 prototype */

#define HPI_ADAPTER_FAMILY_ASI6000		0x6000
#define HPI_ADAPTER_ASI6000             0x6000          /**< ASI6000 - generic 1 DSP adapter, exact config undefined */
#define HPI_ADAPTER_ASI6012             0x6012	        /**< ASI6012 - 1 in, 2 out analog only */
#define HPI_ADAPTER_ASI6022             0x6022          /**< ASI6022 - 2 in, 2 out analog only */
#define HPI_ADAPTER_ASI6044             0x6044          /**< ASI6044 - 4 in/out analog only */

#define HPI_ADAPTER_FAMILY_ASI6100		0x6100
#define HPI_ADAPTER_ASI6111             0x6111          /**< ASI6111 - 1 in/out, analog and AES3  */
#define HPI_ADAPTER_ASI6102             0x6102          /**< ASI6102 - 2out,analog and AES3  */
#define HPI_ADAPTER_ASI6113             0x6113          /**< 300MHz version of ASI6114 for testing*/
#define HPI_ADAPTER_ASI6122             0x6122          /**< ASI6122 - 2 in/out, analog and AES3  */
#define HPI_ADAPTER_ASI6114             0x6114          /**< ASI6114 - 4os,1is,4out,1in,analog and AES3  */
#define HPI_ADAPTER_ASI6118             0x6118          /**< ASI6118 - 8os,1is,8out,1in analog+AES3 */

#define HPI_ADAPTER_FAMILY_ASI6200		0x6200
#define HPI_ADAPTER_ASI6201             0x6201          /**< ASI6201 - OEM  */
#define HPI_ADAPTER_ASI6244             0x6244          /**< ASI6244 - 4os,4is,4out,4in,analog and AES3 */
#define HPI_ADAPTER_ASI6246             0x6246          /**< ASI6246 - 6os,2is,6out,4in,analog and AES3 */
#define HPI_ADAPTER_ASI6200             0x6200          /**< ASI6200 - generic 2 DSP adapter, exact config undefined */
#define HPI_ADAPTER_ASI6100             0x6100          /**< ASI6100 - generic 1 DSP adapter, exact config undefined */

#define HPI_ADAPTER_FAMILY_ASI6400		0x6400
#define HPI_ADAPTER_ASI6408             0x6408          /**< ASI6408 - cobranet PCI 8 mono in/out */
#define HPI_ADAPTER_ASI6416             0x6416          /**< ASI6416 - cobranet PCI 16 mono in/out */

#define HPI_ADAPTER_FAMILY_ASI6500		0x6500			/**< ASI6500 PCI sound cards */
#define HPI_ADAPTER_ASI6511             0x6511          /**< ASI6511 - 1 in/out, analog and AES3  */
#define HPI_ADAPTER_ASI6514             0x6514			/**< ASI6514 - ASI6114 replacement, 12os,2is,4out,1in,analog and AES3  */
#define HPI_ADAPTER_ASI6518             0x6518          /**< ASI6518 - ASI6118 replacement, 8os,1is,8out,1in analog+AES3 */
#define HPI_ADAPTER_ASI6520             0x6520			/**< ASI6520 - 6os,4is,2out,2in,analog only  */
#define HPI_ADAPTER_ASI6522             0x6522			/**< ASI6522 - 6os,4is,2out,2in,analog and AES3  */
#define HPI_ADAPTER_ASI6540             0x6540			/**< ASI6540 - 12os,8is,4out,4in,analog only  */
#define HPI_ADAPTER_ASI6544             0x6544			/**< ASI6544 - 12os,8is,4out,4in,analog and AES3  */
#define HPI_ADAPTER_ASI6548             0x6548			/**< ASI6548 - 16os,8is,8out,4in,analog and AES3  */
#define HPI_ADAPTER_ASI6585             0x6585			/**< ASI6585  - 8in, 8out, Livewire */

#define HPI_ADAPTER_FAMILY_ASI6600		0x6600			/**< ASI6600 PCI Express sound cards */
#define HPI_ADAPTER_ASI6611             0x6611          /**< ASI6611 - 1 in/out, analog and AES3  */
#define HPI_ADAPTER_ASI6614             0x6614			/**< ASI6614 - ASI6114 replacement, 12os,2is,4out,1in,analog and AES3  */
#define HPI_ADAPTER_ASI6618             0x6618          /**< ASI6618 - ASI6118 replacement, 8os,1is,8out,1in analog+AES3 */
#define HPI_ADAPTER_ASI6620             0x6620			/**< ASI6620 - 6os,4is,2out,2in,analog only  */
#define HPI_ADAPTER_ASI6622             0x6622			/**< ASI6622 - 6os,4is,2out,2in,analog and AES3  */
#define HPI_ADAPTER_ASI6640             0x6640			/**< ASI6640 - 12os,8is,4out,4in,analog only  */
#define HPI_ADAPTER_ASI6644             0x6644			/**< ASI6644 - 12os,8is,4out,4in,analog and AES3  */
#define HPI_ADAPTER_ASI6648             0x6648			/**< ASI6648 - 16os,8is,8out,4in,analog and AES3  */
#define HPI_ADAPTER_ASI6685             0x6685			/**< ASI6685  - 8in, 8out, Livewire */

/*#define HPI_ADAPTER_ASI8401             0x8401*/          /**< OEM 4 record */
/*#define HPI_ADAPTER_ASI8411             0x8411*/          /**< OEM RF switcher */

/*#define HPI_ADAPTER_ASI8601             0x8601*/          /**< OEM 8 record */

#define HPI_ADAPTER_FAMILY_ASI8700		0x8700
#define HPI_ADAPTER_ASI8701             0x8701          /**< OEM 8 record 2 AM/FM + 6 FM/TV , AM has 10kHz b/w*/
#define HPI_ADAPTER_ASI8702             0x8702          /**< 8 AM/FM record */
#define HPI_ADAPTER_ASI8703             0x8703          /**< 8 TV/FM record */
#define HPI_ADAPTER_ASI8704             0x8704          /**< standard product 2 AM/FM + 6 FM/TV */
#define HPI_ADAPTER_ASI8705             0x8705          /**< 4 TV/FM, 4 AM/FM record */
#define HPI_ADAPTER_ASI8706             0x8706          /**< 8 record 2 AM/FM + 6 FM/TV + 2 ext antenna jacks*/
#define HPI_ADAPTER_ASI8707             0x8707          /**< 8 record AM/FM - 4 ext antenna jacks */
#define HPI_ADAPTER_ASI8708             0x8708          /**< 8 record AM/FM - 6 ext antenna jacks */
#define HPI_ADAPTER_ASI8709             0x8709          /**< 8 record - no tuners */
#define HPI_ADAPTER_ASI8710             0x8710          /**< 8 record AM/FM - 1 ext antenna jacks*/
#define HPI_ADAPTER_ASI8711             0x8711          /**< 8 record AM/FM - 2 ext antenna jacks*/

#define HPI_ADAPTER_ASI8712             0x8712          /**< 4 record AM/FM */
#define HPI_ADAPTER_ASI8713             0x8713          /**< 4 record NTSC-TV/FM */

#define HPI_ADAPTER_ASI8722             0x8722          /**< 8 record 6xAM/FM+2xNTSC */
#define HPI_ADAPTER_ASI8723             0x8723          /**< 8 record NTSC */
#define HPI_ADAPTER_ASI8724             0x8724          /**< 4 record NTSC */
#define HPI_ADAPTER_ASI8725             0x8725          /**< 4 record 4xAM/FM+4xNTSC */

#define HPI_ADAPTER_ASI8732             0x8732          /**< 8 record 6xAM/FM+2xPAL */
#define HPI_ADAPTER_ASI8733             0x8733          /**< 8 record PAL */
#define HPI_ADAPTER_ASI8734             0x8734          /**< 4 record PAL */
#define HPI_ADAPTER_ASI8735             0x8735          /**< 4 record 4xAM/FM+4xPAL */

#define HPI_ADAPTER_ASI8801             0x8801          /**< OEM 8 record */

#define HPI_ADAPTER_FAMILY_ASI8900		0x8900
#define HPI_ADAPTER_ASI8920             0x8920          /**< 2 module tuner card */

#define HPI_ADAPTER_ILLEGAL             0xFFFF          /**< Used in DLL to indicate device not present */
/**\}*/

/*******************************************/
/** \defgroup formats Audio format types
\{
*/
#define HPI_FORMAT_MIXER_NATIVE         0   /**< Used internally on adapter. */
#define HPI_FORMAT_PCM8_UNSIGNED        1	/**< 8-bit unsigned PCM. Windows equivalent is WAVE_FORMAT_PCM. */
#define HPI_FORMAT_PCM16_SIGNED         2   /**< 16-bit signed PCM. Windows equivalent is WAVE_FORMAT_PCM. */
#define HPI_FORMAT_MPEG_L1              3   /**< MPEG-1 Layer-1. */
/** MPEG-1 Layer-2.

Windows equivalent is WAVE_FORMAT_MPEG.

The following table shows what combinations of mode and bitrate are possible:

<table border=1 cellspacing=0 cellpadding=5>
<tr>
<td><p><b>Bitrate (kbs)</b></p></td>
<td><p><b>Mono</b></p></td>
<td><p><b>Stereo,<br>Joint Stereo or<br>Dual Channel</b></p></td>
</tr>
<tr><td>32</td><td align=center>X</td><td align=center>_</td></tr>
<tr><td>40</td><td align=center>_</td><td align=center>_</td></tr>
<tr><td>48</td><td align=center>X</td><td align=center>_</td></tr>
<tr><td>56</td><td align=center>X</td><td align=center>_</td></tr>
<tr><td>64</td><td align=center>X</td><td align=center>X</td></tr>
<tr><td>80</td><td align=center>X</td><td align=center>_</td></tr>
<tr><td>96</td><td align=center>X</td><td align=center>X</td></tr>
<tr><td>112</td><td align=center>X</td><td align=center>X</td></tr>
<tr><td>128</td><td align=center>X</td><td align=center>X</td></tr>
<tr><td>160</td><td align=center>X</td><td align=center>X</td></tr>
<tr><td>192</td><td align=center>X</td><td align=center>X</td></tr>
<tr><td>224</td><td align=center>_</td><td align=center>X</td></tr>
<tr><td>256</td><td align=center>-</td><td align=center>X</td></tr>
<tr><td>320</td><td align=center>-</td><td align=center>X</td></tr>
<tr><td>384</td><td align=center>_</td><td align=center>X</td></tr>
</table>
*/
#define HPI_FORMAT_MPEG_L2              4
/** MPEG-1 Layer-3.
Windows equivalent is WAVE_FORMAT_MPEG.

The following table shows what combinations of mode and bitrate are possible:

<table border=1 cellspacing=0 cellpadding=5>
<tr>
<td><p><b>Bitrate (kbs)</b></p></td>
<td><p><b>Mono<br>Stereo @ 8,<br>11.025 and<br>12kHz*</b></p></td>
<td><p><b>Mono<br>Stereo @ 16,<br>22.050 and<br>24kHz*</b></p></td>
<td><p><b>Mono<br>Stereo @ 32,<br>44.1 and<br>48kHz</b></p></td>
</tr>
<tr><td>8</td><td align=center>X</td><td align=center>X</td></tr><td align=center>_</td></tr>
<tr><td>16</td><td align=center>X</td><td align=center>X</td></tr><td align=center>_</td></tr>
<tr><td>24</td><td align=center>X</td><td align=center>X</td></tr><td align=center>_</td></tr>
<tr><td>32</td><td align=center>X</td><td align=center>X</td></tr><td align=center>X</td></tr>
<tr><td>40</td><td align=center>X</td><td align=center>X</td></tr><td align=center>X</td></tr>
<tr><td>48</td><td align=center>X</td><td align=center>X</td></tr><td align=center>X</td></tr>
<tr><td>56</td><td align=center>X</td><td align=center>X</td></tr><td align=center>X</td></tr>
<tr><td>64</td><td align=center>X</td><td align=center>X</td></tr><td align=center>X</td></tr>
<tr><td>80</td><td align=center>_</td><td align=center>X</td></tr><td align=center>X</td></tr>
<tr><td>96</td><td align=center>_</td><td align=center>X</td></tr><td align=center>X</td></tr>
<tr><td>112</td><td align=center>_</td><td align=center>X</td></tr><td align=center>X</td></tr>
<tr><td>128</td><td align=center>_</td><td align=center>X</td></tr><td align=center>X</td></tr>
<tr><td>144</td><td align=center>_</td><td align=center>X</td></tr><td align=center>_</td></tr>
<tr><td>160</td><td align=center>_</td><td align=center>X</td></tr><td align=center>X</td></tr>
<tr><td>192</td><td align=center>_</td><td align=center>_</td></tr><td align=center>X</td></tr>
<tr><td>224</td><td align=center>_</td><td align=center>_</td></tr><td align=center>X</td></tr>
<tr><td>256</td><td align=center>-</td><td align=center>_</td></tr><td align=center>X</td></tr>
<tr><td>320</td><td align=center>-</td><td align=center>_</td></tr><td align=center>X</td></tr>
</table>
\b * Available on the ASI6000 series only
*/
#define HPI_FORMAT_MPEG_L3              5
#define HPI_FORMAT_DOLBY_AC2            6   /**< Dolby AC-2. */
#define HPI_FORMAT_DOLBY_AC3            7   /**< Dolbt AC-3. */
#define HPI_FORMAT_PCM16_BIGENDIAN      8   /**< 16-bit PCM big-endian. */
#define HPI_FORMAT_AA_TAGIT1_HITS       9   /**< TAGIT-1 algorithm - hits. */
#define HPI_FORMAT_AA_TAGIT1_INSERTS    10  /**< TAGIT-1 algorithm - inserts. */
/** 32-bit signed PCM. Windows equivalent is WAVE_FORMAT_PCM. Each sample is a 32bit word.
The most significant 24 bits contain a 24-bit sample and the least significant 8 bits are
set to 0.
*/
#define HPI_FORMAT_PCM32_SIGNED         11
#define HPI_FORMAT_RAW_BITSTREAM        12  /**< Raw bitstream - unknown format. */
#define HPI_FORMAT_AA_TAGIT1_HITS_EX1   13  /**< TAGIT-1 algorithm hits - extended. */
/** 32-bit PCM as an IEEE float. Windows equivalent is WAVE_FORMAT_IEEE_FLOAT. Each sample is
a 32bit word in IEEE754 floating point format. The range is +1.0 to -1.0, which corresponds
to digital fullscale.
*/
#define HPI_FORMAT_PCM32_FLOAT          14
#define HPI_FORMAT_PCM24_SIGNED         15  /**< 24-bit PCM signed. Windows equivalent is WAVE_FORMAT_PCM. */
#define HPI_FORMAT_OEM1                 16  /**< OEM format 1 - private. */
#define HPI_FORMAT_OEM2                 17  /**< OEM format 2 - private. */
#define HPI_FORMAT_UNDEFINED            (0xffff)  /**< Undefined format. */
/**\}*/

/******************************************* bus types */
#define HPI_BUS_ISAPNP                  1
#define HPI_BUS_PCI                     2
#define HPI_BUS_USB                     3

/******************************************* in/out Stream states */
/*******************************************/
/** \defgroup stream_states Stream States
\{
*/
#define HPI_STATE_STOPPED               1	/**< State stopped - stream is stopped. */
#define HPI_STATE_PLAYING               2	/**< State playing - stream is playing audio. */
#define HPI_STATE_RECORDING             3	/**< State recording - stream is recording. */
#define HPI_STATE_DRAINED               4	/**< State drained - playing stream ran out of data to play. */
#define HPI_STATE_SINEGEN               5	/**< State generate sine - to be implemented. */
/**\}*/
/******************************************* mixer source node types */
/** \defgroup source_nodes Source node types
\{
*/
#define HPI_SOURCENODE_BASE             100
#define HPI_SOURCENODE_OSTREAM          101	/**< Out Stream (Play) node. */
#define HPI_SOURCENODE_LINEIN           102 /**< Line in node. */
#define HPI_SOURCENODE_AESEBU_IN        103 /**< AES/EBU input node. */
#define HPI_SOURCENODE_TUNER            104 /**< Tuner node. */
#define HPI_SOURCENODE_RF               105 /**< RF input node. */
#define HPI_SOURCENODE_CLOCK_SOURCE     106 /**< Clock source node. */
#define HPI_SOURCENODE_RAW_BITSTREAM    107 /**< Raw bitstream node. */
#define HPI_SOURCENODE_MICROPHONE    	108 /**< Microphone node. */
#define HPI_SOURCENODE_COBRANET    		109 /**< Cobranet input node. Audio samples come from the Cobranet network and into the device. */
/*! Update this if you add a new sourcenode type, AND hpidebug.h */
#define HPI_SOURCENODE_LAST_INDEX         109
/* AX4 max sourcenode type = 15 */
/* AX6 max sourcenode type = 15 */
/**\}*/


/******************************************* mixer dest node types */
/** \defgroup dest_nodes Destination node types
\{
*/
#define HPI_DESTNODE_BASE               200
#define HPI_DESTNODE_ISTREAM            201 /**< In Stream (Record) node. */
#define HPI_DESTNODE_LINEOUT            202	/**< Line Out node. */
#define HPI_DESTNODE_AESEBU_OUT         203 /**< AES/EBU output node. */
#define HPI_DESTNODE_RF                 204 /**< RF output node. */
#define HPI_DESTNODE_SPEAKER            205 /**< Speaker output node. */
#define HPI_DESTNODE_COBRANET    		206 /**< Cobranet output node. Audio samples from the device are sent out on the Cobranet network.*/
/*! Update this if you add a new destnode type. , AND hpidebug.h  */
#define HPI_DESTNODE_LAST_INDEX         206
/* AX4 max destnode type = 7 */
/* AX6 max destnode type = 15 */
/**\}*/


/*******************************************/
/** \defgroup control_types Mixer control types
\{
*/
#define HPI_CONTROL_GENERIC             0	/**< Generic control. */
#define HPI_CONTROL_CONNECTION          1	/**< Connection control for connecting source and destination nodes. */
#define HPI_CONTROL_VOLUME              2	/**< Volume control - works in dBFs. */
#define HPI_CONTROL_METER               3	/**< Peak meter control. */
#define HPI_CONTROL_MUTE                4	/**< Mute control - not used at present. */
#define HPI_CONTROL_MULTIPLEXER         5	/**< Multiplexer control. */
#define HPI_CONTROL_AESEBU_TRANSMITTER  6	/**< AES/EBU transmitter control. */
#define HPI_CONTROL_AESEBU_RECEIVER     7	/**< AES/EBU receiver control. */
#define HPI_CONTROL_LEVEL               8	/**< Level/trim control - works in dBu. */
#define HPI_CONTROL_TUNER               9	/**< Tuner control. */
/*#define HPI_CONTROL_ONOFFSWITCH         10 */
#define HPI_CONTROL_VOX                 11	/**< Vox control. */
#define HPI_CONTROL_AES18_TRANSMITTER   12	/**< AES-18 transmitter control. */
#define HPI_CONTROL_AES18_RECEIVER      13	/**< AES-18 receiver control. */
#define HPI_CONTROL_AES18_BLOCKGENERATOR   14	/**< AES-18 block generator control. */
#define HPI_CONTROL_CHANNEL_MODE   		15	/**< Channel mode control. */
/* WARNING types 16 or greater impact bit packing in AX4100 and AX4500 DSP code */
#define HPI_CONTROL_BITSTREAM      		16	/**< Bitstream control. */
#define HPI_CONTROL_SAMPLECLOCK			17	/**< Sample clock control. */
#define HPI_CONTROL_MICROPHONE			18	/**< Microphone control. */
#define HPI_CONTROL_PARAMETRIC_EQ		19	/**< Parametric EQ control. */
#define HPI_CONTROL_COMPANDER			20	/**< Compander control. */
#define HPI_CONTROL_COBRANET			21	/**< Cobranet control. */
#define HPI_CONTROL_TONEDETECTOR 		22	/**< Tone detector control. */
#define HPI_CONTROL_SILENCEDETECTOR     23	/**< Silence detector control. */

/*! Update this if you add a new control type. , AND hpidebug.h */
#define HPI_CONTROL_LAST_INDEX			23

/* WARNING types 32 or greater impact bit packing in all AX4 DSP code */
/* WARNING types 256 or greater impact bit packing in all AX6 DSP code */
/**\}*/

/******************************************* ADAPTER ATTRIBUTES ****/


/** \defgroup adapter_properties Adapter properties
These are used in HPI_AdapterSetProperty() and HPI_AdapterGetProperty()
\{
*/
/** \internal Used in dwProperty field of HPI_AdapterSetProperty() and HPI_AdapterGetProperty(). This errata applies to all
ASI6000 cards with both analog and digital outputs. The CS4224 A/D+D/A has a one sample
delay between left and right channels on both its input (ADC) and output (DAC). More details are available in
Cirrus Logic errata ER284B2.PDF available from www.cirrus.com, released by
Cirrus in 2001.
*/
#define HPI_ADAPTER_PROPERTY_ERRATA_1	(1)

/** Adapter grouping property
Indicates whether the adapter supports the grouping API (for ASIO and SSX2)
*/
#define HPI_ADAPTER_PROPERTY_GROUPING		(2)

/** Adapter SSX2 property
Indicates whether the adapter supports SSX2 multichannel streams
*/
#define HPI_ADAPTER_PROPERTY_ENABLE_SSX2	(3)

#define HPI_ADAPTER_PROPERTY_READONLYBASE  (256)	/**< Base number for readonly properties */
/** Readonly adapter latency property.
This property returns in the input and output latency in samples.Property 1 is the estimated input latency
in samples, while Property 2 is that output latency in  samples.
*/
#define HPI_ADAPTER_PROPERTY_LATENCY    		(HPI_ADAPTER_PROPERTY_READONLYBASE+0)
/** Readonly adapter granularity property.
The granulariy is the smallest size chunk of stereo samples that is processed by
the adapter. This property returns the record granularity in samples in Property 1. Property 2 returns the
play granularity.
*/
#define HPI_ADAPTER_PROPERTY_GRANULARITY    	(HPI_ADAPTER_PROPERTY_READONLYBASE+1)
/** Readonly adapter number of current channels property.
The CURCHANNELs property returns is the number of record and playback channels per device. Property 1 is the number
of record channels per record device, while Property 2 is the number of play channels per playback device.*/
#define HPI_ADAPTER_PROPERTY_CURCHANNELS     	(HPI_ADAPTER_PROPERTY_READONLYBASE+2)

/** Readonly adapter software version.
The SOFTWARE_VERSION property returns the version of the software running on the adapter as Major.Minor.Release.
Property 1 contains Major in bits 15..8 and Minor in bits 7..0.  Property 2 contains Release in bits 7..0. */
#define HPI_ADAPTER_PROPERTY_SOFTWARE_VERSION     (HPI_ADAPTER_PROPERTY_READONLYBASE+3)
/**\}*/


#define HPI_ADAPTER_MODE_SET 	(0)	/**< Used in wQueryOrSet field of HPI_AdapterSetModeEx(). */
#define HPI_ADAPTER_MODE_QUERY (1)/**< Used in wQueryOrSet field of HPI_AdapterSetModeEx(). */

/** \defgroup adapter_modes Adapter Modes
    These are used by HPI_AdapterSetModeEx()

\warning - more than 16 possible modes breaks a bitmask in the Windows WAVE DLL
\{
*/
/** 4 outstream mode.
- ASI6114: 1 instream
- ASI6044: 4 instreams
- ASI6012: 1 instream
- ASI6102: no instreams
- ASI6022, ASI6122: 2 instreams
- ASI5111, ASI5101: 2 instreams
- ASI652x, ASI662x: 2 instreams
- ASI654x, ASI664x: 4 instreams
*/
#define HPI_ADAPTER_MODE_4OSTREAM (1)

/** 6 outstream mode.
- ASI6012: 1 instream,
- ASI6022, ASI6122: 2 instreams
- ASI652x, ASI662x: 4 instreams
*/
#define HPI_ADAPTER_MODE_6OSTREAM (2)

/** 8 outstream mode.
- ASI6114: 8 instreams
- ASI6118: 8 instreams
- ASI6585: 8 instreams
*/
#define HPI_ADAPTER_MODE_8OSTREAM (3)

/** 16 outstream mode.
- ASI6416 16 instreams
- ASI6518, ASI6618 16 instreams
- ASI6118 16 mono out and in streams
*/
#define HPI_ADAPTER_MODE_16OSTREAM (4)

/** one outstream mode.
- ASI5111 1 outstream, 1 instream
*/
#define HPI_ADAPTER_MODE_1OSTREAM (5)

/** ASI504X mode 1. 12 outstream, 4 instream 0 to 48kHz sample rates
    (see ASI504X datasheet for more info).
*/
#define HPI_ADAPTER_MODE_1 (6)

/** ASI504X mode 2. 4 outstreams, 4 instreams at 0 to 192kHz sample rates
    (see ASI504X datasheet for more info).
*/
#define HPI_ADAPTER_MODE_2 (7)

/** ASI504X mode 3. 4 outstreams, 4 instreams at 0 to 192kHz sample rates
    (see ASI504X datasheet for more info).
*/
#define HPI_ADAPTER_MODE_3 (8)

/** ASI504X multichannel mode. 2 outstreams -> 4 line outs (1 to 8 channel streams),
    4 lineins -> 1 instream (1 to 8 channel streams) at 0-48kHz.
    For more info see the SSX Specification.
*/
#define HPI_ADAPTER_MODE_MULTICHANNEL (9)

/** 12 outstream mode.
- ASI6514, ASI6614: 2 instreams
- ASI6540,ASI6544: 8 instreams
- ASI6640,ASI6644: 8 instreams
*/
#define HPI_ADAPTER_MODE_12OSTREAM (10)

/** 9 outstream mode.
- ASI6044: 8 instreams
*/
 #define HPI_ADAPTER_MODE_9OSTREAM (11)

/**\}*/

/* Note, adapters can have more than one capability - encoding as bitfield is recommended. */
#define HPI_CAPABILITY_NONE  (0)
#define HPI_CAPABILITY_MPEG_LAYER3  (1)
/* Set this equal to maximum capability index, Must not be greater than 32 - see axnvdef.h */
#define HPI_CAPABILITY_MAX          1
/* #define HPI_CAPABILITY_AAC          2 */

/******************************************* STREAM ATTRIBUTES ****/

/* Ancillary Data modes */
#define HPI_MPEG_ANC_HASENERGY	(0)
#define HPI_MPEG_ANC_RAW		(1)
#define HPI_MPEG_ANC_ALIGN_LEFT (0)
#define HPI_MPEG_ANC_ALIGN_RIGHT (1)

/** \defgroup mpegmodes MPEG modes
\{
MPEG modes - can be used optionally for HPI_FormatCreate() parameter dwAttributes.

The operation of the below modes varies acCording to the number of channels. Using HPI_MPEG_MODE_DEFAULT
causes the MPEG-1 Layer II bitstream to be recorded in single_channel mode when the number
of channels is 1 and in stereo when the number of channels is 2. Using any mode setting other
than HPI_MPEG_MODE_DEFAULT when the number of channels is set to 1 will return an error.
*/
#define HPI_MPEG_MODE_DEFAULT	(0)
#define HPI_MPEG_MODE_STEREO	(1)
#define HPI_MPEG_MODE_JOINTSTEREO	(2)
#define HPI_MPEG_MODE_DUALCHANNEL	(3)
/** \} */

/* Locked memory buffer alloc/free phases */
#define HPI_BUFFER_CMD_EXTERNAL					0       /**< use one message to allocate or free physical memory */
#define HPI_BUFFER_CMD_INTERNAL_ALLOC			1       /**< alloc physical memory */
#define HPI_BUFFER_CMD_INTERNAL_GRANTADAPTER	2       /**< send physical memory address to adapter */
#define HPI_BUFFER_CMD_INTERNAL_REVOKEADAPTER	3       /**< notify adapter to stop using physical buffer */
#define HPI_BUFFER_CMD_INTERNAL_FREE			4       /**< free physical buffer */

/******************************************* MIXER ATTRIBUTES ****/

/** \defgroup mixer_flags Mixer flags used in processing function HPI_MIXER_GET_CONTROL_MULTIPLE_VALUES
\{
*/
#define HPI_MIXER_GET_CONTROL_MULTIPLE_CHANGED (0)
#define HPI_MIXER_GET_CONTROL_MULTIPLE_RESET (1)
/**\}*/


/** Commands used by HPI_MixerStore()
*/
enum HPI_MIXER_STORE_COMMAND {
	HPI_MIXER_STORE_SAVE         =1,    /**< Save all mixer control settings. */
	HPI_MIXER_STORE_RESTORE      =2,    /**< Restore all controls from saved. */
	HPI_MIXER_STORE_DELETE       =3,    /**< Delete saved control settings. */
	HPI_MIXER_STORE_ENABLE       =4,    /**< Enable auto storage of some control settings. */
	HPI_MIXER_STORE_DISABLE      =5,    /**< Disable auto storage of some control settings. */
	HPI_MIXER_STORE_SAVE_SINGLE  =6     /**< Save the attributes of a single control. */
};


/******************************************* CONTROL ATTRIBUTES ****/
/* (in order of control type ID as above */

/* This allows for 255 control types, 255 unique attributes each */
#ifndef HPI_OS_DSP_563XX
/*#define HPI_CONTROL_SPACING (0x100)*/
#define HPI_MAKE_ATTRIBUTE(obj,index) (obj*0x100+index)
#endif

/* Generic control attributes.  If a control uses any of these attributes
   its other attributes must also be defined using HPI_MAKE_ATTRIBUTE()
*/

/** Enable a control.
0=disable, 1=enable
\note generic to all mixer plugins?
*/
/*#define HPI_CONTROL_ENABLE     HPI_MAKE_ATTRIBUTE(HPI_CONTROL_GENERIC,1)*/
#define HPI_GENERIC_ENABLE     HPI_MAKE_ATTRIBUTE(HPI_CONTROL_GENERIC,1)


/** Enable event generation for a control.
0=disable, 1=enable
\note generic to all controls that can generate events
*/
/*#define HPI_EVENT_ENABLE       HPI_MAKE_ATTRIBUTE(HPI_CONTROL_GENERIC,2)*/
#define HPI_GENERIC_EVENT_ENABLE       HPI_MAKE_ATTRIBUTE(HPI_CONTROL_GENERIC,2)

/* Volume Control attributes */
#define HPI_VOLUME_GAIN             1
#define HPI_VOLUME_RANGE            10	/* make this very different from the other HPI_VOLUME_ defines below */

/* Volume Control attributes */
#define HPI_LEVEL_GAIN             1


#define HPI_VOLUME_AUTOFADE_LOG     2  /**< log fade - dB attenuation changes linearly over time */
#define HPI_VOLUME_AUTOFADE         HPI_VOLUME_AUTOFADE_LOG
#define HPI_VOLUME_AUTOFADE_LINEAR  3  /**< linear fade - amplitude changes linearly */
#define HPI_VOLUME_AUTOFADE_1       4  /**< Not implemented yet -may be special profiles */
#define HPI_VOLUME_AUTOFADE_2       5
#define HPI_VOLUME_NUM_CHANNELS     6  /**< Used in HPI_ControlQuery() to get the number of channels a volume control has. */

/* Volume control special gain values */
#define HPI_UNITS_PER_dB			(100)		/**< volumes units are 100ths of a dB */
#define HPI_GAIN_OFF                (-100*HPI_UNITS_PER_dB)  /**< turns volume control OFF or MUTE */
#define HPI_METER_MINIMUM           (-150*HPI_UNITS_PER_dB)  /**< value returned for no signal */

/* Meter Control attributes */
#define HPI_METER_RMS               1       /**< return RMS signal level */
#define HPI_METER_PEAK              2       /**< return peak signal level */
#define HPI_METER_RMS_BALLISTICS    3       /**< ballistics for ALL rms meters on adapter */
#define HPI_METER_PEAK_BALLISTICS   4       /**< ballistics for ALL peak meters on adapter */
#define HPI_METER_NUM_CHANNELS      5       /**< Used in HPI_ControlQuery() to get the number of channels a meter control has. */

/* Multiplexer control attributes */
#define HPI_MULTIPLEXER_SOURCE      1
#define HPI_MULTIPLEXER_QUERYSOURCE 2

/** AES/EBU control attributes */
#define HPI_AESEBU_FORMAT           1 /**< AESEBU or SPDIF */

#ifndef HPI_EXCLUDE_DEPRECATED
#define HPI_AESEBU_SOURCE           HPI_AESEBU_FORMAT /**< \deprecated old name, use HPI_AESEBU_FORMAT instead */
#endif

#define HPI_AESEBU_ERRORSTATUS      2
#define HPI_AESEBU_SAMPLERATE       3
#define HPI_AESEBU_CHANNELSTATUS    4
#define HPI_AESEBU_USERDATA         5

/** AES/EBU physical format - AES/EBU balanced "professional"  */
#define HPI_AESEBU_FORMAT_AESEBU    1
/** AES/EBU physical format - S/PDIF unbalanced "consumer"  */
#define HPI_AESEBU_FORMAT_SPDIF     2

#ifndef HPI_EXCLUDE_DEPRECATED
#define HPI_AESEBU_SOURCE_AESEBU    HPI_AESEBU_FORMAT_AESEBU /**< \deprecated use HPI_AESEBU_FORMAT_AESEBU */
#define HPI_AESEBU_SOURCE_SPDIF     HPI_AESEBU_FORMAT_SPDIF /**< \deprecated use HPI_AESEBU_FORMAT_SPDIF */
#endif

/** \defgroup aesebu_errors AES/EBU error status bits
\{ */

#define HPI_AESEBU_ERROR_NOT_LOCKED     0x01    /**<  bit0: 1 when PLL is not locked */
#define HPI_AESEBU_ERROR_POOR_QUALITY   0x02    /**<  bit1: 1 when signal quality is poor */
#define HPI_AESEBU_ERROR_PARITY_ERROR   0x04    /**< bit2: 1 when there is a parity error */
#define HPI_AESEBU_ERROR_BIPHASE_VIOLATION  0x08    /**<  bit3: 1 when there is a bi-phase coding violation */
#define HPI_AESEBU_ERROR_VALIDITY       0x10    /**<  bit4: 1 when the validity bit is high */
/**\}*/

/** \defgroup tuner_defs Tuners
\{
*/
/** \defgroup tuner_attrs Tuner control attributes
\{
*/
#define HPI_TUNER_BAND                  1
#define HPI_TUNER_FREQ                  2
#define HPI_TUNER_LEVEL                 3
#define HPI_TUNER_AUDIOMUTE             4
#define HPI_TUNER_VIDEO_STATUS          5          /* AGE 11/10/97 <------------ use TUNER_STATUS instead */
#define HPI_TUNER_GAIN		            6				/* AGE 09/03/03 */
#define HPI_TUNER_STATUS		        7				/* SGT JUN-08-04 */
#define HPI_TUNER_MODE                  8
#define HPI_TUNER_RDS                   9		///< RDS data.
/** \} */

/** \defgroup tuner_bands Tuner bands

Used for HPI_Tuner_SetBand(),HPI_Tuner_GetBand()
\{
*/
#define HPI_TUNER_BAND_AM               1    /**< AM band */
#define HPI_TUNER_BAND_FM               2    /**< FM band (mono) */
#define HPI_TUNER_BAND_TV               3    /**< TV band = NTSC-M*/
#define HPI_TUNER_BAND_FM_STEREO        4    /**< FM band (stereo) */
#define HPI_TUNER_BAND_AUX              5    /**< Auxiliary input */
#define HPI_TUNER_BAND_TV_NTSC_M        HPI_TUNER_BAND_TV    /**< NTSC-M TV band*/
#define HPI_TUNER_BAND_TV_PAL_BG        6    /**< PAL-B/G TV band*/
#define HPI_TUNER_BAND_TV_PAL_I         7    /**< PAL-I TV band*/
#define HPI_TUNER_BAND_TV_PAL_DK        8    /**< PAL-D/K TV band*/
#define HPI_TUNER_BAND_TV_SECAM_L       9    /**< SECAM-L TV band*/
#define HPI_TUNER_BAND_LAST				9 /**< The index of the last tuner band. */
/** \} */

/* Tuner mode attributes */
#define HPI_TUNER_MODE_RSS              1	/**< Tuner mode attribute RSS */

/* RSS attribute values */
#define HPI_TUNER_MODE_RSS_DISABLE      0	/**< Tuner mode attribute RSS disable */
#define HPI_TUNER_MODE_RSS_ENABLE       1	/**< Tuner mode attribute RSS enable */

/** Tuner Level settings */
#define HPI_TUNER_LEVEL_AVERAGE         0
#define HPI_TUNER_LEVEL_RAW   			1

/** Tuner video status */
#define HPI_TUNER_VIDEO_COLOR_PRESENT               0x0001	/**< Video color is present. */
#define HPI_TUNER_VIDEO_IS_60HZ                     0x0020	/**< 60 Hz video detected. */
#define HPI_TUNER_VIDEO_HORZ_SYNC_MISSING           0x0040	/**< Video HSYNC is missing. */
#define HPI_TUNER_VIDEO_STATUS_VALID                0x0100	/**< Video status is valid. */
#define HPI_TUNER_PLL_LOCKED						0x1000	/**< The tuner's PLL is locked. */
#define HPI_TUNER_FM_STEREO							0x2000	/**< Tuner reports back FM stereo. */
/** \} */

/* Used by mixer plugin enable functions */
#define HPI_SWITCH_OFF              0	/**< Turn the mixer plugin on. */
#define HPI_SWITCH_ON               1	/**< Turn the mixer plugin off. */

/* VOX control attributes */
#define HPI_VOX_THRESHOLD           1       /* AGE 9/10/98 */

/** \defgroup channel_modes Channel Modes
Used for HPI_ChannelModeSet/Get()
\{
*/
#define HPI_CHANNEL_MODE_NORMAL     		1   /**< Left channel out = left channel in, Right channel out = right channel in. */
#define HPI_CHANNEL_MODE_SWAP       		2	/**< Left channel out = right channel in, Right channel out = left channel in. */
#define HPI_CHANNEL_MODE_LEFT_TO_STEREO		3	/**< Left channel out = left channel in, Right channel out = left channel in. */
#define HPI_CHANNEL_MODE_RIGHT_TO_STEREO	4	/**< Left channel out = right channel in, Right channel out = right channel in. */
#define HPI_CHANNEL_MODE_STEREO_TO_LEFT		5	/**< Left channel out = (left channel in + right channel in)/2, Right channel out = mute. */
#define HPI_CHANNEL_MODE_STEREO_TO_RIGHT 	6	/**< Left channel out = mute, Right channel out = (right channel in + left channel in)/2. */
#define HPI_CHANNEL_MODE_LAST				6
/** \} */

/* Bitstream control set attributes */
#define HPI_BITSTREAM_DATA_POLARITY     1
#define HPI_BITSTREAM_CLOCK_EDGE        2
#define HPI_BITSTREAM_CLOCK_SOURCE      3

#define HPI_POLARITY_POSITIVE           0
#define HPI_POLARITY_NEGATIVE           1

/*
Currently fixed at EXTERNAL
#define HPI_SOURCE_EXTERNAL             0
#define HPI_SOURCE_INTERNAL             1
*/

/* Bitstream control get attributes */
#define HPI_BITSTREAM_ACTIVITY       1

/* SampleClock control attributes */
#define HPI_SAMPLECLOCK_SOURCE			1
#define HPI_SAMPLECLOCK_SAMPLERATE		2
#define HPI_SAMPLECLOCK_SOURCE_INDEX	3
#define HPI_SAMPLECLOCK_LOCAL_SAMPLERATE	4

/** \defgroup sampleclock_source SampleClock source values
\{
*/
#define HPI_SAMPLECLOCK_SOURCE_ADAPTER		1 /**< The adapter is clocked from its on-board samplerate generator. When this source is set, the adapter samplerate may be set using HPI_SampleClock_SetSampleRate(). */

#define HPI_SAMPLECLOCK_SOURCE_AESEBU_SYNC	2 /**< The adapter is clocked from a dedicated AES/EBU SampleClock input.  */

#ifndef HPI_EXCLUDE_DEPRECATED
/**
 * \deprecated HPI_SAMPLECLOCK_SOURCE_AESEBU_SYNC should be used instead
 */
#define HPI_SAMPLECLOCK_SOURCE_AESEBU HPI_SAMPLECLOCK_SOURCE_AESEBU_SYNC
#endif

#define HPI_SAMPLECLOCK_SOURCE_WORD			3	/**< From external connector */
#define HPI_SAMPLECLOCK_SOURCE_WORD_HEADER	4	/**< Board-to-board header */
#define HPI_SAMPLECLOCK_SOURCE_SMPTE		5   /**< FUTURE - SMPTE clcok. */
#define HPI_SAMPLECLOCK_SOURCE_AESEBU_INPUT	6	/**< One of the aesebu inputs */
#define HPI_SAMPLECLOCK_SOURCE_AESEBU_AUTO	7	/**< The first aesebu input with a valid signal */
#define HPI_SAMPLECLOCK_SOURCE_COBRANET		8	/**< From the Cobranet interface at either 48 or 96kHz*/
#define HPI_SAMPLECLOCK_SOURCE_LOCAL        9   /**< From local clock source on module (ASI2416 only) */
#define HPI_SAMPLECLOCK_SOURCE_PREV_MODULE 10   /**< From previous adjacent module 9ASI2416 only)*/
/*! Update this if you add a new clock source2.*/
#define HPI_SAMPLECLOCK_SOURCE_LAST			10
/** \} */

/* Microphone control attributes */
#define HPI_MICROPHONE_PHANTOM_POWER		1

/** Equalizer control attributes
*/
#define HPI_EQUALIZER_NUM_FILTERS	1  /*!< Used to get number of filters in an EQ. (Can't set) */
#define HPI_EQUALIZER_FILTER		2  /*!< Set/get the filter by type, freq, Q, gain */
#define HPI_EQUALIZER_COEFFICIENTS	3  /*!< Get the biquad coefficients */

/** Equalizer filter types. Used by HPI_ParametricEQ_SetBand() */
enum HPI_FILTER_TYPE {
	HPI_FILTER_TYPE_BYPASS    = 0, /*!< Filter is turned off */

	HPI_FILTER_TYPE_LOWSHELF  = 1, /*!< EQ low shelf */
	HPI_FILTER_TYPE_HIGHSHELF = 2, /*!< EQ high shelf */
	HPI_FILTER_TYPE_EQ_BAND   = 3, /*!< EQ gain */

	HPI_FILTER_TYPE_LOWPASS   = 4, /*!< Standard low pass */
	HPI_FILTER_TYPE_HIGHPASS  = 5, /*!< Standard high pass */
	HPI_FILTER_TYPE_BANDPASS  = 6, /*!< Standard band pass */
	HPI_FILTER_TYPE_BANDSTOP  = 7  /*!< Standard band stop/notch */
};

/* Cobranet control attributes. MUST be distinct from all other control attributes.
   This is so that host side processing can easily identify a Cobranet control and
   apply additional host side operations (like copying data) as required.
*/
#ifndef HPI_OS_DSP_563XX
#define HPI_COBRANET_SET         HPI_MAKE_ATTRIBUTE(HPI_CONTROL_COBRANET,1)
#define HPI_COBRANET_GET         HPI_MAKE_ATTRIBUTE(HPI_CONTROL_COBRANET,2)
#define HPI_COBRANET_SET_DATA    HPI_MAKE_ATTRIBUTE(HPI_CONTROL_COBRANET,3)
#define HPI_COBRANET_GET_DATA    HPI_MAKE_ATTRIBUTE(HPI_CONTROL_COBRANET,4)
#define HPI_COBRANET_GET_STATUS  HPI_MAKE_ATTRIBUTE(HPI_CONTROL_COBRANET,5)
#define HPI_COBRANET_SEND_PACKET HPI_MAKE_ATTRIBUTE(HPI_CONTROL_COBRANET,6)
#define HPI_COBRANET_GET_PACKET  HPI_MAKE_ATTRIBUTE(HPI_CONTROL_COBRANET,7)
#endif

/*------------------------------------------------------------
 Cobranet Chip Bridge - copied from HMI.H
------------------------------------------------------------*/
#define  HPI_COBRANET_HMI_cobraBridge                    0x20000
#define  HPI_COBRANET_HMI_cobraBridgeTxPktBuf            ( HPI_COBRANET_HMI_cobraBridge + 0x1000 )
#define  HPI_COBRANET_HMI_cobraBridgeRxPktBuf            ( HPI_COBRANET_HMI_cobraBridge + 0x2000 )
#define  HPI_COBRANET_HMI_cobraIfTable1                  0x110000
#define  HPI_COBRANET_HMI_cobraIfPhyAddress              ( HPI_COBRANET_HMI_cobraIfTable1 + 0xd )
#define  HPI_COBRANET_HMI_cobraProtocolIP                0x72000
#define  HPI_COBRANET_HMI_cobraIpMonCurrentIP            ( HPI_COBRANET_HMI_cobraProtocolIP + 0x0 )
#define  HPI_COBRANET_HMI_cobraIpMonStaticIP             ( HPI_COBRANET_HMI_cobraProtocolIP + 0x2 )

/*------------------------------------------------------------
 Cobranet Chip Status bits
------------------------------------------------------------*/
#define HPI_COBRANET_HMI_STATUS_RXPACKET 2
#define HPI_COBRANET_HMI_STATUS_TXPACKET 3

/*------------------------------------------------------------
 Ethernet header size
------------------------------------------------------------*/
#define HPI_ETHERNET_HEADER_SIZE (16)

/* These defines are used to fill in protocol information for an Ethernet packet sent using HMI on CS18102 */
#define HPI_ETHERNET_PACKET_ID 0x85		/*!< ID supplied by Cirrius for ASI packets. */
#define HPI_ETHERNET_PACKET_V1 0x01		/*!< Simple packet - no special routing required */
#define HPI_ETHERNET_PACKET_HOSTED_VIA_HMI 0x20	/*!< This packet must make its way to the host across the HPI interface */
#define HPI_ETHERNET_PACKET_HOSTED_VIA_HMI_V1 0x21	/*!< This packet must make its way to the host across the HPI interface */
#define HPI_ETHERNET_PACKET_HOSTED_VIA_HPI 0x40	/*!< This packet must make its way to the host across the HPI interface */
#define HPI_ETHERNET_PACKET_HOSTED_VIA_HPI_V1 0x41	/*!< This packet must make its way to the host across the HPI interface */

#define HPI_ETHERNET_UDP_PORT (44600)						/*!< UDP messaging port */

/** \defgroup tonedet_attr Tonedetector attributes
\{
Used by HPI_ToneDetector_Set() and HPI_ToneDetector_Get()
*/

/** Set the threshold level of a tonedetector,
Threshold is a -ve number in units of dB/100,
*/
#define HPI_TONEDETECTOR_THRESHOLD HPI_MAKE_ATTRIBUTE(HPI_CONTROL_TONEDETECTOR,1)
/** Get the current state of tonedetection
The result is a bitmap of detected tones.  pairs of bits represent the left and right
channels, with left channel in LSB.  The lowest frequency detector state is in the LSB
*/
#define HPI_TONEDETECTOR_STATE HPI_MAKE_ATTRIBUTE(HPI_CONTROL_TONEDETECTOR,2)

/** Get the frequency of a tonedetector band.
*/
#define HPI_TONEDETECTOR_FREQUENCY HPI_MAKE_ATTRIBUTE(HPI_CONTROL_TONEDETECTOR,3)

/**\}*/

/** \defgroup silencedet_attr SilenceDetector attributes
\{
*/

/** Get the current state of tonedetection
The result is a bitmap with 1s for silent channels. Left channel is in LSB
*/
#define HPI_SILENCEDETECTOR_STATE HPI_MAKE_ATTRIBUTE(HPI_CONTROL_SILENCEDETECTOR,2)

/** Set the threshold level of a SilenceDetector,
Threshold is a -ve number in units of dB/100,
*/
#define HPI_SILENCEDETECTOR_THRESHOLD HPI_MAKE_ATTRIBUTE(HPI_CONTROL_SILENCEDETECTOR,1)

/** get/set the silence time before the detector triggers
*/
#define HPI_SILENCEDETECTOR_DELAY HPI_MAKE_ATTRIBUTE(HPI_CONTROL_SILENCEDETECTOR,3)

/**\}*/

/******************************************* CONTROLX ATTRIBUTES ****/
/* NOTE: All controlx attributes must be unique, unlike control attributes */
#ifdef HPI_SUPPORT_AES18
/* AES18 attributes */
#define HPI_AES18_CONFIG                101
#define HPI_AES18_INTERNAL_BUFFER_SIZE  102
#define HPI_AES18_INTERNAL_BUFFER_STATE 103
#define HPI_AES18_MESSAGE               104
#define HPI_AES18_ADDRESS               105

#define HPI_AES18_MODE_MASTER           0
#define HPI_AES18_MODE_SLAVE            1
#define HPI_AES18_CHANNEL_MODE_INDEPENDENT            (2)   /**< Left and Right channels operate independently */
#define HPI_AES18_CHANNEL_MODE_JOINT            (1) /**< Messages use alternating bits on the the left and right channels */

#endif
/******************************************* ASYNC ATTRIBUTES ****/
/** \defgroup async_event Async Event sources
\{
*/
#define HPI_ASYNC_EVENT_GPIO    1		/**< GPIO event. */
#define HPI_ASYNC_EVENT_SILENCE 2		/**< Silence event detected. */
#define HPI_ASYNC_EVENT_TONE    3	    /**< tone event detected. */
/** \} */

/*******************************************/
/** \defgroup errorcodes HPI Error codes

Almost all HPI functions return an error code
A return value of zero means there was no error.
Otherwise one of these error codes is returned.
Error codes can be converted to a descriptive string using HPI_GetErrorText()

\note When a new error code is added HPI_GetErrorText() MUST be updated.
\note codes 1-100 are reserved for driver use
\{
*/
#define HPI_ERROR_INVALID_TYPE          100 /**< Message type does not exist. */
#define HPI_ERROR_INVALID_OBJ           101 /**< Object type does not exist. */
#define HPI_ERROR_INVALID_FUNC          102 /**< Function does not exist. */
#define HPI_ERROR_INVALID_OBJ_INDEX     103 /**< The specified object (adapter/Stream) does not exist. */
#define HPI_ERROR_OBJ_NOT_OPEN          104 /**< Trying to access an object that has not been opened yet. */
#define HPI_ERROR_OBJ_ALREADY_OPEN      105 /**< Trying to open an already open object. */
#define HPI_ERROR_INVALID_RESOURCE      106 /**< PCI, ISA resource not valid. */
#define HPI_ERROR_SUBSYSFINDADAPTERS_GETINFO      107 /**< GetInfo call from SubSysFindAdapters failed. */
#define HPI_ERROR_INVALID_RESPONSE      108 /**< Default response was never updated with actual error code. */
#define HPI_ERROR_PROCESSING_MESSAGE    109 /**< wSize field of response was not updated, indicating that the message was not processed. */
#define HPI_ERROR_NETWORK_TIMEOUT       110 /**< The network did not respond in a timely manner. */
#define HPI_ERROR_INVALID_HANDLE        111 /**< An HPI handle is invalid (uninitialised?). */
#define HPI_ERROR_UNIMPLEMENTED         112 /**< A function or attribute has not been implemented yet. */

#define HPI_ERROR_TOO_MANY_ADAPTERS     200    /**< Too many adapters.*/
#define HPI_ERROR_BAD_ADAPTER           201    /**< Bad adpater. */
#define HPI_ERROR_BAD_ADAPTER_NUMBER    202    /**< Adapter number out of range or not set properly. */
#define HPI_DUPLICATE_ADAPTER_NUMBER    203    /**< 2 adapters with the same adapter number. */
#define HPI_ERROR_DSP_BOOTLOAD          204    /**< DSP code failed to bootload. */
#define HPI_ERROR_DSP_SELFTEST          205    /**< Adapter falied DSP code self test. */
#define HPI_ERROR_DSP_FILE_NOT_FOUND    206    /**< Couldn't find or open the DSP code file. */
#define HPI_ERROR_DSP_HARDWARE          207    /**< Internal DSP hardware error. */
#define HPI_ERROR_DOS_MEMORY_ALLOC      208    /**< Could not allocate memory in DOS. */
#define HPI_ERROR_MEMORY_ALLOC          208    /**< Could not allocate memory */
#define HPI_ERROR_PLD_LOAD              209    /**< Failed to correctly load/config PLD .*/
#define HPI_ERROR_DSP_FILE_FORMAT 		210    /**< Unexpected end of file, block length too big etc. */

#define HPI_ERROR_DSP_FILE_ACCESS_DENIED 211   /**< Found but could not open DSP code file. */
#define HPI_ERROR_DSP_FILE_NO_HEADER	212    /**< First DSP code section header not found in DSP file. */
#define HPI_ERROR_DSP_FILE_READ_ERROR	213    /**< File read operation on DSP code file failed. */
#define HPI_ERROR_DSP_SECTION_NOT_FOUND 214    /**< DSP code for adapter family not found. */
#define HPI_ERROR_DSP_FILE_OTHER_ERROR  215    /**< Other OS specific error opening DSP file. */
#define HPI_ERROR_DSP_FILE_SHARING_VIOLATION 216	/**< Sharing violation opening DSP code file. */
#define HPI_ERROR_DSP_FILE_NULL_HEADER	217    /**< DSP code section header had size == 0. */

#define HPI_ERROR_FLASH 220						/**< Base number for flash errors. */

#define HPI_ERROR_BAD_CHECKSUM (HPI_ERROR_FLASH+1)	/**< Flash has bad checksum */
#define HPI_ERROR_BAD_SEQUENCE (HPI_ERROR_FLASH+2)
#define HPI_ERROR_FLASH_ERASE (HPI_ERROR_FLASH+3)
#define HPI_ERROR_FLASH_PROGRAM (HPI_ERROR_FLASH+4)
#define HPI_ERROR_FLASH_VERIFY (HPI_ERROR_FLASH+5)
#define HPI_ERROR_FLASH_TYPE (HPI_ERROR_FLASH+6)
#define HPI_ERROR_FLASH_START (HPI_ERROR_FLASH+7)


#define HPI_ERROR_RESERVED_1            290    /**< Reserved for OEMs. */

#define HPI_ERROR_INVALID_STREAM        300 /**< Stream does not exist. */
#define HPI_ERROR_INVALID_FORMAT        301 /**< Invalid compression format. */
#define HPI_ERROR_INVALID_SAMPLERATE    302 /**< Invalid format samplerate */
#define HPI_ERROR_INVALID_CHANNELS      303 /**< Invalid format number of channels. */
#define HPI_ERROR_INVALID_BITRATE       304 /**< Invalid format bitrate. */
#define HPI_ERROR_INVALID_DATASIZE      305 /**< Invalid datasize used for stream read/write. */
#define HPI_ERROR_BUFFER_FULL           306 /**< Stream buffer is full during stream write. */
#define HPI_ERROR_BUFFER_EMPTY          307 /**< Stream buffer is empty during stream read. */
#define HPI_ERROR_INVALID_DATA_TRANSFER 308 /**< Invalid datasize used for stream read/write. */
#define HPI_ERROR_INVALID_OPERATION     310 /**< Object can't do requested operation in its current
                                                state, e.g. set format, change rec mux state while
                                                recording.*/
#define HPI_ERROR_INCOMPATIBLE_SAMPLERATE 311 /**< Where an SRG is shared amongst streams, an
                                                incompatible samplerate is one that is different to
                                                any currently playing or recording stream. */
#define HPI_ERROR_BAD_ADAPTER_MODE      312	/**< Adapter mode is illegal.*/
#define HPI_ERROR_TOO_MANY_CAPABILITY_CHANGE_ATTEMPTS 313 /**< There have been too many attempts to set the adapter's
                                                 capabilities (using bad keys). The card should be returned
                                                 to ASI if further capabilities updates are required */
#define HPI_ERROR_NO_INTERADAPTER_GROUPS 314 /**< Streams on different adapters cannot be grouped. */
#define HPI_ERROR_NO_INTERDSP_GROUPS 315     /**< Streams on different DSPs cannot be grouped. */

/** Mixer controls */
#define HPI_ERROR_INVALID_NODE          400     /**< Invalid mixer node for this adapter. */
#define HPI_ERROR_INVALID_CONTROL       401	    /**< Invalid control. */
#define HPI_ERROR_INVALID_CONTROL_VALUE  402    /**< Invalid control value was passed. */
#define HPI_ERROR_INVALID_CONTROL_ATTRIBUTE 403 /**< Control attribute not supported by this control. */
#define HPI_ERROR_CONTROL_DISABLED      404     /**< Control is disbaled. */
#define HPI_ERROR_CONTROL_I2C_MISSING_ACK 405   /**< I2C transaction failed due to a missing ACK. */
#define HPI_ERROR_UNSUPPORTED_CONTROL_ATTRIBUTE 406 /**< Control attribute is valid, but not supported by this hardware. */
#define HPI_ERROR_CONTROL_NOT_READY 407   /**< Control is busy, or coming out of reset and cannot be accessed at this time. */

/** Non volatile memory */
#define HPI_ERROR_NVMEM_BUSY  			450
#define HPI_ERROR_NVMEM_FULL 			451
#define HPI_ERROR_NVMEM_FAIL 			452

/** I2C */
#define HPI_ERROR_I2C_MISSING_ACK 		HPI_ERROR_CONTROL_I2C_MISSING_ACK
#define HPI_ERROR_I2C_BAD_ADR 			460


#ifdef HPI_SUPPORT_AES18
/** AES18 specific errors */
#define HPI_ERROR_AES18BG_BLOCKSPERSEC  500
#define HPI_ERROR_AES18BG_PRIORITYMASK  501
#define HPI_ERROR_AES18BG_MODE          502
#define HPI_ERROR_AES18_INVALID_PRIORITY    503
#define HPI_ERROR_AES18_INVALID_ADDRESS 504
#define HPI_ERROR_AES18_INVALID_REPETITION  505
#define HPI_ERROR_AES18BG_CHANNEL_MODE      506
#define HPI_ERROR_AES18_INVALID_CHANNEL  507
#endif

#define HPI_ERROR_CUSTOM                600 /**< custom error to use for debugging AGE 6/22/99 */

#define HPI_ERROR_MUTEX_TIMEOUT         700 /**< hpioct32.c can't obtain mutex */

#define HPI_ERROR_ILLEGAL_CACHE_VALUE   0xffff /**< indicates a cached HW16 value is invalid. */
/**\}*/

/* maximums */
#define HPI_MAX_ADAPTERS        20     /**< Maximum number of adapters per HPI sub-system WARNING: modifying this value changes the response structure size.*/
#define HPI_MAX_STREAMS         16       /**< Maximum number of in or out streams per adapter */
#define HPI_MAX_CHANNELS        2       /* per stream */
#define HPI_MAX_NODES           8       /* per mixer ? */
#define HPI_MAX_CONTROLS        4       /* per node ? */
#define HPI_MAX_ANC_BYTES_PER_FRAME		(64)	/**< maximum number of ancillary bytes per MPEG frame */
#define HPI_STRING_LEN          16
#ifdef HPI_SUPPORT_AES18
#define HPI_AES18_MAX_CHANNELS  2
#define HPI_AES18_MAX_PRIORITIES 4
#define HPI_AES18_MAX_PRIORITYMASK      0x000F
#define HPI_AES18BG_MAX_BLOCKSPERSEC    100     /**< AES18-1996, 100 blocks/sec */
#define HPI_AES18BG_MIN_BLOCKSPERSEC    2
#define HPI_AES18_MAX_ADDRESS           65536   /**< 16 bits address */
#define HPI_AES18_MAX_REPETITION        5       /**< AES18-1996, pg 5 */

/* defaults */
#define HPI_AES18_DEFAULT_PRIORITY_ENB  0x000F
/*
    See pg 16, AES18-1996
     @ 40 ms
    44.1 kHz = 1764 bits/block
    48 kHz = 1920 bits/block
*/
#define HPI_AES18_DEFAULT_BLOCK_LENGTH  1764    /* Bits.  Corresponds to 40ms at 44.1kHz */
#define HPI_AES18_DEFAULT_OP_MODE       HPI_AES18_MODE_MASTER
#endif

/* units */
#define HPI_OSTREAM_VELOCITY_UNITS        4096      /* AGE 6/4/99 */
#define HPI_OSTREAM_TIMESCALE_UNITS       (10000)

/**
\}
*/  /* end group hpi_defines */

/* ////////////////////////////////////////////////////////////////////// */
/* STRUCTURES */
#ifndef DISABLE_PRAGMA_PACK1
#pragma pack(push,1)
#endif

#ifndef SWIG
typedef union {
    struct {
		  unsigned int hwassembly  : 3;   /**< Assembly variant 0..7 */
        unsigned int hwrev       : 4;   /**< encode A-P as 0-15    */
        unsigned int swminor     : 6;   /**< sw minor version 0-63 */
		  unsigned int swmajor     : 3;   /**< sw major version 0-7  */
	 } s;
	 HW16 w;
    HW32 dw;
} HPI_VERSION;
#endif

#ifndef HPI_ON_DSP
/** Structure containing sample format information. See also HPI_FormatCreate().
  */
typedef  struct  sHPI_FORMAT {
    HW32    dwSampleRate;   /**< 11025, 32000, 44100 ... */
    HW32    dwBitRate;      /**< for MPEG */
    HW32    dwAttributes;   /**< Stereo/JointStereo/Mono */
    HW16    wModeLegacy;    /**< Legacy ancillary mode or idle bit  */
    HW16    wUnused;        /**< Unused */
    HW16    wChannels;      /**< 1,2..., (or ancillary mode or idle bit */
    HW16    wFormat;        /**< HPI_FORMAT_PCM16, _AC2, _MPEG. See \ref formats. */
} HPI_FORMAT;
#endif

#ifndef HPI_EXCLUDE_DEPRECATED
/**
Don't access fields in HPI_DATA.  Preferably switch  to
using HPI_OutStreamWriteBuf(), HPI_InStreamReadBuf().
Less desirably, always use HPI_DataCreate() to fill in the fields of
HPI_DATA.

\deprecated HPI_DATA will eventually disappear from this API
*/
typedef struct sHPI_DATA
{
    HW32  opaque_fixed_size[7];
} HPI_DATA;
#endif


typedef struct {
	HW32    dwValidBitsInThisFrame;
	HW8     bData[HPI_MAX_ANC_BYTES_PER_FRAME];
} HPI_ANC_FRAME;

typedef struct
{   HW32 dwPunchInSample;
    HW32 dwPunchOutSample;
}   HPI_PUNCHINOUT;

typedef struct
{   HW16 wObjectType;	/**< Type of object, HPI_OBJ_OSTREAM or HPI_OBJ_ISTREAM. */
    HW16 wStreamIndex;	/**< OStream or IStream index. */
}   HPI_STREAMID;

/** An object for containing a single async event.
*/
typedef struct {
		  HW16			wEventType; /**< Type of event. See HPI_ASYNC_EVENT_GPIO (etc.) defines. */
		  HW16			wSequence;  /**< Sequence number of event, allows lost event detection */
		  HW32			dwState;	/**< New state */
		  HW32			hObject;    /**< Handle to the object returning the event. */
		  union {
		  	struct {
		  		HW16	wIndex;		/**< GPIO bit index. */
		  	} gpio;
		  	struct {
		  		HW16	wNodeIndex;		/**< What node is the control on ? */
		  		HW16	wNodeType;		/**< What type of node is the control on ? */
		  	} control;
		  } u;
} HPI_ASYNC_EVENT;

typedef unsigned char HPI_ETHERNET_MAC_ADR[6];	/**< Used for sending ethernet packets VIA HMI interface */


/*////////////////////////////////////////////////////////////////////////// */
/* HPI FUNCTIONS */
/** An HPI error code \ref errorcodes */
typedef HW16 HPI_ERR;
typedef HW16 HPI_BOOL;
typedef HW32 HPI_HANDLE;

/* handles that reference various objects */
typedef HW16 HPI_HADAPTER;

typedef HPI_HANDLE HPI_HOSTREAM;
typedef HPI_HANDLE HPI_HISTREAM;
typedef HPI_HANDLE HPI_HSTREAM;	/* either InStream or OutStream */
typedef HPI_HANDLE HPI_HMIXER;
typedef HPI_HANDLE HPI_HCONTROL;
typedef HPI_HANDLE HPI_HNVMEMORY;
typedef HPI_HANDLE HPI_HGPIO;
typedef HPI_HANDLE HPI_HWATCHDOG;
typedef HPI_HANDLE HPI_HCLOCK;
typedef HPI_HANDLE HPI_HPROFILE;
typedef HPI_HANDLE HPI_HASYNC;

#ifndef HPI_ON_DSP
/* skip host side function declarations for DSP compile and documentation extraction */

typedef struct
{
	int not_really_used;
} HPI_HSUBSYS,*PHPI_HSUBSYS;

#ifndef DISABLE_PRAGMA_PACK1
#pragma pack(pop)
#endif

/*/////////////////////////// */
/* DATA and FORMAT and STREAM */

HPI_ERR HPI_API HPI_StreamEstimateBufferSize(
					 HPI_FORMAT *pF,
					 HW32 dwHostPollingRateInMilliSeconds,
					 HW32 *dwRecommendedBufferSize);


/*/////////// */
/* SUB SYSTEM */
PHPI_HSUBSYS HPI_API HPI_SubSysCreate(void);

void HPI_API HPI_SubSysFree(
					 HPI_HSUBSYS *phSubSys
					 );

HPI_ERR HPI_API HPI_SubSysGetVersion(
					 HPI_HSUBSYS *phSubSys,
					 HW32 *pdwVersion
					 );

HPI_ERR HPI_API HPI_SubSysGetVersionEx(
					 HPI_HSUBSYS *phSubSys,
					 HW32 *pdwVersionEx
					 );

HPI_ERR HPI_API HPI_SubSysGetInfo(
					 HPI_HSUBSYS *phSubSys,
					 HW32 *pdwVersion,
					 HW16 *pwNumAdapters,
					 HW16 awAdapterList[],
					 HW16 wListLength
					 );

/* SGT added 3-2-97 */
HPI_ERR HPI_API HPI_SubSysFindAdapters(
							HPI_HSUBSYS *phSubSys,
					 HW16 *pwNumAdapters,
					 HW16 awAdapterList[],
					 HW16 wListLength
					 );

HPI_ERR HPI_API HPI_SubSysGetNumAdapters(
					 HPI_HSUBSYS *phSubSys,
					 int *pnNumAdapters
					 );

HPI_ERR HPI_API HPI_SubSysGetAdapter(
					 HPI_HSUBSYS *phSubSys,
					 int nIterator,
					 HW32 *pdwAdapterIndex,
					 HW16 *pwAdapterType
					 );

HW16 HPI_API HPI_SubSysSsx2Bypass(
					 HPI_HSUBSYS *phSubSys,
					 HW16 wBypass
					 );

/*///////// */
/* ADAPTER */

HPI_ERR HPI_API HPI_AdapterOpen(
				HPI_HSUBSYS *phSubSys,
				HW16 wAdapterIndex
				);

HPI_ERR HPI_API HPI_AdapterClose(
				HPI_HSUBSYS *phSubSys,
				HW16 wAdapterIndex
				);

HPI_ERR HPI_API HPI_AdapterGetInfo(
				HPI_HSUBSYS *phSubSys,
				HW16 wAdapterIndex,
				HW16 *pwNumOutStreams,
				HW16 *pwNumInStreams,
				HW16 *pwVersion,
				HW32 *pdwSerialNumber,
				HW16 *pwAdapterType
				);

HPI_ERR HPI_API HPI_AdapterSetMode(
				HPI_HSUBSYS *phSubSys,
				HW16 wAdapterIndex,
				HW32 dwAdapterMode
				);
HPI_ERR HPI_API HPI_AdapterSetModeEx(
				HPI_HSUBSYS *phSubSys,
				HW16 wAdapterIndex,
				HW32 dwAdapterMode,
				HW16 wQueryOrSet
				);

HPI_ERR HPI_API HPI_AdapterGetMode(
				HPI_HSUBSYS *phSubSys,
				HW16 wAdapterIndex,
				HW32 *pdwAdapterMode
				);

HPI_ERR HPI_API HPI_AdapterGetAssert(
				HPI_HSUBSYS *phSubSys,
				HW16    wAdapterIndex,
				HW16    *wAssertPresent,
				char    *pszAssert,
				HW16    *pwLineNumber
				 );

HPI_ERR HPI_API HPI_AdapterGetAssertEx(
			  HPI_HSUBSYS *phSubSys,
			HW16    wAdapterIndex,
			HW16    *wAssertPresent,
			char    *pszAssert,
			HW32    *pdwLineNumber,
			HW16    *pwAssertOnDsp
			);

HPI_ERR HPI_API HPI_AdapterTestAssert(
				HPI_HSUBSYS *phSubSys,
				HW16    wAdapterIndex,
				HW16    wAssertId
				 );

HPI_ERR HPI_API HPI_AdapterEnableCapability(
				HPI_HSUBSYS *phSubSys,
				HW16 wAdapterIndex,
				HW16 wCapability,
				HW32 dwKey
				);

HPI_ERR HPI_API HPI_AdapterSelfTest(
				HPI_HSUBSYS *phSubSys,
				HW16 wAdapterIndex
				);

HPI_ERR HPI_API HPI_AdapterSetProperty(
				HPI_HSUBSYS 	*phSubSys,
				HW16			wAdapterIndex,
				HW16            wProperty,
				HW16            wParamter1,
				HW16            wParamter2
				);

HPI_ERR HPI_API HPI_AdapterGetProperty(
				HPI_HSUBSYS 	*phSubSys,
				HW16			wAdapterIndex,
				HW16            wProperty,
				HW16            *pwParamter1,
				HW16            *pwParamter2
				);

HPI_ERR HPI_API HPI_AdapterFindObject(
	const HPI_HSUBSYS	*phSubSys,
	HW16			wAdapterIndex,
	HW16			wObjectType,
	HW16			wObjectIndex,
	HW16			*pDspIndex
	);

HPI_ERR HPI_API HPI_AdapterEnumerateProperty(
	HPI_HSUBSYS	*phSubSys,
	HW16		wAdapterIndex,
	HW16		wIndex,
	HW16 		wWhatToEnumerate,
	HW16		wPropertyIndex,
	HW32		*pdwSetting
	);

/*////////////// */
/* NonVol Memory */
HPI_ERR HPI_API HPI_NvMemoryOpen(
		  HPI_HSUBSYS *phSubSys,
		  HW16    wAdapterIndex,
		  HPI_HNVMEMORY *phNvMemory,
		  HW16    *pwSizeInBytes
		  );

HPI_ERR HPI_API HPI_NvMemoryReadByte(
		  HPI_HSUBSYS *phSubSys,
		  HPI_HNVMEMORY hNvMemory,
		  HW16    wIndex,
		  HW16    *pwData
		  );

HPI_ERR HPI_API HPI_NvMemoryWriteByte(
		  HPI_HSUBSYS *phSubSys,
		  HPI_HNVMEMORY hNvMemory,
		  HW16    wIndex,
		  HW16    wData
		  );

/*////////////// */
/* Digital I/O */
HPI_ERR HPI_API HPI_GpioOpen(
		  HPI_HSUBSYS     *phSubSys,
		  HW16            wAdapterIndex,
		  HPI_HGPIO       *phGpio,
		  HW16            *pwNumberInputBits,
		  HW16            *pwNumberOutputBits
		  );

HPI_ERR HPI_API HPI_GpioReadBit(
		  HPI_HSUBSYS     *phSubSys,
		  HPI_HGPIO       hGpio,
		  HW16            wBitIndex,
		  HW16            *pwBitData
		  );

HPI_ERR HPI_API HPI_GpioReadAllBits(
		  HPI_HSUBSYS     *phSubSys,
		  HPI_HGPIO       hGpio,
		  HW16            *pwBitData
		  );

HPI_ERR HPI_API HPI_GpioWriteBit(
		  HPI_HSUBSYS     *phSubSys,
		  HPI_HGPIO       hGpio,
		  HW16            wBitIndex,
		  HW16            wBitData
        );

/**********************/
/* Async Event Object */
/**********************/
HPI_ERR HPI_API HPI_AsyncEventOpen(
		  HPI_HSUBSYS     *phSubSys,
		  HW16            wAdapterIndex,
		  HPI_HASYNC      *phAsync
		  );

HPI_ERR HPI_API HPI_AsyncEventClose(
		  HPI_HSUBSYS     *phSubSys,
		  HPI_HASYNC      hAsync
		  );

HPI_ERR HPI_API HPI_AsyncEventWait(
		  HPI_HSUBSYS   *phSubSys,
		  HPI_HASYNC    hAsync,
		  HW16			wMaximumEvents,
		  HPI_ASYNC_EVENT *pEvents,
		  HW16			*pwNumberReturned
		  );

HPI_ERR HPI_API HPI_AsyncEventGetCount(
		  HPI_HSUBSYS   *phSubSys,
		  HPI_HASYNC    hAsync,
		  HW16			*pwCount
		  );

HPI_ERR HPI_API HPI_AsyncEventGet(
		  HPI_HSUBSYS   *phSubSys,
		  HPI_HASYNC    hAsync,
		  HW16			wMaximumEvents,
		  HPI_ASYNC_EVENT *pEvents,
		  HW16			*pwNumberReturned
		  );

/*/////////// */
/* WATCH-DOG  */
HPI_ERR HPI_API HPI_WatchdogOpen(
        HPI_HSUBSYS     *phSubSys,
        HW16            wAdapterIndex,
        HPI_HWATCHDOG   *phWatchdog
        );

HPI_ERR HPI_API HPI_WatchdogSetTime(
        HPI_HSUBSYS     *phSubSys,
        HPI_HWATCHDOG   hWatchdog,
		  HW32            dwTimeMillisec
        );

HPI_ERR HPI_API HPI_WatchdogPing(
        HPI_HSUBSYS     *phSubSys,
        HPI_HWATCHDOG   hWatchdog
        );

/**************/
/* OUT STREAM */
/**************/
HPI_ERR HPI_API HPI_OutStreamOpen(
                HPI_HSUBSYS     *phSubSys,
                HW16            wAdapterIndex,
                HW16            wOutStreamIndex,
                HPI_HOSTREAM    *phOutStreamHandle );

HPI_ERR HPI_API HPI_OutStreamClose(
                HPI_HSUBSYS     *phSubSys,
                HPI_HOSTREAM    hOutStreamHandle );

#ifndef HPI_EXCLUDE_DEPRECATED
DEPRECATED HPI_ERR HPI_API HPI_OutStreamGetInfo(
                HPI_HSUBSYS     *phSubSys,
					 HPI_HOSTREAM    hOutStreamHandle,
                HW16            *pwState,
                HW32            *pdwBufferSize,
                HW32            *pdwDataToPlay );
#endif

HPI_ERR HPI_API HPI_OutStreamGetInfoEx(
                HPI_HSUBSYS     *phSubSys,
                HPI_HOSTREAM    hOutStreamHandle,
                HW16            *pwState,
                HW32            *pdwBufferSize,
                HW32            *pdwDataToPlay,
                HW32            *pdwSamplesPlayed,
                HW32            *pdwAuxiliaryDataToPlay);

HPI_ERR HPI_API HPI_OutStreamWriteBuf(
			    HPI_HSUBSYS *phSubSys,
				HPI_HOSTREAM hOutStream,
				HW8 * pbWriteBuf,
				HW32  dwBytesToWrite,
				HPI_FORMAT *pFormat);

#ifndef HPI_EXCLUDE_DEPRECATED
DEPRECATED HPI_ERR HPI_API HPI_OutStreamWrite(
                HPI_HSUBSYS     *phSubSys,
                HPI_HOSTREAM    hOutStreamHandle,
                HPI_DATA        *pData );
#endif

HPI_ERR HPI_API HPI_OutStreamStart(
                HPI_HSUBSYS     *phSubSys,
                HPI_HOSTREAM    hOutStreamHandle );

HPI_ERR HPI_API HPI_OutStreamStop(
                HPI_HSUBSYS     *phSubSys,
                HPI_HOSTREAM    hOutStreamHandle );

HPI_ERR HPI_API HPI_OutStreamSinegen(
        HPI_HSUBSYS *phSubSys,
        HPI_HOSTREAM hOutStream
);

HPI_ERR HPI_API HPI_OutStreamReset(
                HPI_HSUBSYS     *phSubSys,
					 HPI_HOSTREAM    OutStreamHandle );

HPI_ERR HPI_API HPI_OutStreamQueryFormat(
                HPI_HSUBSYS     *phSubSys,
                     HPI_HOSTREAM    OutStreamHandle,
					 HPI_FORMAT      *pFormat );

HPI_ERR HPI_API HPI_OutStreamSetPunchInOut(
					 HPI_HSUBSYS     *phSubSys,
					 HPI_HOSTREAM    hOutStreamHandle,
					 HW32            dwPunchInSample,
					 HW32            dwPunchOutSample );

HPI_ERR HPI_API HPI_OutStreamSetVelocity(
		  HPI_HSUBSYS     *phSubSys,
		  HPI_HOSTREAM    hOutStream,
		  short           nVelocity);

HPI_ERR HPI_API HPI_OutStreamAncillaryReset(
				HPI_HSUBSYS 	*phSubSys,
				HPI_HOSTREAM 	OutStreamHandle,
				HW16			wMode);

HPI_ERR HPI_API HPI_OutStreamAncillaryGetInfo(
				HPI_HSUBSYS *phSubSys,
				HPI_HOSTREAM hOutStreamHandle,
				HW32		*pdwFramesAvailable);

HPI_ERR HPI_API HPI_OutStreamAncillaryRead(
				HPI_HSUBSYS *phSubSys,
				HPI_HOSTREAM hOutStreamHandle,
				HPI_ANC_FRAME	*pAncFrameBuffer,
				HW32	dwAncFrameBufferSizeInBytes,
				HW32	dwNumberOfAncillaryFramesToRead);


HPI_ERR HPI_API HPI_OutStreamSetTimeScale(
                HPI_HSUBSYS  *phSubSys,
                HPI_HOSTREAM hOutStreamHandle,
                HW32         dwTimeScaleX10000);

HPI_ERR HPI_API HPI_OutStreamHostBufferAllocate(
				HPI_HSUBSYS *phSubSys,
				HPI_HOSTREAM hOutStreamHandle,
				HW32	dwSizeInBytes);

HPI_ERR HPI_API HPI_OutStreamHostBufferFree(
				HPI_HSUBSYS *phSubSys,
				HPI_HOSTREAM hOutStreamHandle);

HPI_ERR HPI_API HPI_OutStreamGroupAdd(
				HPI_HSUBSYS *phSubSys,
				HPI_HOSTREAM hOutStreamHandle,
				HPI_HSTREAM hStreamHandle );

HPI_ERR HPI_API HPI_OutStreamGroupGetMap(
				HPI_HSUBSYS *phSubSys,
				HPI_HOSTREAM hOutStreamHandle,
				HW32 *pdwOutStreamMap,
				HW32 *pdwInStreamMap );

HPI_ERR HPI_API HPI_OutStreamGroupReset(
				HPI_HSUBSYS *phSubSys,
				HPI_HOSTREAM hOutStreamHandle );

/*////////// */
/* IN_STREAM */
HPI_ERR HPI_API HPI_InStreamOpen(
                HPI_HSUBSYS *phSubSys,
                HW16        wAdapterIndex,
					 HW16        wInStreamIndex,
                HPI_HISTREAM *phInStream
                );

HPI_ERR HPI_API HPI_InStreamClose(
                HPI_HSUBSYS *phSubSys,
                HPI_HISTREAM hInStream
                );

HPI_ERR HPI_API HPI_InStreamQueryFormat(
                HPI_HSUBSYS *phSubSys,
                HPI_HISTREAM hInStream,
                HPI_FORMAT  *pFormat
                );

HPI_ERR HPI_API HPI_InStreamSetFormat(
                HPI_HSUBSYS *phSubSys,
                HPI_HISTREAM hInStream,
                HPI_FORMAT  *pFormat
                );

HPI_ERR HPI_API HPI_InStreamReadBuf(
				HPI_HSUBSYS *phSubSys,
				HPI_HISTREAM hInStream,
				HW8 * pbReadBuf,
				HW32 dwBytesToRead);

#ifndef HPI_EXCLUDE_DEPRECATED
DEPRECATED HPI_ERR HPI_API HPI_InStreamRead(
                HPI_HSUBSYS *phSubSys,
                HPI_HISTREAM hInStream,
                HPI_DATA    *pData
                );
#endif

HPI_ERR HPI_API HPI_InStreamStart(
                HPI_HSUBSYS *phSubSys,
					 HPI_HISTREAM hInStream
					 );

HPI_ERR HPI_API HPI_InStreamStop(
                HPI_HSUBSYS *phSubSys,
                HPI_HISTREAM hInStream
                );

HPI_ERR HPI_API HPI_InStreamReset(
		 HPI_HSUBSYS *phSubSys,
                HPI_HISTREAM hInStream
                );

#ifndef HPI_EXCLUDE_DEPRECATED
DEPRECATED HPI_ERR HPI_API HPI_InStreamGetInfo(
                HPI_HSUBSYS *phSubSys,
                HPI_HISTREAM hInStream,
                HW16        *pwState,
                HW32        *pdwBufferSize,
                HW32        *pdwDataRecorded
					 );
#endif

HPI_ERR HPI_API HPI_InStreamGetInfoEx(
					 HPI_HSUBSYS *phSubSys,
                HPI_HISTREAM hInStream,
                HW16        *pwState,
                HW32        *pdwBufferSize,
                HW32        *pdwDataRecorded,
				HW32        *pdwSamplesRecorded,
                HW32        *pdwAuxiliaryDataRecorded
                );

HPI_ERR HPI_API HPI_InStreamAncillaryReset(
				HPI_HSUBSYS 	*phSubSys,
				HPI_HISTREAM 	InStreamHandle,
				HW16			wBytesPerFrame,
				HW16 			wMode,
				HW16			wAlignment,
				HW16			wIdleBit);

HPI_ERR HPI_API HPI_InStreamAncillaryGetInfo(
				HPI_HSUBSYS 	*phSubSys,
				HPI_HISTREAM 	hInStreamHandle,
				HW32	*pdwFrameSpace);

HPI_ERR HPI_API HPI_InStreamAncillaryWrite(
				HPI_HSUBSYS *phSubSys,
				HPI_HISTREAM hInStream,
				HPI_ANC_FRAME	*pAncFrameBuffer,
				HW32	dwAncFrameBufferSizeInBytes,
				HW32	dwNumberOfAncillaryFramesToWrite);

HPI_ERR HPI_API HPI_InStreamHostBufferAllocate(
				HPI_HSUBSYS *phSubSys,
				HPI_HISTREAM hInStream,
				HW32	dwSizeInBytes);

HPI_ERR HPI_API HPI_InStreamHostBufferFree(
				HPI_HSUBSYS *phSubSys,
				HPI_HISTREAM hInStream);

HPI_ERR HPI_API HPI_InStreamGroupAdd(
				HPI_HSUBSYS *phSubSys,
				HPI_HISTREAM hInStreamHandle,
				HPI_HSTREAM hStreamHandle );

HPI_ERR HPI_API HPI_InStreamGroupGetMap(
				HPI_HSUBSYS *phSubSys,
				HPI_HISTREAM hInStreamHandle,
				HW32 *pdwOutStreamMap,
				HW32 *pdwInStreamMap );

HPI_ERR HPI_API HPI_InStreamGroupReset(
				HPI_HSUBSYS *phSubSys,
				HPI_HISTREAM hInStreamHandle );

/*********/
/* MIXER */
/*********/
HPI_ERR HPI_API HPI_MixerOpen(
					 HPI_HSUBSYS *phSubSys,
					 HW16        wAdapterIndex,
					 HPI_HMIXER  *phMixerHandle
                );

HPI_ERR HPI_API HPI_MixerClose(
                HPI_HSUBSYS *phSubSys,
                HPI_HMIXER  hMixerHandle
                );

HPI_ERR HPI_API HPI_MixerGetControl(
                HPI_HSUBSYS     *phSubSys,
                HPI_HMIXER      hMixerHandle,
                HW16            wSrcNodeType,
                HW16            wSrcNodeTypeIndex,
                HW16            wDstNodeType,
                HW16            wDstNodeTypeIndex,
                HW16            wControlType,
                HPI_HCONTROL    *phControlHandle
                );

HPI_ERR HPI_API HPI_MixerGetControlByIndex(
                HPI_HSUBSYS *phSubSys,
                HPI_HMIXER   hMixerHandle,
                HW16         wControlIndex,
                HW16         *pwSrcNodeType,
                HW16         *pwSrcNodeIndex,
                HW16         *pwDstNodeType,
                HW16         *pwDstNodeIndex,
                HW16         *pwControlType,
                HPI_HCONTROL *phControlHandle
);

HPI_ERR HPI_API HPI_MixerStore(
                HPI_HSUBSYS *phSubSys,
                HPI_HMIXER   hMixerHandle,
                enum HPI_MIXER_STORE_COMMAND Command,
                HW16         wIndex
);
/*************************/
/* mixer CONTROLS        */
/*************************/

/* Generic query of available control settings */
HPI_ERR HPI_API HPI_ControlQuery(
	 const HPI_HSUBSYS     *phSubSys,
	 const HPI_HCONTROL    hControlHandle,
	 const HW16            wAttrib,
	 const HW32	           dwIndex,
	 const HW32            dwParam,
	 HW32        		   *pdwSetting
);

#ifndef SWIG
/* Generic setting of control attribute value */
HPI_ERR HPI_API HPI_ControlParamSet(
	 const HPI_HSUBSYS     *phSubSys,
	 const HPI_HCONTROL    hControlHandle,
	 const HW16            wAttrib,
	 const HW32            dwParam1,
	 const HW32            dwParam2
);

/* generic getting of control attribute value.
   Null pointers allowed for return values
*/
HPI_ERR HPI_API HPI_ControlParamGet(
	 const HPI_HSUBSYS     *phSubSys,
	 const HPI_HCONTROL    hControlHandle,
	 const HW16            wAttrib,
	 HW32                  dwParam1,
	 HW32                  dwParam2,
	 HW32        		   *pdwParam1,
	 HW32                  *pdwParam2
);
#endif

/*************************/
/* volume control        */
/*************************/
HPI_ERR HPI_API HPI_VolumeSetGain(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                short           anGain0_01dB[HPI_MAX_CHANNELS]
                );

HPI_ERR HPI_API HPI_VolumeGetGain(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                short           anGain0_01dB_out[HPI_MAX_CHANNELS]
                );

#define HPI_VolumeGetRange HPI_VolumeQueryRange
HPI_ERR HPI_API HPI_VolumeQueryRange(
					 HPI_HSUBSYS     *phSubSys,
					 HPI_HCONTROL    hControlHandle,
					 short           *nMinGain_01dB,
					 short           *nMaxGain_01dB,
					 short           *nStepGain_01dB
					 );

HPI_ERR HPI_API HPI_VolumeAutoFade(
					 HPI_HSUBSYS     *phSubSys,
					 HPI_HCONTROL    hControlHandle,
					 short           anStopGain0_01dB[HPI_MAX_CHANNELS],
					 HW32            wDurationMs
					 );

HPI_ERR HPI_API HPI_VolumeAutoFadeProfile(
					 HPI_HSUBSYS     *phSubSys,
					 HPI_HCONTROL    hControlHandle,
					 short           anStopGain0_01dB[HPI_MAX_CHANNELS],
					 HW32            dwDurationMs,
					 HW16            dwProfile
					 );

/*************************/
/* level control         */
/*************************/
HPI_ERR HPI_API HPI_LevelSetGain(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                short           anGain0_01dB[HPI_MAX_CHANNELS]
                );

HPI_ERR HPI_API HPI_LevelGetGain(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                short           anGain0_01dB_out[HPI_MAX_CHANNELS]
                );

/*************************/
/* meter control         */
/*************************/
HPI_ERR HPI_API HPI_MeterGetPeak(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                short           anPeak0_01dB_out[HPI_MAX_CHANNELS]
					 );

HPI_ERR HPI_API HPI_MeterGetRms(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                short           anPeak0_01dB_out[HPI_MAX_CHANNELS]
					 );

HPI_ERR HPI_API HPI_MeterSetPeakBallistics(
				HPI_HSUBSYS 	*phSubSys,
				HPI_HCONTROL	hControlHandle,
				HW16			nAttack,
				HW16			nDecay
				);

HPI_ERR HPI_API HPI_MeterSetRmsBallistics(
				HPI_HSUBSYS 	*phSubSys,
				HPI_HCONTROL	hControlHandle,
				HW16			nAttack,
				HW16			nDecay
				);

HPI_ERR HPI_API HPI_MeterGetPeakBallistics(
				HPI_HSUBSYS 	*phSubSys,
				HPI_HCONTROL	hControlHandle,
				HW16			*nAttack,
				HW16			*nDecay
				);

HPI_ERR HPI_API HPI_MeterGetRmsBallistics(
				HPI_HSUBSYS 	*phSubSys,
				HPI_HCONTROL	hControlHandle,
				HW16			*nAttack,
				HW16			*nDecay
				);


/*************************/
/* channel mode control  */
/*************************/
HPI_ERR HPI_API HPI_ChannelModeSet(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW16            wMode
                );

HPI_ERR HPI_API HPI_ChannelModeGet(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW16            *wMode
                );

/*************************/
/* Tuner control         */
/*************************/
HPI_ERR HPI_API HPI_Tuner_SetBand(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW16            wBand
                );

HPI_ERR HPI_API HPI_Tuner_GetBand(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW16             *pwBand
                );

HPI_ERR HPI_API HPI_Tuner_SetFrequency(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW32            wFreqInkHz
                );


HPI_ERR HPI_API HPI_Tuner_GetFrequency(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW32            *pwFreqInkHz
                );

HPI_ERR HPI_API HPI_Tuner_GetRFLevel(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                short           *pwLevel
                );

HPI_ERR HPI_API HPI_Tuner_GetRawRFLevel(
    HPI_HSUBSYS     *phSubSys,
    HPI_HCONTROL    hControlHandle,
    short            *pwLevel
);

HPI_ERR HPI_API HPI_Tuner_SetGain(
					HPI_HSUBSYS     *phSubSys,
					HPI_HCONTROL    hControlHandle,
					short           nGain
					);

HPI_ERR HPI_API HPI_Tuner_GetGain(
					HPI_HSUBSYS     *phSubSys,
					HPI_HCONTROL    hControlHandle,
					short           *pnGain
					);

#ifndef HPI_EXCLUDE_DEPRECATED
DEPRECATED HPI_ERR HPI_API HPI_Tuner_GetVideoStatus(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
					 HW16            *pwVideoStatus
					 );
#endif

HPI_ERR HPI_API HPI_Tuner_GetStatus(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
           		HW16            *pwStatusMask,
				HW16            *pwStatus
					 );

HPI_ERR HPI_API HPI_Tuner_SetMode(
	 HPI_HSUBSYS     *phSubSys,
	 HPI_HCONTROL    hControlHandle,
	 HW32            nMode,
	 HW32            nValue
);

HPI_ERR HPI_API HPI_Tuner_GetMode(
	 HPI_HSUBSYS     *phSubSys,
	 HPI_HCONTROL    hControlHandle,
	 HW32            nMode,
	 HW32            *pnValue
);

HPI_ERR HPI_API HPI_Tuner_GetRDS(
	 HPI_HSUBSYS     *phSubSys,
	 HPI_HCONTROL    hControlHandle,
	 char             *pRdsData
);


/****************************/
/* AES/EBU Receiver control */
/****************************/
HPI_ERR HPI_API HPI_AESEBU_Receiver_SetFormat(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW16            wSource
                );

HPI_ERR HPI_API HPI_AESEBU_Receiver_GetFormat(
				HPI_HSUBSYS     *phSubSys,
				HPI_HCONTROL    hControlHandle,
				HW16            *pwSource
				);

#ifndef HPI_EXCLUDE_DEPRECATED
DEPRECATED HPI_ERR HPI_API HPI_AESEBU_Receiver_SetSource(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW16            wSource
                );

DEPRECATED HPI_ERR HPI_API HPI_AESEBU_Receiver_GetSource(
				HPI_HSUBSYS     *phSubSys,
				HPI_HCONTROL    hControlHandle,
				HW16            *pwSource
				);
#endif

HPI_ERR HPI_API HPI_AESEBU_Receiver_GetSampleRate(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW32            *pdwSampleRate
                );

HPI_ERR HPI_API HPI_AESEBU_Receiver_GetUserData(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW16            wIndex,
                HW16            *pwData
                );

HPI_ERR HPI_API HPI_AESEBU_Receiver_GetChannelStatus(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
					 HW16            wIndex,
                HW16            *pwData
                );

HPI_ERR HPI_API HPI_AESEBU_Receiver_GetErrorStatus(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW16            *pwErrorData
                );

/*******************************/
/* AES/EBU Transmitter control */
/*******************************/
HPI_ERR HPI_API HPI_AESEBU_Transmitter_SetSampleRate(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW32            dwSampleRate
                );

HPI_ERR HPI_API HPI_AESEBU_Transmitter_SetUserData(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW16            wIndex,
                HW16            wData
                );

HPI_ERR HPI_API HPI_AESEBU_Transmitter_SetChannelStatus(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW16            wIndex,
                HW16            wData
                );

HPI_ERR HPI_API HPI_AESEBU_Transmitter_GetChannelStatus(
				HPI_HSUBSYS     *phSubSys,
				HPI_HCONTROL    hControlHandle,
				HW16            wIndex,
				HW16            *pwData
				);

HPI_ERR HPI_API HPI_AESEBU_Transmitter_SetFormat(
				HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
				HW16            wOutputFormat
                );

HPI_ERR HPI_API HPI_AESEBU_Transmitter_GetFormat(
				HPI_HSUBSYS     *phSubSys,
				HPI_HCONTROL    hControlHandle,
				HW16            *pwOutputFormat
				);

/***********************/
/* multiplexer control */
/***********************/
HPI_ERR HPI_API HPI_Multiplexer_SetSource(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW16            wSourceNodeType,
                HW16            wSourceNodeIndex
                );
HPI_ERR HPI_API HPI_Multiplexer_GetSource(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW16            *wSourceNodeType,
                HW16            *wSourceNodeIndex
                );

HPI_ERR HPI_API HPI_Multiplexer_QuerySource(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW16            nIndex,
                HW16            *wSourceNodeType,
                HW16            *wSourceNodeIndex
                );

/***************/
/* VOX control */
/***************/
HPI_ERR HPI_API HPI_VoxSetThreshold(
        HPI_HSUBSYS     *phSubSys,
        HPI_HCONTROL    hControlHandle,
        short           anGain0_01dB
        );

HPI_ERR HPI_API HPI_VoxGetThreshold(
        HPI_HSUBSYS     *phSubSys,
        HPI_HCONTROL    hControlHandle,
        short           *anGain0_01dB
        );

/*********************/
/* Bitstream control */
/*********************/
HPI_ERR HPI_API HPI_Bitstream_SetClockEdge(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW16            wEdgeType
                );

HPI_ERR HPI_API HPI_Bitstream_SetDataPolarity(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW16            wPolarity
                );

HPI_ERR HPI_API HPI_Bitstream_GetActivity(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW16            * pwClkActivity,
                HW16            * pwDataActivity
                );


/***********************/
/* SampleClock control */
/***********************/
HPI_ERR HPI_API HPI_SampleClock_SetSource(
				HPI_HSUBSYS     *phSubSys,
				HPI_HCONTROL    hControlHandle,
				HW16            wSource
				);

HPI_ERR HPI_API HPI_SampleClock_GetSource(
				HPI_HSUBSYS     *phSubSys,
				HPI_HCONTROL    hControlHandle,
				HW16            *pwSource
				);

HPI_ERR HPI_API HPI_SampleClock_SetSourceIndex(
				HPI_HSUBSYS     *phSubSys,
				HPI_HCONTROL    hControlHandle,
				HW16            wSourceIndex
);

HPI_ERR HPI_API HPI_SampleClock_GetSourceIndex(
    			HPI_HSUBSYS     *phSubSys,
				HPI_HCONTROL    hControlHandle,
				HW16            *pwSourceIndex
				);

HPI_ERR HPI_API HPI_SampleClock_SetSampleRate(
				HPI_HSUBSYS     *phSubSys,
				HPI_HCONTROL    hControlHandle,
				HW32            dwSampleRate
				);

HPI_ERR HPI_API HPI_SampleClock_GetSampleRate(
				HPI_HSUBSYS     *phSubSys,
				HPI_HCONTROL    hControlHandle,
				HW32            * pdwSampleRate
				);


/***********************/
/* Microphone control */
/***********************/
HPI_ERR HPI_API HPI_Microphone_SetPhantomPower(
				HPI_HSUBSYS     *phSubSys,
				HPI_HCONTROL    hControlHandle,
				HW16            wOnOff
				);

HPI_ERR HPI_API HPI_Microphone_GetPhantomPower(
				HPI_HSUBSYS     *phSubSys,
				HPI_HCONTROL    hControlHandle,
				HW16            *pwOnOff
				);

/*******************************
  Parametric Equalizer control
*******************************/
HPI_ERR HPI_API HPI_ParametricEQ_GetInfo(
		HPI_HSUBSYS 	*phSubSys,
		HPI_HCONTROL	hControlHandle,
		HW16			*pwNumberOfBands,
		HW16			*pwEnabled
		);

HPI_ERR HPI_API HPI_ParametricEQ_SetState(
		HPI_HSUBSYS 	*phSubSys,
		HPI_HCONTROL	hControlHandle,
		HW16			   wOnOff
		);

HPI_ERR HPI_API HPI_ParametricEQ_SetBand(
	HPI_HSUBSYS		*phSubSys,
	HPI_HCONTROL	hControlHandle,
	HW16            wIndex,
	HW16            nType,
	HW32            dwFrequencyHz,
	short           nQ100,
	short           nGain0_01dB
	);

HPI_ERR HPI_API HPI_ParametricEQ_GetBand(
	HPI_HSUBSYS		*phSubSys,
	HPI_HCONTROL	hControlHandle,
	HW16            wIndex,
	HW16            *pnType,
	HW32            *pdwFrequencyHz,
	short           *pnQ100,
	short           *pnGain0_01dB
	);

HPI_ERR HPI_API HPI_ParametricEQ_GetCoeffs(
	 HPI_HSUBSYS		*phSubSys,
	 HPI_HCONTROL	hControlHandle,
	 HW16            wIndex,
    short			coeffs[5]
);

/*******************************
  Compressor Expander control
*******************************/

HPI_ERR HPI_API HPI_Compander_Set(
	HPI_HSUBSYS		*phSubSys,
	HPI_HCONTROL	hControlHandle,
	HW16            wAttack,
	HW16            wDecay,
	short           wRatio100,
	short           nThreshold0_01dB,
	short           nMakeupGain0_01dB
	);

HPI_ERR HPI_API HPI_Compander_Get(
	HPI_HSUBSYS		*phSubSys,
	HPI_HCONTROL	hControlHandle,
	HW16            *pwAttack,
	HW16            *pwDecay,
	short           *pwRatio100,
	short           *pnThreshold0_01dB,
	short           *pnMakeupGain0_01dB
	);

/*******************************
  Cobranet HMI control
*******************************/
HPI_ERR HPI_API HPI_Cobranet_HmiWrite(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW32 dwHmiAddress,
                HW32 dwByteCount,
                HW8 *pbData
                );

HPI_ERR HPI_API HPI_Cobranet_HmiRead(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW32 dwHmiAddress,
                HW32 dwMaxByteCount,
                HW32 *pdwByteCount,
                HW8  *pbData
                );

HPI_ERR HPI_API HPI_Cobranet_HmiGetStatus(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW32 *pdwStatus,
				HW32 *pdwReadableSize,
				HW32 *pdwWriteableSize
				);

/*Read the current IP address
*/
HPI_ERR HPI_API HPI_Cobranet_GetIPaddress(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW32 *pdwIPaddress
				);
/* Write the current IP address
*/
HPI_ERR HPI_API HPI_Cobranet_SetIPaddress(
				HPI_HSUBSYS *phSubSys,
				HPI_HCONTROL hControlHandle,
				HW32 dwIPaddress
				);
/* Read the static IP address
*/
HPI_ERR HPI_API HPI_Cobranet_GetStaticIPaddress(
				HPI_HSUBSYS *phSubSys,
				HPI_HCONTROL hControlHandle,
				HW32 *pdwIPaddress
				);
/* Write the static IP address
*/
HPI_ERR HPI_API HPI_Cobranet_SetStaticIPaddress(
				HPI_HSUBSYS *phSubSys,
				HPI_HCONTROL hControlHandle,
				HW32 dwIPaddress
				);
/* Read the MAC address
*/
HPI_ERR HPI_API HPI_Cobranet_GetMACaddress(
                HPI_HSUBSYS     *phSubSys,
                HPI_HCONTROL    hControlHandle,
                HW32 *pdwMAC_MSBs,
                HW32 *pdwMAC_LSBs
				);
/*******************************
  Tone Detector control
*******************************/
HPI_ERR HPI_API HPI_ToneDetector_GetState(HPI_HSUBSYS *phSubSys, HPI_HCONTROL hC, HW32 * State);
HPI_ERR HPI_API HPI_ToneDetector_SetEnable(HPI_HSUBSYS *phSubSys, HPI_HCONTROL hC, HW32 Enable);
HPI_ERR HPI_API HPI_ToneDetector_GetEnable(HPI_HSUBSYS *phSubSys, HPI_HCONTROL hC, HW32 * Enable);
HPI_ERR HPI_API HPI_ToneDetector_SetEventEnable(HPI_HSUBSYS *phSubSys, HPI_HCONTROL hC, HW32 EventEnable);
HPI_ERR HPI_API HPI_ToneDetector_GetEventEnable(HPI_HSUBSYS *phSubSys, HPI_HCONTROL hC, HW32 * EventEnable);
HPI_ERR HPI_API HPI_ToneDetector_SetThreshold(HPI_HSUBSYS *phSubSys, HPI_HCONTROL hC, int Threshold);
HPI_ERR HPI_API HPI_ToneDetector_GetThreshold(HPI_HSUBSYS *phSubSys, HPI_HCONTROL hC, int * Threshold);
HPI_ERR HPI_API HPI_ToneDetector_GetFrequency(HPI_HSUBSYS *phSubSys,HPI_HCONTROL hC, HW32 nIndex, HW32 * dwFrequency);

/*******************************
  Silence Detector control
*******************************/
HPI_ERR HPI_API HPI_SilenceDetector_GetState(HPI_HSUBSYS *phSubSys, HPI_HCONTROL hC, HW32 * State);
HPI_ERR HPI_API HPI_SilenceDetector_SetEnable(HPI_HSUBSYS *phSubSys, HPI_HCONTROL hC, HW32 Enable);
HPI_ERR HPI_API HPI_SilenceDetector_GetEnable(HPI_HSUBSYS *phSubSys, HPI_HCONTROL hC, HW32 * Enable);
HPI_ERR HPI_API HPI_SilenceDetector_SetEventEnable(HPI_HSUBSYS *phSubSys, HPI_HCONTROL hC, HW32 EventEnable);
HPI_ERR HPI_API HPI_SilenceDetector_GetEventEnable(HPI_HSUBSYS *phSubSys, HPI_HCONTROL hC, HW32 * EventEnable);
HPI_ERR HPI_API HPI_SilenceDetector_SetDelay(HPI_HSUBSYS *phSubSys, HPI_HCONTROL hC, HW32 Delay);
HPI_ERR HPI_API HPI_SilenceDetector_GetDelay(HPI_HSUBSYS *phSubSys, HPI_HCONTROL hC, HW32 * Delay);
HPI_ERR HPI_API HPI_SilenceDetector_SetThreshold(HPI_HSUBSYS *phSubSys, HPI_HCONTROL hC, int Threshold);
HPI_ERR HPI_API HPI_SilenceDetector_GetThreshold(HPI_HSUBSYS *phSubSys, HPI_HCONTROL hC, int * Threshold);

/*/////////// */
/* DSP CLOCK  */
/*/////////// */
HPI_ERR HPI_API HPI_ClockOpen(
        HPI_HSUBSYS     *phSubSys,
        HW16            wAdapterIndex,
        HPI_HCLOCK      *phDspClock
        );

HPI_ERR HPI_API HPI_ClockSetTime(
        HPI_HSUBSYS     *phSubSys,
        HPI_HCLOCK      hClock,
        HW16            wHour,
        HW16            wMinute,
        HW16            wSecond,
        HW16            wMilliSecond
        );

HPI_ERR HPI_API HPI_ClockGetTime(
        HPI_HSUBSYS     *phSubSys,
        HPI_HCLOCK      hClock,
        HW16            *pwHour,
        HW16            *pwMinute,
        HW16            *pwSecond,
        HW16            *pwMilliSecond
        );

/*/////////// */
/* PROFILE    */
/*/////////// */
HPI_ERR HPI_API HPI_ProfileOpenAll(
        HPI_HSUBSYS     *phSubSys,
        HW16            wAdapterIndex,
        HW16		    wProfileIndex,
        HPI_HPROFILE    *phProfile,
        HW16            *pwMaxProfiles
        );

HPI_ERR HPI_API HPI_ProfileGet(
        HPI_HSUBSYS     *phSubSys,
        HPI_HPROFILE    hProfile,
        HW16            wIndex,
        HW16            *pwSeconds,
        HW32            *pdwMicroSeconds,
        HW32            *pdwCallCount,
        HW32            *pdwMaxMicroSeconds,
        HW32            *pdwMinMicroSeconds
          );

#ifndef HPI_EXCLUDE_DEPRECATED
DEPRECATED HPI_ERR HPI_API HPI_ProfileGetIdleCount(
          HPI_HSUBSYS     *phSubSys,
          HPI_HPROFILE    hProfile,
          HW16            wIndex,
          HW32            *pdwCycleCount,
          HW32            *pdwCount
          );
#endif

HPI_ERR HPI_API HPI_ProfileStartAll(
          HPI_HSUBSYS     *phSubSys,
          HPI_HPROFILE    hProfile
          );

HPI_ERR HPI_API HPI_ProfileStopAll(
          HPI_HSUBSYS     *phSubSys,
          HPI_HPROFILE    hProfile
          );

HPI_ERR HPI_API HPI_ProfileGetName(
          HPI_HSUBSYS     *phSubSys,
          HPI_HPROFILE    hProfile,
          HW16            wIndex,
          char             *szProfileName,
          HW16            nProfileNameLength
          );

HPI_ERR HPI_API HPI_ProfileGetUtilization(
          HPI_HSUBSYS     *phSubSys,
          HPI_HPROFILE    hProfile,
          HW32            *pdwUtilization
          );


#ifdef HPI_SUPPORT_AES18
/* /////////////////////// */
/* AES18 functions */

#ifdef HAS_AES18_ON_HOST
/* !!! Test code only !!! */
void HPI_Aes18Init(void);
#endif

/* Block Generator */
HPI_ERR HPI_API HPI_AES18BGSetConfiguration(
        HPI_HSUBSYS     *phSubSys,
        HPI_HCONTROL    hControlHandle,
        HW16            wBlocksPerSec [HPI_AES18_MAX_CHANNELS],
        HW16            wPriorityEnableMask [HPI_AES18_MAX_CHANNELS],
          HW16            wOperatingMode [HPI_AES18_MAX_CHANNELS],
          HW16            wChannelMode
          );

/* Transmitter */
HPI_ERR HPI_API HPI_AES18TxGetInternalBufferState(
        HPI_HSUBSYS     *phSubSys,
        HPI_HCONTROL    hControlHandle,
        HW16            awInternalBufferBusy [HPI_AES18_MAX_CHANNELS][HPI_AES18_MAX_PRIORITIES]
        );

HPI_ERR HPI_API HPI_AES18TxGetInternalBufferSize(
        HPI_HSUBSYS     *phSubSys,
        HPI_HCONTROL    hControlHandle,
        HW16            awBytesPerBuffer[HPI_AES18_MAX_PRIORITIES]
        );

HPI_ERR HPI_API HPI_AES18TxSendMessage(
    HPI_HSUBSYS     *phSubSys,
    HPI_HCONTROL    hControlHandle,
    HW16            wChannel,
    HW8             *pbMessage,
    HW16            wMessageLength,
    HW16            wDestinationAddress,
    HW16            wPriorityIndex,
    HW16            wRepetitionIndex
	);

/* Receiver */
HPI_ERR HPI_API HPI_AES18RxSetAddress(
        HPI_HSUBSYS     *phSubSys,
        HPI_HCONTROL    hControlHandle,
        HW16            awDecoderAddress [HPI_AES18_MAX_CHANNELS]
        );

HPI_ERR HPI_API HPI_AES18RxGetInternalBufferSize(
        HPI_HSUBSYS     *phSubSys,
        HPI_HCONTROL    hControlHandle,
        HW16            awBytesPerBuffer[HPI_AES18_MAX_PRIORITIES]
        );

HPI_ERR HPI_API HPI_AES18RxGetInternalBufferState(
		HPI_HSUBSYS     *phSubSys,
		HPI_HCONTROL    hControlHandle,
        HW16            awFrameError[HPI_AES18_MAX_CHANNELS],
        HW16            awMessageWaiting [HPI_AES18_MAX_CHANNELS][HPI_AES18_MAX_PRIORITIES],
        HW16            awInternalBufferOverFlow  [HPI_AES18_MAX_CHANNELS][HPI_AES18_MAX_PRIORITIES],
        HW16            awMissedMessage [HPI_AES18_MAX_CHANNELS][HPI_AES18_MAX_PRIORITIES]
        );

HPI_ERR HPI_API HPI_AES18RxGetMessage(
        HPI_HSUBSYS     *phSubSys,
        HPI_HCONTROL    hControlHandle,
        HW16            wChannel,
        HW16            wPriority,
        HW16            wQueueSize,         /* in bytes */
        HW8             *pbMessage,
        HW16            *pwMessageLength    /* in bytes */
        );

#endif
/*//////////////////// */
/* UTILITY functions */

void HPI_API HPI_GetErrorText(
				HPI_ERR wError,
				char *pszErrorText
				);

HPI_ERR HPI_API HPI_FormatCreate(
                HPI_FORMAT *pFormat,
                HW16 wChannels,
                HW16 wFormat,
                HW32 dwSampleRate,
                HW32 dwBitRate,
                HW32 dwAttributes
                );

#ifndef HPI_EXCLUDE_DEPRECATED
DEPRECATED HPI_ERR HPI_API HPI_DataCreate(
                HPI_DATA *pData,
                HPI_FORMAT *pFormat,
                HW8 *pbData,
                HW32 dwDataSize
                );
#endif

/* Until it's verified, this function is for Windows OSs only */
#if defined(HPI_OS_WIN16) || defined(HPI_OS_WIN32_USER) || defined(INCLUDE_WINDOWS_ON_LINUX)

#ifdef HPI_OS_LINUX
#include <asimmdef.h>
#else
#include <mmreg.h>
#endif

HPI_ERR HPI_API HPI_WaveFormatToHpiFormat(
				const PWAVEFORMATEX lpFormatEx,
				HPI_FORMAT *pHpiFormat
				);

HPI_ERR HPI_API HPI_HpiFormatToWaveFormat(
				const HPI_FORMAT *pHpiFormat,
				PWAVEFORMATEX lpFormatEx
				);

#endif /* defined(HPI_OS_WIN16) || defined(HPI_OS_WIN32_USER) */

#endif /* ndef HPI_ON_DSP  */
/******************************************************************************/
/******************************************************************************/
/********                     HPI LOW LEVEL MESSAGES                  *********/
/******************************************************************************/
/******************************************************************************/
#ifndef HPI_EXCLUDE_IMPLEMENTATION
/** Pnp ids */
#define HPI_ID_ISAPNP_AUDIOSCIENCE  	0x0669   /*"ASI"  - actual is "ASX" - need to change */
#define HPI_PCI_VENDOR_ID_AUDIOSCIENCE 	0x175C   /**< PCI vendor ID that AudioScience uses */
#define HPI_PCI_VENDOR_ID_MOTOROLA  	0x1057   /**< PCI vendor ID that the DSP56301 has */
#define HPI_PCI_VENDOR_ID_TI        	0x104C   /**< PCI vendor ID that TI uses */

#define HPI_USB_VENDOR_ID_AUDIOSCIENCE 	0x1257
#define HPI_USB_W2K_TAG		0x57495341	/* "ASIW"	*/
#define HPI_USB_LINUX_TAG	0x4C495341	/* "ASIL"	*/

/******************************************* message types */
#define HPI_TYPE_MESSAGE                1
#define HPI_TYPE_RESPONSE               2
#define HPI_TYPE_DATA                   3
#define HPI_TYPE_SSX2BYPASS_MESSAGE		4

/******************************************* object types */
#define HPI_OBJ_SUBSYSTEM               1
#define HPI_OBJ_ADAPTER                 2
#define HPI_OBJ_OSTREAM                 3
#define HPI_OBJ_ISTREAM                 4
#define HPI_OBJ_MIXER                   5
#define HPI_OBJ_NODE                    6
#define HPI_OBJ_CONTROL                 7
#define HPI_OBJ_NVMEMORY                8
#define HPI_OBJ_GPIO                    9
#define HPI_OBJ_WATCHDOG                10
#define HPI_OBJ_CLOCK                   11
#define HPI_OBJ_PROFILE                 12
#define HPI_OBJ_CONTROLEX               13
#define HPI_OBJ_ASYNCEVENT              14

#define HPI_OBJ_MAXINDEX                14

/******************************************* methods/functions */

#define HPI_OBJ_FUNCTION_SPACING (0x100)
#define HPI_MAKE_INDEX(obj,index) (obj*HPI_OBJ_FUNCTION_SPACING+index)
#define HPI_EXTRACT_INDEX(fn) (fn & 0xff)

/* SUB-SYSTEM */
#define HPI_SUBSYS_OPEN                 HPI_MAKE_INDEX(HPI_OBJ_SUBSYSTEM,1)
#define HPI_SUBSYS_GET_VERSION          HPI_MAKE_INDEX(HPI_OBJ_SUBSYSTEM,2)
#define HPI_SUBSYS_GET_INFO             HPI_MAKE_INDEX(HPI_OBJ_SUBSYSTEM,3)
#define HPI_SUBSYS_FIND_ADAPTERS        HPI_MAKE_INDEX(HPI_OBJ_SUBSYSTEM,4)    /* SGT feb-3-97 */
#define HPI_SUBSYS_CREATE_ADAPTER       HPI_MAKE_INDEX(HPI_OBJ_SUBSYSTEM,5)    /* SGT feb-3-97 - not used any more */
#define HPI_SUBSYS_CLOSE                HPI_MAKE_INDEX(HPI_OBJ_SUBSYSTEM,6)
#define HPI_SUBSYS_DELETE_ADAPTER       HPI_MAKE_INDEX(HPI_OBJ_SUBSYSTEM,7)
#define HPI_SUBSYS_DRIVER_LOAD          HPI_MAKE_INDEX(HPI_OBJ_SUBSYSTEM,8)
#define HPI_SUBSYS_DRIVER_UNLOAD        HPI_MAKE_INDEX(HPI_OBJ_SUBSYSTEM,9)
/*SGT*/
#define HPI_SUBSYS_READ_PORT_8          HPI_MAKE_INDEX(HPI_OBJ_SUBSYSTEM,10)
#define HPI_SUBSYS_WRITE_PORT_8         HPI_MAKE_INDEX(HPI_OBJ_SUBSYSTEM,11)
/*SGT*/
#define HPI_SUBSYS_GET_NUM_ADAPTERS		HPI_MAKE_INDEX(HPI_OBJ_SUBSYSTEM,12)
#define HPI_SUBSYS_GET_ADAPTER          HPI_MAKE_INDEX(HPI_OBJ_SUBSYSTEM,13)

/* ADAPTER */
#define HPI_ADAPTER_OPEN                HPI_MAKE_INDEX(HPI_OBJ_ADAPTER,1)
#define HPI_ADAPTER_CLOSE               HPI_MAKE_INDEX(HPI_OBJ_ADAPTER,2)
#define HPI_ADAPTER_GET_INFO            HPI_MAKE_INDEX(HPI_OBJ_ADAPTER,3)
#define HPI_ADAPTER_GET_ASSERT          HPI_MAKE_INDEX(HPI_OBJ_ADAPTER,4)  /* AGE SEP-12-97 */
#define HPI_ADAPTER_TEST_ASSERT         HPI_MAKE_INDEX(HPI_OBJ_ADAPTER,5)  /* AGE SEP-12-97 */
#define HPI_ADAPTER_SET_MODE            HPI_MAKE_INDEX(HPI_OBJ_ADAPTER,6)
#define HPI_ADAPTER_GET_MODE            HPI_MAKE_INDEX(HPI_OBJ_ADAPTER,7)
#define HPI_ADAPTER_ENABLE_CAPABILITY   HPI_MAKE_INDEX(HPI_OBJ_ADAPTER,8)
#define HPI_ADAPTER_SELFTEST	        HPI_MAKE_INDEX(HPI_OBJ_ADAPTER,9)
#define HPI_ADAPTER_FIND_OBJECT	        HPI_MAKE_INDEX(HPI_OBJ_ADAPTER,10)
#define HPI_ADAPTER_QUERY_FLASH       HPI_MAKE_INDEX(HPI_OBJ_ADAPTER,11)
#define HPI_ADAPTER_START_FLASH       HPI_MAKE_INDEX(HPI_OBJ_ADAPTER,12)
#define HPI_ADAPTER_PROGRAM_FLASH       HPI_MAKE_INDEX(HPI_OBJ_ADAPTER,13)
#define HPI_ADAPTER_SET_PROPERTY        HPI_MAKE_INDEX(HPI_OBJ_ADAPTER,14)
#define HPI_ADAPTER_GET_PROPERTY        HPI_MAKE_INDEX(HPI_OBJ_ADAPTER,15)
#define HPI_ADAPTER_ENUM_PROPERTY       HPI_MAKE_INDEX(HPI_OBJ_ADAPTER,16)

/* OUTPUT STREAM */
#define HPI_OSTREAM_OPEN                HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,1)
#define HPI_OSTREAM_CLOSE               HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,2)
#define HPI_OSTREAM_WRITE               HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,3)
#define HPI_OSTREAM_START               HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,4)
#define HPI_OSTREAM_STOP                HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,5)
#define HPI_OSTREAM_RESET               HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,6)
#define HPI_OSTREAM_GET_INFO            HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,7)
#define HPI_OSTREAM_QUERY_FORMAT        HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,8)
#define HPI_OSTREAM_DATA                HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,9)
#define HPI_OSTREAM_SET_VELOCITY        HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,10)
#define HPI_OSTREAM_SET_PUNCHINOUT      HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,11)
#define HPI_OSTREAM_SINEGEN             HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,12)
#define HPI_OSTREAM_ANC_RESET           HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,13)     /* MP2 ancillary data reset */
#define HPI_OSTREAM_ANC_GET_INFO        HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,14)	   /* MP2 ancillary data get info */
#define HPI_OSTREAM_ANC_READ            HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,15)	   /* MP2 ancillary data read */
#define HPI_OSTREAM_SET_TIMESCALE       HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,16)
#define HPI_OSTREAM_SET_FORMAT          HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,17)
#define HPI_OSTREAM_HOSTBUFFER_ALLOC    HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,18)
#define HPI_OSTREAM_HOSTBUFFER_FREE     HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,19)
#define HPI_OSTREAM_GROUP_ADD			HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,20)
#define HPI_OSTREAM_GROUP_GETMAP		HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,21)
#define HPI_OSTREAM_GROUP_RESET			HPI_MAKE_INDEX(HPI_OBJ_OSTREAM,22)

#define HPI_OSTREAM_FUNCTION_COUNT		(22)

/* INPUT STREAM */
#define HPI_ISTREAM_OPEN                HPI_MAKE_INDEX(HPI_OBJ_ISTREAM,1)
#define HPI_ISTREAM_CLOSE               HPI_MAKE_INDEX(HPI_OBJ_ISTREAM,2)
#define HPI_ISTREAM_SET_FORMAT          HPI_MAKE_INDEX(HPI_OBJ_ISTREAM,3)     /* SGT mar-19-97 */
#define HPI_ISTREAM_READ                HPI_MAKE_INDEX(HPI_OBJ_ISTREAM,4)     /* SGT mar-19-97 */
#define HPI_ISTREAM_START               HPI_MAKE_INDEX(HPI_OBJ_ISTREAM,5)     /* SGT mar-19-97 */
#define HPI_ISTREAM_STOP                HPI_MAKE_INDEX(HPI_OBJ_ISTREAM,6)
#define HPI_ISTREAM_RESET               HPI_MAKE_INDEX(HPI_OBJ_ISTREAM,7)     /* SGT mar-19-97 */
#define HPI_ISTREAM_GET_INFO            HPI_MAKE_INDEX(HPI_OBJ_ISTREAM,8)
#define HPI_ISTREAM_QUERY_FORMAT        HPI_MAKE_INDEX(HPI_OBJ_ISTREAM,9)
#define HPI_ISTREAM_ANC_RESET           HPI_MAKE_INDEX(HPI_OBJ_ISTREAM,10)		/* MP2 ancillary data reset */
#define HPI_ISTREAM_ANC_GET_INFO        HPI_MAKE_INDEX(HPI_OBJ_ISTREAM,11)
#define HPI_ISTREAM_ANC_WRITE           HPI_MAKE_INDEX(HPI_OBJ_ISTREAM,12)
#define HPI_ISTREAM_HOSTBUFFER_ALLOC    HPI_MAKE_INDEX(HPI_OBJ_ISTREAM,13)
#define HPI_ISTREAM_HOSTBUFFER_FREE     HPI_MAKE_INDEX(HPI_OBJ_ISTREAM,14)
#define HPI_ISTREAM_GROUP_ADD			HPI_MAKE_INDEX(HPI_OBJ_ISTREAM,15)
#define HPI_ISTREAM_GROUP_GETMAP		HPI_MAKE_INDEX(HPI_OBJ_ISTREAM,16)
#define HPI_ISTREAM_GROUP_RESET			HPI_MAKE_INDEX(HPI_OBJ_ISTREAM,17)

#define HPI_ISTREAM_FUNCTION_COUNT		(17)

/* MIXER */
/* NOTE: (EWB 2003-05-13)
   GET_INFO, GET_NODE_INFO, SET_CONNECTION, GET_CONNECTIONS are not currently used */
#define HPI_MIXER_OPEN                  HPI_MAKE_INDEX(HPI_OBJ_MIXER,1)
#define HPI_MIXER_CLOSE                 HPI_MAKE_INDEX(HPI_OBJ_MIXER,2)
#define HPI_MIXER_GET_INFO              HPI_MAKE_INDEX(HPI_OBJ_MIXER,3)     /* gets list of source and dest node objects */
#define HPI_MIXER_GET_NODE_INFO         HPI_MAKE_INDEX(HPI_OBJ_MIXER,4)     /* gets info on a particular node */
#define HPI_MIXER_GET_CONTROL           HPI_MAKE_INDEX(HPI_OBJ_MIXER,5)     /* gets specified control type on given connection */
#define HPI_MIXER_SET_CONNECTION        HPI_MAKE_INDEX(HPI_OBJ_MIXER,6)     /* between a destination and source */
#define HPI_MIXER_GET_CONNECTIONS       HPI_MAKE_INDEX(HPI_OBJ_MIXER,7)     /* for a given destination */
#define HPI_MIXER_GET_CONTROL_BY_INDEX  HPI_MAKE_INDEX(HPI_OBJ_MIXER,8)     /* get a control index */
#define HPI_MIXER_GET_CONTROL_ARRAY_BY_INDEX  HPI_MAKE_INDEX(HPI_OBJ_MIXER,9)     /* get a control array index (internal call for the moment) */
#define HPI_MIXER_GET_CONTROL_MULTIPLE_VALUES HPI_MAKE_INDEX(HPI_OBJ_MIXER,10)    /* get an array of control values (internal call for the moment) */
#define HPI_MIXER_STORE                 HPI_MAKE_INDEX(HPI_OBJ_MIXER,11)     /* Access the mixer control store */

#define HPI_MIXER_FUNCTION_COUNT	11

/* MIXER CONTROLS */
#define HPI_CONTROL_GET_INFO            HPI_MAKE_INDEX(HPI_OBJ_CONTROL,1)   /* used by HPI_ControlQuery() */
#define HPI_CONTROL_GET_STATE           HPI_MAKE_INDEX(HPI_OBJ_CONTROL,2)
#define HPI_CONTROL_SET_STATE           HPI_MAKE_INDEX(HPI_OBJ_CONTROL,3)

/* NONVOL MEMORY */
#define HPI_NVMEMORY_OPEN               HPI_MAKE_INDEX(HPI_OBJ_NVMEMORY,1)
#define HPI_NVMEMORY_READ_BYTE          HPI_MAKE_INDEX(HPI_OBJ_NVMEMORY,2)
#define HPI_NVMEMORY_WRITE_BYTE         HPI_MAKE_INDEX(HPI_OBJ_NVMEMORY,3)

/* GPIO */
#define HPI_GPIO_OPEN                   HPI_MAKE_INDEX(HPI_OBJ_GPIO,1)
#define HPI_GPIO_READ_BIT               HPI_MAKE_INDEX(HPI_OBJ_GPIO,2)
#define HPI_GPIO_WRITE_BIT              HPI_MAKE_INDEX(HPI_OBJ_GPIO,3)
#define HPI_GPIO_READ_ALL               HPI_MAKE_INDEX(HPI_OBJ_GPIO,4)

/* ASYNC EVENT */
#define HPI_ASYNCEVENT_OPEN             HPI_MAKE_INDEX(HPI_OBJ_ASYNCEVENT,1)
#define HPI_ASYNCEVENT_CLOSE            HPI_MAKE_INDEX(HPI_OBJ_ASYNCEVENT,2)
#define HPI_ASYNCEVENT_WAIT             HPI_MAKE_INDEX(HPI_OBJ_ASYNCEVENT,3)
#define HPI_ASYNCEVENT_GETCOUNT         HPI_MAKE_INDEX(HPI_OBJ_ASYNCEVENT,4)
#define HPI_ASYNCEVENT_GET              HPI_MAKE_INDEX(HPI_OBJ_ASYNCEVENT,5)
#define HPI_ASYNCEVENT_SENDEVENTS       HPI_MAKE_INDEX(HPI_OBJ_ASYNCEVENT,6)

/* WATCH-DOG */
#define HPI_WATCHDOG_OPEN               HPI_MAKE_INDEX(HPI_OBJ_WATCHDOG,1)
#define HPI_WATCHDOG_SET_TIME           HPI_MAKE_INDEX(HPI_OBJ_WATCHDOG,2)
#define HPI_WATCHDOG_PING               HPI_MAKE_INDEX(HPI_OBJ_WATCHDOG,3)

/* CLOCK */
#define HPI_CLOCK_OPEN                  HPI_MAKE_INDEX(HPI_OBJ_CLOCK,1)
#define HPI_CLOCK_SET_TIME              HPI_MAKE_INDEX(HPI_OBJ_CLOCK,2)
#define HPI_CLOCK_GET_TIME              HPI_MAKE_INDEX(HPI_OBJ_CLOCK,3)

/* PROFILE */
#define HPI_PROFILE_OPEN_ALL            HPI_MAKE_INDEX(HPI_OBJ_PROFILE,1)
#define HPI_PROFILE_START_ALL           HPI_MAKE_INDEX(HPI_OBJ_PROFILE,2)
#define HPI_PROFILE_STOP_ALL            HPI_MAKE_INDEX(HPI_OBJ_PROFILE,3)
#define HPI_PROFILE_GET                 HPI_MAKE_INDEX(HPI_OBJ_PROFILE,4)
#define HPI_PROFILE_GET_IDLECOUNT       HPI_MAKE_INDEX(HPI_OBJ_PROFILE,5)
#define HPI_PROFILE_GET_NAME            HPI_MAKE_INDEX(HPI_OBJ_PROFILE,6)
#define HPI_PROFILE_GET_UTILIZATION     HPI_MAKE_INDEX(HPI_OBJ_PROFILE,7)


/* ////////////////////////////////////////////////////////////////////// */
/* STRUCTURES */
#ifndef DISABLE_PRAGMA_PACK1
#pragma pack(push,1)
#endif

/** PCI bus resource */
typedef struct sHPI_PCI
{
	HW32 __iomem *	apMemBase[HPI_MAX_ADAPTER_MEM_SPACES];
	struct pci_dev * pOsData;

#ifndef HPI_64BIT /* keep structure size constant */
	HW32	dwPadding[HPI_MAX_ADAPTER_MEM_SPACES+1];
#endif
	HW16 	wVendorId;
	HW16 	wDeviceId;
	HW16   wSubSysVendorId;
	HW16   wSubSysDeviceId;
	HW16	wBusNumber;
	HW16	wDeviceNumber;
	HW32	wInterrupt;
} HPI_PCI;

typedef struct
{
	union
		{
			HPI_PCI * Pci;
		} r;
#ifndef HPI_64BIT /* keep structure size constant */
		HW32	dwPadTo64;
#endif
		HW16    wBusType;               /* HPI_BUS_PNPISA, _PCI, _USB etc */
		HW16    wPadding;

} HPI_RESOURCE;

/** Format info used inside HPI_MESSAGE. Not the same as public API HPI_FORMAT */
typedef  struct  sHPI_MSG_FORMAT {
        HW32    dwSampleRate;   /**< 11025, 32000, 44100 ... */
        HW32    dwBitRate;      /**< for MPEG */
        HW32    dwAttributes;   /**< Stereo/JointStereo/Mono */
        HW16    wChannels;      /**< 1,2..., (or ancillary mode or idle bit */
        HW16    wFormat;        /**< HPI_FORMAT_PCM16, _AC2, _MPEG ... see \ref formats. */
    } HPI_MSG_FORMAT;

/**  Buffer+format structure.
     Must be kept 7 * 32 bits to match public HPI_DATA struct */
typedef struct sHPI_MSG_DATA
{
    HPI_MSG_FORMAT Format;
	HW8 *           pbData;
#ifdef HPI_OS_DSP_563XX
	HW16		wPadding; /* 563xx pointer is only 24 bits, effectively HW16*/
#endif
#ifndef HPI_64BIT
	HW32		dwPadding;
#endif
	HW32            dwDataSize;
} HPI_MSG_DATA;

#ifndef HPI_ON_DSP
/** HPI_DATA structure used up to 3.04 driver */
typedef struct
{
	HPI_FORMAT      Format;
	HW32            pbData;
	HW32            dwDataSize;
} HPI_DATA_LEGACY32;
#endif

#ifdef HPI_64BIT
/* Compatibility version of HPI_DATA */
typedef struct
{
	HPI_MSG_FORMAT	Format;
	HW32            pbData;
	HW32		dwPadding;
	HW32            dwDataSize;
} HPI_DATA_COMPAT32;
#endif

typedef struct
{
	HPI_MSG_FORMAT	reserved;	/**< placehoder for backward compatability (see dwBufferSize) */
	HW32		dwCommand;	/**< HPI_BUFFER_CMD_xxx*/
	HW32		dwPciAddress;	/**< PCI physical address of buffer for DSP DMA */
	HW32		dwBufferSize;	/**< must line up with dwDataSize of HPI_DATA*/
} HPI_BUFFER;

typedef struct
{
	HPI_RESOURCE Resource;
}HPI_SUBSYS_MSG;

typedef struct
{
		  HW32    dwVersion;
		  HW32    dwData;							/* used to return extended version */
		  HW16    wNumAdapters;                       /* number of adapters */
		  HW16    wAdapterIndex;
		  HW16    awAdapterList[HPI_MAX_ADAPTERS];    /* array of adapters */
}HPI_SUBSYS_RES;


typedef struct
{
	HW32    dwAdapterMode;          /* adapter mode */
	HW16    wAssertId;              /* assert number for "test assert" call
											also wObjectIndex for find object call
											also wQueryOrSet for HPI_AdapterSetModeEx() */
	HW16    wObjectType;            /* for adapter find object call */
} HPI_ADAPTER_MSG;

typedef union {
		HPI_ADAPTER_MSG adapter;
		struct {
			HW32 dwOffset;
		} query_flash;
		struct {
			HW32 dwOffset;
			HW32 dwLength;
			HW32 dwKey;
		} start_flash;
		struct {
			HW32 dwChecksum;
			HW16 wSequence;
			HW16 wLength;
		} program_flash;
#ifndef HPI_OS_DSP_563XX		/* gets rid of compiler internal assert */
		struct {
			HW16 wProperty;
			HW16 wParameter1;
			HW16 wParameter2;
		} property_set;
		struct {
			HW16 wIndex;
			HW16 wWhat;
			HW16 wPropertyIndex;
		} property_enum;
#endif
} HPI_ADAPTERX_MSG;

typedef struct
{
		HW32        dwSerialNumber;
        HW16        wAdapterType;
        HW16        wAdapterIndex; /* Is this needed? also used for wDspIndex */
        HW16        wNumIStreams;
        HW16        wNumOStreams;
        HW16        wNumMixers;
        HW16        wVersion;
        HW8         szAdapterAssert[STR_SIZE(HPI_STRING_LEN)];
} HPI_ADAPTER_RES;

typedef union {
		HPI_ADAPTER_RES adapter;
		struct {
			HW32 dwChecksum;
			HW32 dwLength;
			HW32 dwVersion;
		} query_flash;
		struct {
			HW16 wSequence;
		} program_flash;
#ifndef HPI_OS_DSP_563XX		/* gets rid of compiler internal assert */
		struct {
			HW16 wParameter1;
			HW16 wParameter2;
		} property_get;
#endif
} HPI_ADAPTERX_RES;

typedef struct
{
		  union
		  {
				HPI_MSG_DATA        Data;
#ifndef HPI_ON_DSP
              HPI_DATA_LEGACY32 Data32;
#endif
              HW16            wVelocity;
				HPI_PUNCHINOUT  Pio;
				HW32            dwTimeScale;
				HPI_BUFFER		Buffer;
				HPI_STREAMID	Stream;
		  }u;
		  HW16            wStreamIndex;
		  HW16            wIStreamIndex;
} HPI_STREAM_MSG;

typedef struct
{
	union
	{
		struct {
			HW32    dwBufferSize;		    /* size of hardware buffer */
			HW32    dwDataAvailable;		/* OutStream - data to play, InStream - data recorded */
			HW32    dwSamplesTransferred;	/* OutStream - samples played, InStream - samples recorded */
			HW32    dwAuxiliaryDataAvailable; /* Adapter - OutStream - data to play, InStream - data recorded */
			HW16    wState;				/* HPI_STATE_PLAYING, _STATE_STOPPED */
			HW16    wPadding;
		} stream_info;
		struct {
			HW32    dwBufferSize;           /* size of hardware buffer */
			HW32    dwDataAvailable;        /* OutStream - data to play, InStream - data recorded */
			HW32    dwSamplesTransfered;    /* OutStream - samples played, InStream - samples recorded */
			HW16    wState;                 /* HPI_STATE_PLAYING, _STATE_STOPPED */
			HW16    wOStreamIndex;
			HW16    wIStreamIndex;
			HW16    wPadding;
			HW32    dwAuxiliaryDataAvailable; /* Adapter - OutStream - data to play */
		} legacy_stream_info;
		struct {
			HW32    dwOutStreamGroupMap;  /* bitmap of grouped OutStreams */
			HW32    dwInStreamGroupMap;	/* bitmap of grouped InStreams */
		} group_info;
	}u;
} HPI_STREAM_RES;

typedef struct
{
		  HW16    wControlIndex;
		  HW16    wControlType;   /* = HPI_CONTROL_METER _VOLUME etc */
		  HW16    wPadding1;      /* Maintain alignment of subsequent fields */
		  HW16    wNodeType1;     /* = HPI_SOURCENODE_LINEIN etc */
		  HW16    wNodeIndex1;    /* = 0..N */
		  HW16    wNodeType2;
		  HW16    wNodeIndex2;
		  HW16    wPadding2;      /* round to 4 bytes */
} HPI_MIXER_MSG;

typedef struct
{
		  HW16    wSrcNodeType;     /* = HPI_SOURCENODE_LINEIN etc */
		  HW16    wSrcNodeIndex;    /* = 0..N */
		  HW16    wDstNodeType;
		  HW16    wDstNodeIndex;
          HW16    wControlIndex;  /* Also returns controlType for HPI_MixerGetControlByIndex */
          HW16    wDspIndex;      /* may indicate which DSP the control is located on */
} HPI_MIXER_RES;

typedef union HPI_MIXERX_MSG
{
	struct {
		HW16    wStartingIndex;
		HW16    wFlags;
		HW32    dwLengthInBytes;  /* length in bytes of pData */
		HW32    pData;	      	/* pointer to a data array */
	} gcabi;
	struct {
		HW16	wCommand;
		HW16	wIndex;
	} store; /* for HPI_MIXER_STORE message */
} HPI_MIXERX_MSG;

typedef union
{
	struct {
		  HW32    dwBytesReturned;	/* number of items returned */
		  HW32	  pData;	      	/* pointer to data array */
		  HW16	  wMoreToDo;		/* indicates if there is more to do */
	} gcabi;
} HPI_MIXERX_RES;


typedef struct
{
	HW32    dwParam1;        /* generic parameter 1 */
	HW32    dwParam2;        /* generic parameter 2 */
	short   anLogValue[HPI_MAX_CHANNELS];
	HW16    wAttribute;     /* control attribute or property */
	HW16    wControlIndex;
}HPI_CONTROL_MSG;

typedef struct {
	union {
		struct {
			HW32    dwParam1;        /* generic parameter 1 */
			HW32    dwParam2;        /* generic parameter 2 */
			short   anLogValue[HPI_MAX_CHANNELS];
		} old;
		union {
			 HW32 dwFrequency;
			 HW32 dwGain;
			 HW32 dwBand;
	         struct {
	              HW32 dwMode;
	              HW32 dwValue;
	         } mode;
		} tuner;
	} u;
	HW16    wAttribute;     /* control attribute or property */
	HW16    wControlIndex;
} HPI_CONTROL_UNION_MSG;

typedef struct
{
	/* Could make union.  dwParam, anLogValue never used in same response */
          HW32    dwParam1;
          HW32    dwParam2;
          short   anLogValue[HPI_MAX_CHANNELS];
/*          short   anLogValue2[HPI_MAX_CHANNELS]; */ /* to return RMS and Peak in same call */
} HPI_CONTROL_RES;

typedef union {
		struct {
	          HW32    dwParam1;
       		  HW32    dwParam2;
	          short   anLogValue[HPI_MAX_CHANNELS];
		} old;
		union {
	          HW32 dwBand;
	          HW32 dwFrequency;
	          HW32 dwGain;
	          HW32 dwLevel;
	          struct {
	              HW32 dwData[2];
	              HW32 dwBLER;
	          } rds;
		} tuner;
} HPI_CONTROL_UNION_RES;

/* HPI_CONTROLX_STRUCTURES */

/* Message */
#ifdef HPI_SUPPORT_AES18
typedef struct
{
     HW16    wBlocksPerSec[HPI_AES18_MAX_CHANNELS];
     HW16    wPriorityMask [HPI_AES18_MAX_CHANNELS];
     HW16    wOperatingMode [HPI_AES18_MAX_CHANNELS];
     HW16    wChannelMode;
     HW16    wPadding;
} HPI_CONTROLX_MSG_AES18BG;

typedef struct
{
     HW16    wAddress[HPI_AES18_MAX_CHANNELS];
} HPI_CONTROLX_MSG_AES18RX_ADDRESS;

typedef struct
{
     HW32    dwpbMessage;
     HW16    wChannel;
	  HW16    wMessageLength;
	  HW16    wDestinationAddress;
	  HW16    wPriorityIndex;
	  HW16    wRepetitionIndex;
	  HW16    wPadding;
} HPI_CONTROLX_MSG_AES18TX_SEND_MESSAGE;

typedef struct
{
	  HW32    dwpbMessage; /* actually a pointer to byte */
	  HW16    wChannel;
	 HW16    wPriority;
	 HW16    wQueueSize;
	 HW16    wPadding;
} HPI_CONTROLX_MSG_AES18RX_GET_MESSAGE;
#endif

/** Used for all HMI variables where max length <= 8 bytes
*/
typedef struct
{
  HW32 dwHmiAddress;
  HW32 dwByteCount;
  HW32 dwData[2];
} HPI_CONTROLX_MSG_COBRANET_DATA;

/** Used for string data, and for packet bridge
*/
typedef struct
{
  HW32 dwHmiAddress;
  HW32 dwByteCount;
  HW8 * pbData;
  #ifndef HPI_64BIT
  HW32 dwPadding;
  #endif
} HPI_CONTROLX_MSG_COBRANET_BIGDATA;

/** Used for generic data
*/
typedef struct
{
  HW32 dwParam1;
  HW32 dwParam2;
} HPI_CONTROLX_MSG_GENERIC;

typedef struct
{
	union {
#if defined (HPI_SUPPORT_AES18)
		HPI_CONTROLX_MSG_AES18BG                aes18bg;
		HPI_CONTROLX_MSG_AES18RX_ADDRESS        aes18rx_address;
		HPI_CONTROLX_MSG_AES18TX_SEND_MESSAGE   aes18tx_send_message;
		HPI_CONTROLX_MSG_AES18RX_GET_MESSAGE    aes18rx_get_message;
#endif
		HPI_CONTROLX_MSG_COBRANET_DATA          cobranet_data;
		HPI_CONTROLX_MSG_COBRANET_BIGDATA       cobranet_bigdata;
		HPI_CONTROLX_MSG_GENERIC                generic;
		/* nothing extra to send for status read */
	} u;
	HW16    wControlIndex;
	HW16    wAttribute;     /* control attribute or property */
} HPI_CONTROLX_MSG;


/* Response */


#ifdef HPI_SUPPORT_AES18
typedef struct
{
	 HW16    awFrameErrorPacked;
	 HW16    awMessageWaitingPacked;
	 HW16    awInternalBufferOverFlowPacked;
    HW16    awMissedMessagePacked;
} HPI_CONTROLX_RES_AES18RX_BUFFER_STATE;

typedef struct
{
    HW16    awBytesPerBuffer[HPI_AES18_MAX_PRIORITIES];
} HPI_CONTROLX_RES_AES18RX_BUFFER_SIZE;

typedef struct
{
    HW16    wReturnedMessageSize;
} HPI_CONTROLX_RES_AES18RX_GET_MESSAGE;

typedef struct
{
    HW16    wInternalBufferBusyPacked;
} HPI_CONTROLX_RES_AES18TX_BUFFER_STATE;

typedef struct
{
    HW16    awBytesPerBuffer[HPI_AES18_MAX_PRIORITIES];
} HPI_CONTROLX_RES_AES18TX_BUFFER_SIZE;
#endif

/**
*/
typedef struct
{
  HW32 dwByteCount;
  HW32 dwData[2];
} HPI_CONTROLX_RES_COBRANET_DATA;

typedef struct
{
  HW32 dwByteCount;
} HPI_CONTROLX_RES_COBRANET_BIGDATA;

typedef struct
{
    HW32 dwStatus;
    HW32 dwReadableSize;
	HW32 dwWriteableSize;
} HPI_CONTROLX_RES_COBRANET_STATUS;

typedef struct
{
    HW32 dwParam1;
    HW32 dwParam2;
} HPI_CONTROLX_RES_GENERIC;

typedef struct
{
	 union {
#ifdef HPI_SUPPORT_AES18
	HPI_CONTROLX_RES_AES18RX_BUFFER_STATE   aes18rx_internal_buffer_state;
	HPI_CONTROLX_RES_AES18RX_BUFFER_SIZE    aes18rx_internal_buffer_size;
	HPI_CONTROLX_RES_AES18RX_GET_MESSAGE    aes18rx_get_message;
	HPI_CONTROLX_RES_AES18TX_BUFFER_STATE   aes18tx_internal_buffer_state;
	HPI_CONTROLX_RES_AES18TX_BUFFER_SIZE    aes18tx_internal_buffer_size;
#endif
	HPI_CONTROLX_RES_COBRANET_BIGDATA       cobranet_bigdata;
	HPI_CONTROLX_RES_COBRANET_DATA          cobranet_data;
	HPI_CONTROLX_RES_COBRANET_STATUS        cobranet_status;
	HPI_CONTROLX_RES_GENERIC                generic;
	 }u;
} HPI_CONTROLX_RES;


typedef struct
{
        HW16    wIndex;
        HW16    wData;
} HPI_NVMEMORY_MSG;

typedef struct
{
        HW16    wSizeInBytes;
        HW16    wData;
} HPI_NVMEMORY_RES;

typedef struct
{
          HW16    wBitIndex;
				 HW16    wBitData;
} HPI_GPIO_MSG;

typedef struct
{
          HW16    wNumberInputBits;
          HW16    wNumberOutputBits;
          HW16    wBitData;
			 HW16    wPadding;
} HPI_GPIO_RES;


typedef struct
{
          HW32	dwEvents;
          HW16  wMaximumEvents;
          HW16  wPadding;
} HPI_ASYNC_MSG;

typedef struct
{
	union {
		struct {
			HW16 wCount;
		} count;
		struct {
          HW32	dwEvents;
          HW16  wNumberReturned;
          HW16  wPadding;
		} get;
		HPI_ASYNC_EVENT event;
	} u;
} HPI_ASYNC_RES;

typedef struct
{
        HW32    dwTimeMs;
} HPI_WATCHDOG_MSG;

typedef struct
{
    HW32    dwTimeMs;
} HPI_WATCHDOG_RES;

typedef struct
{
        HW16    wHours;
        HW16    wMinutes;
        HW16    wSeconds;
        HW16    wMilliSeconds;
} HPI_CLOCK_MSG;

typedef struct
{
        HW16    wSizeInBytes;
		  HW16    wHours;
        HW16    wMinutes;
        HW16    wSeconds;
		  HW16    wMilliSeconds;
        HW16    wPadding;
} HPI_CLOCK_RES;

typedef struct
{
        HW16    wIndex;
        HW16    wPadding;
} HPI_PROFILE_MSG;


typedef struct
{
          HW16    wMaxProfiles;
} HPI_PROFILE_RES_OPEN;

typedef struct
{
          HW32    dwMicroSeconds;
             HW32    dwCallCount;
          HW32    dwMaxMicroSeconds;
			 HW32    dwMinMicroSeconds;
			 HW16    wSeconds;
} HPI_PROFILE_RES_TIME;

typedef struct
{
          HW16  szName[16];             /* HW8 messes up response size for 56301 DSP */
} HPI_PROFILE_RES_NAME;

typedef struct
{
	 union {
        HPI_PROFILE_RES_OPEN    o;
        HPI_PROFILE_RES_TIME    t;
        HPI_PROFILE_RES_NAME    n;
    } u;
} HPI_PROFILE_RES;


/* the size of the part of the message outside the union.  MUST update if more elements are added */
typedef struct
{
    HW16    wSize;
    HW16    wType;          /* HPI_MSG_MESSAGE, HPI_MSG_RESPONSE */
    HW16    wObject;        /* HPI_OBJ_SUBSYS, _OBJ_ADAPTER, _OBJ_DEVICE, _OBJ_MIXER */
    HW16    wFunction;      /* HPI_SUBSYS_xxx, HPI_ADAPTER_xxx */
    HW16    wAdapterIndex;  /* the adapter index */
    HW16    wDspIndex;      /* the dsp index on the adapter */
} HPI_MESSAGE_HEADER;

#define HPI_MESSAGE_FIXED_SIZE (sizeof(HPI_MESSAGE_HEADER))

typedef struct sHPI_MESSAGE
{
	/* following fields must match HPI_MESSAGE_HEADER */
	HW16    wSize;
	HW16    wType;          /* HPI_TYPE_MESSAGE, HPI_TYPE_RESPONSE */
	HW16    wObject;        /* HPI_OBJ_SUBSYS, _OBJ_ADAPTER, _OBJ_DEVICE, _OBJ_MIXER */
	HW16    wFunction;      /* HPI_SUBSYS_xxx, HPI_ADAPTER_xxx */
	HW16    wAdapterIndex;  /* the adapter index */
	HW16    wDspIndex;      /* the dsp index on the adapter */
	union
	{
		HPI_SUBSYS_MSG          s; /* Subsys; */
		HPI_ADAPTER_MSG         a; /* Adapter; */
		HPI_ADAPTERX_MSG        ax; /* Adapter; */
		HPI_STREAM_MSG          d; /* Stream; */
		HPI_MIXER_MSG           m; /* Mixer; */
		HPI_MIXERX_MSG          mx; /* extended Mixer; */
		HPI_CONTROL_MSG         c; /* mixer Control; */
		HPI_CONTROL_UNION_MSG	cu; /* identical to HPI_CONTROL_MSG, but field naming is improved */
		HPI_CONTROLX_MSG        cx;/* extended mixer Control; */
		HPI_NVMEMORY_MSG        n; /* non-vol memory; */
		HPI_GPIO_MSG            l; /* digital i/o */
		HPI_WATCHDOG_MSG        w; /* watch-dog */
		HPI_CLOCK_MSG           t; /* dsp time */
		HPI_PROFILE_MSG         p; /* profile */
		HPI_ASYNC_MSG           as; /* async event */
#ifdef HPI_MESSAGE_FORCE_SIZE
		HW8 _x[HPI_MESSAGE_FORCE_SIZE-HPI_MESSAGE_FIXED_SIZE];
#endif
	} u;
} HPI_MESSAGE;

#define HPI_MESSAGE_SIZE_BY_OBJECT { \
	HPI_MESSAGE_FIXED_SIZE ,   /* Default, no object type 0 */ \
	HPI_MESSAGE_FIXED_SIZE + sizeof(HPI_SUBSYS_MSG),\
	HPI_MESSAGE_FIXED_SIZE + sizeof(HPI_ADAPTERX_MSG),\
	HPI_MESSAGE_FIXED_SIZE + sizeof(HPI_STREAM_MSG),\
	HPI_MESSAGE_FIXED_SIZE + sizeof(HPI_STREAM_MSG),\
	HPI_MESSAGE_FIXED_SIZE + sizeof(HPI_MIXER_MSG),\
	HPI_MESSAGE_FIXED_SIZE ,   /* No NODE message */ \
	HPI_MESSAGE_FIXED_SIZE + sizeof(HPI_CONTROL_MSG),\
	HPI_MESSAGE_FIXED_SIZE + sizeof(HPI_NVMEMORY_MSG),\
	HPI_MESSAGE_FIXED_SIZE + sizeof(HPI_GPIO_MSG),\
	HPI_MESSAGE_FIXED_SIZE + sizeof(HPI_WATCHDOG_MSG),\
	HPI_MESSAGE_FIXED_SIZE + sizeof(HPI_CLOCK_MSG),\
	HPI_MESSAGE_FIXED_SIZE + sizeof(HPI_PROFILE_MSG),\
	HPI_MESSAGE_FIXED_SIZE + sizeof(HPI_CONTROLX_MSG),\
 	HPI_MESSAGE_FIXED_SIZE + sizeof(HPI_ASYNC_MSG) \
}


/* the size of the part of the response outside the union.  MUST update if more elements are added */

typedef struct sHPI_RESPONSE_HEADER
{
    HW16            wSize;
    HW16            wType;          /* HPI_MSG_MESSAGE, HPI_MSG_RESPONSE */
    HW16            wObject;        /* HPI_OBJ_SUBSYS, _OBJ_ADAPTER, _OBJ_DEVICE, _OBJ_MIXER */
    HW16            wFunction;      /* HPI_SUBSYS_xxx, HPI_ADAPTER_xxx */
    HW16            wError;         /* HPI_ERROR_xxx */
    HW16            wSpecificError; /* Adapter specific error */
} HPI_RESPONSE_HEADER;
#define HPI_RESPONSE_FIXED_SIZE (sizeof(HPI_RESPONSE_HEADER))

typedef struct sHPI_RESPONSE
{	/* following fields must match HPI_RESPONSE_HEADER */
        HW16            wSize;
        HW16            wType;          /* HPI_MSG_MESSAGE, HPI_MSG_RESPONSE */
        HW16            wObject;        /* HPI_OBJ_SUBSYS, _OBJ_ADAPTER, _OBJ_DEVICE, _OBJ_MIXER */
        HW16            wFunction;      /* HPI_SUBSYS_xxx, HPI_ADAPTER_xxx */
        HW16            wError;         /* HPI_ERROR_xxx */
        HW16            wSpecificError; /* Adapter specific error */
        union
        {
                HPI_SUBSYS_RES          s; /* SubSys; */
                HPI_ADAPTER_RES         a; /* Adapter; */
                HPI_ADAPTERX_RES        ax; /* Adapter; */
                HPI_STREAM_RES          d; /* stream; */
                HPI_MIXER_RES           m; /* Mixer; */
                HPI_MIXERX_RES          mx; /* extended Mixer; */
                HPI_CONTROL_RES         c; /* mixer Control; */
				HPI_CONTROL_UNION_RES	cu; /* identical to HPI_CONTROL_RES, but field naming is improved. */
                HPI_CONTROLX_RES        cx;/* extended mixer Control; */
                HPI_NVMEMORY_RES        n; /* non-vol memory; */
                HPI_GPIO_RES            l; /* digital i/o */
                HPI_WATCHDOG_RES        w; /* watch-dog */
                HPI_CLOCK_RES           t; /* dsp time */
                HPI_PROFILE_RES         p; /* profile */
                HPI_ASYNC_RES           as; /* async event */
        }u;
} HPI_RESPONSE;


#define HPI_RESPONSE_SIZE_BY_OBJECT { \
	HPI_RESPONSE_FIXED_SIZE ,   /* Default, no object type 0 */ \
	HPI_RESPONSE_FIXED_SIZE + sizeof(HPI_SUBSYS_RES),\
	HPI_RESPONSE_FIXED_SIZE + sizeof(HPI_ADAPTERX_RES),\
	HPI_RESPONSE_FIXED_SIZE + sizeof(HPI_STREAM_RES),\
	HPI_RESPONSE_FIXED_SIZE + sizeof(HPI_STREAM_RES),\
	HPI_RESPONSE_FIXED_SIZE + sizeof(HPI_MIXER_RES),\
	HPI_RESPONSE_FIXED_SIZE , /* No NODE RESPONSE */ \
	HPI_RESPONSE_FIXED_SIZE + sizeof(HPI_CONTROL_RES),\
	HPI_RESPONSE_FIXED_SIZE + sizeof(HPI_NVMEMORY_RES),\
	HPI_RESPONSE_FIXED_SIZE + sizeof(HPI_GPIO_RES),\
	HPI_RESPONSE_FIXED_SIZE + sizeof(HPI_WATCHDOG_RES),\
	HPI_RESPONSE_FIXED_SIZE + sizeof(HPI_CLOCK_RES),\
	HPI_RESPONSE_FIXED_SIZE + sizeof(HPI_PROFILE_RES),\
	HPI_RESPONSE_FIXED_SIZE + sizeof(HPI_CONTROLX_RES),\
 	HPI_RESPONSE_FIXED_SIZE + sizeof(HPI_ASYNC_RES) \
}

/*////////////////////////////////////////////////////////////////////////// */
/* declarations for compact control calls                                    */
typedef struct HPI_CONTROL_DEFN {
	HW8 wType;
	HW8 wChannels;
	HW8 wSrcNodeType;
	HW8 wSrcNodeIndex;
	HW8 wDestNodeType;
	HW8 wDestNodeIndex;
} HPI_CONTROL_DEFN;

/*////////////////////////////////////////////////////////////////////////// */
/* declarations for control caching (internal to HPI<->DSP interaction)      */
typedef struct {
	HW16	anLog[2];
} tHPIControlCachedVolume;
typedef struct {
	HW16	anLogPeak[2];
	HW16	anLogRMS[2];
} tHPIControlCachedPeak;
typedef struct {
	HW16	wMode;
} tHPIControlCachedChannelMode;
typedef struct {
	HW32	dwErrorStatus;
	HW32	dwSource;
} tHPIControlCachedAES3RxStatus;
typedef struct {
	HW32	dwFormat;
} tHPIControlCachedAES3Tx;
typedef struct {
	HW16	wSourceNodeType;
	HW16	wSourceNodeIndex;
} tHPIControlCachedMultiplexer;
typedef struct {
	HW16	anLog[2];
} tHPIControlCachedLevel;
typedef struct {			/* only partial caching - some attributes have to go to the DSP. */
	HW32	dwFreqInkHz;
	HW16	wBand;
	HW16	wLevel;
} tHPIControlCachedTuner;
typedef struct {
	HW16	wState;
} tHPIControlCachedToneDetector;
typedef struct {
	HW32	dwState;
	HW32	dwCount;
} tHPIControlCachedSilenceDetector;
typedef struct {
	HW16	wSource;
	HW16	wSourceIndex;
	HW32	dwSampleRate;
} tHPIControlCachedSampleClock;
typedef struct {
	HW32	dw1;
	HW32	dw2;
} tHPIControlCachedGeneric;

/** A compact representation of (part of) a controls state.
Used for efficient transfer of the control state between DSP and host or across a network
*/
typedef struct {
	HW16	ControlType;   /**< one of HPI_CONTROL_* */
	HW16	ControlIndex;  /**< The original index of the control on the DSP */
	union {
		tHPIControlCachedVolume v;
		tHPIControlCachedPeak p;
		tHPIControlCachedChannelMode m;
		tHPIControlCachedMultiplexer x;
		tHPIControlCachedLevel l;
		tHPIControlCachedTuner t;
		tHPIControlCachedAES3RxStatus aes3rx;
		tHPIControlCachedAES3Tx aes3tx;
		tHPIControlCachedToneDetector tone;
		tHPIControlCachedSilenceDetector silence;
		tHPIControlCachedSampleClock clk;
		tHPIControlCachedGeneric g;
	} u;
} tHPIControlCacheSingle;
/*//////////////////////////////////////////////////////////////////////////////// */
/* declarations for 2^N sized FIFO buffer (internal to HPI<->DSP interaction)      */
typedef struct {
	HW32 dwSize;
	HW32 dwDSPIndex;
	HW32 dwHostIndex;
} tHPIFIFOBuffer;

#ifndef DISABLE_PRAGMA_PACK1
#pragma pack(pop)
#endif

#ifndef HPI_ON_DSP
/* skip host side function declarations for DSP compile and documentation extraction */
void HPI_InitMessage( HPI_MESSAGE *phm, HW16 wObject, HW16 wFunction );
void HPI_InitResponse( HPI_RESPONSE *phr, HW16 wObject, HW16 wFunction, HW16 wError);

/*////////////////////////////////////////////////////////////////////////// */
/* main HPI entry point */
HPI_HandlerFunc HPI_Message;

#ifndef HPI_KERNEL_MODE /* ----------------- not Win95,WinNT,WDM kernel */
HW16 HPI_DriverOpen( void );
void HPI_DriverClose( void);
#endif

/* used in PnP OS/driver */
HPI_ERR HPI_API HPI_SubSysCreateAdapter(
				  HPI_HSUBSYS *phSubSys,
				  HPI_RESOURCE *pResource,
				  HW16            *pwAdapterIndex
						);

HPI_ERR HPI_API HPI_SubSysDeleteAdapter(
		  HPI_HSUBSYS *phSubSys,
		  HW16          wAdapterIndex);

void HPI_FormatToMsg(HPI_MSG_FORMAT * pMF, HPI_FORMAT * pF);
void HPI_StreamResponseToLegacy(HPI_STREAM_RES * pSR);


/*////////////////////////////////////////////////////////////////////////// */
/* declarations for individual HPI entry points */
HPI_HandlerFunc HPI_4000;
HPI_HandlerFunc HPI_6000;
HPI_HandlerFunc HPI_6205;
HPI_HandlerFunc HPI_COMMON;

#endif /* ndef HPI_ON_DSP */

#endif /* ndef HPI_EXCLUDE_IMPLEMENTATION */

#ifdef __cplusplus
}
#endif

#endif  /*_H_HPI_ */

/*
///////////////////////////////////////////////////////////////////////////////
// See CVS for history.  Last complete set in rev 1.146
////////////////////////////////////////////////////////////////////////////////
*/
